/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/

/* NOTE:  THIS MODULE AND SOURCE CODE IS FOR USE 
   WITH THE AVS SOFTWARE ENVIRONMENT ONLY */
/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* Module Name: "zone_fence" (Filter) (Subroutine)                        */
/* Author: Ian_Curington                                                   */
/*
 * zone_fence - display, finite fence connecting zones from two wells.
 *
 * This routine takes threee 1D  inputs, one the
 * zonation table, and the deviation files for two wells.
 * It creates a geometric representation of the zonation region
 * between the wells.
 *
 * It looks at the labels on the well dev field to figure
 * out which well it is, then indexes the zonations by this,
 * so there is no menu selection needed here.
 *
 * Author: Ian J. Curington, Advanced Visual Systems Inc.
 *
 * Revision:
 *   1 Sept 93 ianc: Original
 *  10 Sept 93 ianc: added reentrant flag
 *                   changed names for linkage
 *  13 Sept 93 ianc: added magic number conditionals
 *                   added segment parameter
 *   13 Sep 93 ianc: fixed bugs, added z invert, new estimate_zinc algorithm
 *   15 Sep 93 ianc: worked on resample bug
 *
 */

/*****************************************/
/* Global Declarations                   */
/*****************************************/

#include <stdio.h>
#include <avs/avs.h>
#include <avs/port.h>
#include <avs/field.h>
#include <avs/geom.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>


#define PATCH  1
#define EPS   1.0e-5
#define ABS(a) ( ( (a) >0.0) ? (a) : ( -1.0 * (a)) )
#define DELIM  "#"
/**
#define INVERT_Z
 **/
#ifdef  INVERT_Z
#define MAGIC   999.90
#else
#define MAGIC  -999.90
#endif
#define MAGIC_TEST(a) \
                  ( a >= MAGIC-1.0 && a <= MAGIC+1.0 ) 

static float    yellow[3]={ 0.99, 0.99, 0.0 };

 
/* *****************************************/
/*  Module Description                     */
/* *****************************************/
int zone_fence_desc()
{

	int in_port, out_port, param, iresult;
	extern int zone_fence_compute();
	extern int zone_fence_init();
	extern int zone_fence_destroy();

	AVSset_module_name("zone_fence", MODULE_MAPPER);
        AVSset_module_flags( COOPERATIVE | REENTRANT );

	/* Input Port Specifications               */
	in_port = AVScreate_input_port("Zonation", 
		"field 2D 2-space uniform float", REQUIRED);

	in_port = AVScreate_input_port("Well_Devs1", 
		"field 1D 3-space irregular float", REQUIRED);

	in_port = AVScreate_input_port("Well_Devs2", 
		"field 1D 3-space irregular float", REQUIRED);

	in_port = AVScreate_input_port("WellList", "string", REQUIRED );

	in_port = AVScreate_input_port("Texture1", 
		"field 2D 2-space 4-vector uniform byte", OPTIONAL);

	in_port = AVScreate_input_port("Texture2", 
		"field 2D 2-space 4-vector uniform byte", OPTIONAL);


	/* Output Port Specifications              */
	out_port = AVScreate_output_port("Fence", "geom" );


	/* Parameter  Specifications              */

        AVSadd_parameter("On/Off", "boolean", 1, 0, 1 );
        AVSadd_parameter("Update", "oneshot", 0, 0, 1 );
        param = AVSadd_parameter("N Segments", "integer", 6, 1, 100 );
        AVSconnect_widget(param, "typein_integer");


	AVSset_compute_proc(zone_fence_compute);
	AVSset_init_proc(zone_fence_init);
	AVSset_destroy_proc(zone_fence_destroy);

	return(1);
}
 
/* *****************************************/
/* Module Compute Routine                  */
/* *****************************************/
int zone_fence_compute( Zonation, Well_Devs1, Well_Devs2,
                        WellList, Texture1, Texture2,
                         Fence,
                         on_off, update, n_segments )
	AVSfield_float *Zonation;    /* input zone defs */
	AVSfield_float *Well_Devs1;  /* input deviations for one well */
	AVSfield_float *Well_Devs2;  /* input deviations for one well */
        char           *WellList;    /* input well name list */
        AVSfield_char  *Texture1;    /* texture input image */
        AVSfield_char  *Texture2;    /* texture input image */
        GEOMedit_list  *Fence;       /* output geometric pick representation */
        int            on_off;       /* control switch for whole module */
        int            update;       /* update trigger */
        int            n_segments;   /* number of sub-panels */
{
    GEOMobj *obj1;
    GEOMobj *obj2;
    float zinc1;
    float zinc2;
    float *x1, *y1, *z1;
    float *x2, *y2, *z2;
    int   num_zones;
    int   num_wells;
    int   well1, well2;
    char  labels[200], delim[4];
    char  zone_name[80];
    char  well_name1[80];
    char  well_name2[80];
    char  string[80];
    int   i, j;
    float         min_extent[3], max_extent[3];
    float est_zinc();

    /***********/
    /* Statics */
    /***********/


    /***********/
    /* BODY    */
    /***********/
    /* Get size of zone list for this well */
    AVSfield_get_labels ( Well_Devs1, well_name1, delim );
    well1 = choice_num( WellList, well_name1, DELIM ) - 1;
    AVSfield_get_labels ( Well_Devs2, well_name2, delim );
    well2 = choice_num( WellList, well_name2, DELIM ) - 1;
    if ( well1 < 0 ) well1 = 0;
    if ( well2 < 0 ) well2 = 0;
    num_wells = MAXX( Zonation );
    num_zones = Zonation->veclen;

    if ( on_off )
    {
        /* Allocate space for the zone pick location list */
        x1 = (float *) malloc ( sizeof(float) * num_zones );
        y1 = (float *) malloc ( sizeof(float) * num_zones );
        z1 = (float *) malloc ( sizeof(float) * num_zones );
        x2 = (float *) malloc ( sizeof(float) * num_zones );
        y2 = (float *) malloc ( sizeof(float) * num_zones );
        z2 = (float *) malloc ( sizeof(float) * num_zones );

        /* Fill the Z array with Zonation Depths */
        /* "TOP" zone surface only */
        for ( i=0; i < num_zones; i++ )
        {
            z1[i] = Zonation->data[ (0*num_wells+well1)*num_zones+i ];
            z2[i] = Zonation->data[ (0*num_wells+well2)*num_zones+i ];
        }

        /* Estimate Z interval in Deviation Field */
        zinc1 = est_zinc ( Well_Devs1 );
        zinc2 = est_zinc ( Well_Devs2 );

#ifdef INVERT_Z
        zinc1 = -1.0 * zinc1;
        zinc2 = -1.0 * zinc2;
#endif

        /* Locate the spatial location of zone picks */
        resample_z ( Well_Devs1, x1, y1, z1, num_zones, zinc1 ); 
        resample_z ( Well_Devs2, x2, y2, z2, num_zones, zinc2 ); 
    }

    /* Create Geometric Representation */

    *Fence = GEOMinit_edit_list(*Fence);
    obj1 = GEOMcreate_obj(GEOM_POLYHEDRON, NULL);
    obj2 = GEOMcreate_obj(GEOM_POLYHEDRON, NULL);


    if ( on_off )
    {
            draw_fence( obj1, obj2,
                        x1, y1, z1,
                        x2, y2, z2,
                        num_zones, n_segments );
    }

    GEOMgen_normals(obj1, 0);
    GEOMgen_normals(obj2, 0);
    GEOMcvt_polyh_to_polytri(obj1,
         GEOM_NO_CONNECTIVITY | GEOM_SURFACE | GEOM_WIREFRAME);
    GEOMcvt_polyh_to_polytri(obj2,
         GEOM_NO_CONNECTIVITY | GEOM_SURFACE | GEOM_WIREFRAME);

    GEOMedit_geometry(*Fence, "Fence1", obj1);
    GEOMedit_geometry(*Fence, "Fence2", obj2);
    GEOMdestroy_obj( obj1 );
    GEOMdestroy_obj( obj2 );

    GEOMedit_transform_mode(*Fence, "Fence1", "parent",0);
    GEOMedit_transform_mode(*Fence, "Fence2", "parent",0);

    if ( Texture1 && Texture2 )
    {
        GEOMedit_field_texture(*Fence,"Fence1",Texture1);
        GEOMedit_field_texture(*Fence,"Fence2",Texture2);
    }

    /* free temp module storage */
    if ( on_off )
    {

/* IAC CODE CHANGE :         if ( x1 ) free ( x1 ); */
        if ( x1 )  free(x1 );

/* IAC CODE CHANGE :         if ( y1 ) free ( y1 ); */
        if ( y1 )  free(y1 );

/* IAC CODE CHANGE :         if ( z1 ) free ( z1 ); */
        if ( z1 )  free(z1 );

/* IAC CODE CHANGE :         if ( x2 ) free ( x2 ); */
        if ( x2 )  free(x2 );

/* IAC CODE CHANGE :         if ( y2 ) free ( y2 ); */
        if ( y2 )  free(y2 );

/* IAC CODE CHANGE :         if ( z2 ) free ( z2 ); */
        if ( z2 )  free(z2 );
    }

    return(1);
}
 
/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
#ifdef SEP_EXE
int ((*mod_list[])()) = {
	zone_fence_desc,
};
#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
	AVSinit_from_module_list(mod_list, NMODS);
}
#endif
 

/* *****************************************/
/* Module Initialization Routine           */
/* *****************************************/
int zone_fence_init()
{
}

/* *****************************************/
/* Module Destroy Routine                  */
/* *****************************************/
int zone_fence_destroy()
{
}

/* *****************************************/
/* Estimate Depth Increment Routine        */
/* *****************************************/
float
est_zinc ( Well_Devs )
	AVSfield_float *Well_Devs;
{
    int i, n;
    float zinc;

    n = Well_Devs->dimensions[0];

    if ( n <= 1 )
    {
        zinc= 0.0;
    }
    else if ( n == 2 )
    {
        zinc = ABS(Well_Devs->points[ 2 * n + 1 ] -
                   Well_Devs->points[ 2 * n + 0 ]);
    }
    else
    if ( n > 2 )
        zinc = ABS(Well_Devs->points[ 2 * n + 2 ] -
                   Well_Devs->points[ 2 * n + 1 ]);

    return( zinc );

}

/* *****************************************/
/* Resample Depth Routine                  */
/* *****************************************/
resample_z ( Well_Devs, X, Y, Z, num_zones, zinc )
	AVSfield_float *Well_Devs;
        float *X, *Y, *Z;
        int   num_zones;
        float zinc;
{
    int i, n_devs;

    n_devs = MAXX( Well_Devs );

    /* ==== Linear Only  ==== */
        zznmoli ( Well_Devs->points,                /* x-axis deviations */
                 Z,                                /* z of zones */
                 X,                                /* x of zones */
                 zinc,                             /* depth spacing of dev data */
                 n_devs,                           /* number of deviations */
                 num_zones );                      /* number of zone samples */

        zznmoli ( &Well_Devs->points[ n_devs ],     /* y-axis deviations */
                 Z,                                /* z of zones */
                 Y,                                /* y of zones */
                 zinc,                             /* depth spacing of dev data */
                 n_devs,                           /* number of deviations */
                 num_zones );                      /* number of zone samples */


}

/* *****************************************/
/* Resample Utility Routines               */
/* *****************************************/

/*
 * znmoli - depth normal moveout vector resampling, linear interpolation
 *
 * Author: Ian Curington, AVS Inc.
 *
 * Revision:
 * 21 Aug 93 ianc - Original, from original Fortran template
 *
 *
 *  PURPOSE:      To linearly interpolate from an input seismic trace of
 *                sample values, to produce an output trace of sample
 *                values, using an input vector of sample times at which
 *                to perform interpolation.
 *
 *  FORMULA:      The interpolation is linear, in the normal sense that
 *                a weighted average of adjacent (1 preceding, 1
 *                following) sample input values is used in computing
 *                output sample values.  If sample values outside the
 *                input range of X(I) are required by the interpolation
 *                process, the output value of Z(J) will be set to 0.0.
 *
 *                   For k = 1 to NNMO-1
 *                       n = trunc(Y(k)/SR)
 *                       F = frac(Y(k)/SR)
 *
 *                       If ( n+1 <= 0 or n+2 > N ) then
 *                          Z(k) = 0.0
 *                       Else
 *                          Z(k) = (X(n+2) - X(n+1))*F + X(n+1)
 *
 *
 *  DESCRIPTION:  ZNMOLI converts out-of-place,
 *                the intput vector Y of trace
 *                sample values into output vector Z of corresponding
 *                trace sample values, which are interpolated from input
 *                vector X of original trace sample values.
 *
 *      S0 and S1 represent the 2 values used
 *      in the interpolation formula.
 *
 */

zznmoli ( x, y, z, sr, n, nnmo )

    float *x;  /* vector of input trace samples */
    float *y;  /* input vector of time samples to retrieve */
    float *z;  /* output vector of new trace samples at new times */
    float sr;  /* sample rate of input trace vector */
    int    n;  /* number of samples in input trace */
    int nnmo;  /* number of time samples to get */

{
    double k, s0, s1, f;
    int i, j;

    if ( n > 1 && nnmo > 0 && sr != 0.0 )
    {
        for ( j=0; j < nnmo; j ++ )
        {
            k = y[j] / sr;
            i = (int)k;
            f = k - (float)i;
            if ( i < 0 )
            {
                s0 = s1 = x[0];
            }
            else if ( i >= n )
            {
                s0 = s1 = x[n-1];
            }
            else
            {
                s0 = x[i];
                s1 = x[i+1];
            }
            z[j] = (s1 - s0) * f + s0;
        }
    }
}


/******************************************************************************/
/*
 * Resolve choices from string list, curent val, into int offset.
 */
/******************************************************************************/

int
choice_num(name, str, delim)
    char *name;
    char *str;
    char *delim;
{
   char *choices, *cp;
   int i, flag;

   choices = name;
   for (i = 1; choices; i++) {
	   cp = strchr(choices, *delim);
	   if (cp) {
		   *cp = '\0';
	   }
	   flag = strcmp(choices, str);
	   if (cp) {
		   *cp = *delim;
		   choices = cp + 1;
	   } else {
		   choices = 0;
	   }
	   if (flag == 0)
		   return(i);
   }
   return(0);
}


/******************************************************************************/


/******************************************************/
/* Fence Processing Function                          */
/******************************************************/

draw_fence( obj1, obj2,
            x1, y1, z1,
            x2, y2, z2,
            num_zones, n_segments )

       GEOMobj *obj1;
       GEOMobj *obj2;
       float   *x1, *y1, *z1;
       float   *x2, *y2, *z2;
       int     num_zones;
       int     n_segments;
{
       GEOMobj *obj;
       int   i, j;
       float verts [ 4 * 3 ];
       float uvs   [ 4 * 2 ];
       int   cols  [ 4 ];
       int   color;
       float h, s, v, a;
       float t, r;
       float tm, rm;

       h = 0.8; s=0.9; v=0.8; a=1.0;

       for ( i=0; i < 4; i++ )
           PIXELhsva_to_pixel( &cols[i], h, s, v, a);


       for ( i=0; i < num_zones-1; i++ )
       {
           /* put every other row on alternate geoms */
           if ( (i & 1) == 0 )
               obj = obj1;
           else
               obj = obj2;

           for ( j=0; j < n_segments; j++ )
           {
               t = (float)(j  ) / (float)n_segments;
               r = (float)(j+1) / (float)n_segments;
               tm = 1.0 - t;
               rm = 1.0 - r;

               /* draw top quad */
               verts[ 0 * 3 + 0 ] = tm*x1[i]+t*x2[i];
               verts[ 0 * 3 + 1 ] = tm*y1[i]+t*y2[i];
               verts[ 0 * 3 + 2 ] = tm*z1[i]+t*z2[i];
               uvs  [ 0 * 2 + 0 ] = 0.0;
               uvs  [ 0 * 2 + 1 ] = 0.0;

               verts[ 1 * 3 + 0 ] = rm*x1[i]+r*x2[i];
               verts[ 1 * 3 + 1 ] = rm*y1[i]+r*y2[i];
               verts[ 1 * 3 + 2 ] = rm*z1[i]+r*z2[i];
               uvs  [ 1 * 2 + 0 ] = 1.0;
               uvs  [ 1 * 2 + 1 ] = 0.0;

               verts[ 2 * 3 + 0 ] = rm*x1[i+1]+r*x2[i+1];
               verts[ 2 * 3 + 1 ] = rm*y1[i+1]+r*y2[i+1];
               verts[ 2 * 3 + 2 ] = rm*z1[i+1]+r*z2[i+1];
               uvs  [ 2 * 2 + 0 ] = 1.0;
               uvs  [ 2 * 2 + 1 ] = 1.0;

               verts[ 3 * 3 + 0 ] = tm*x1[i+1]+t*x2[i+1];
               verts[ 3 * 3 + 1 ] = tm*y1[i+1]+t*y2[i+1];
               verts[ 3 * 3 + 2 ] = tm*z1[i+1]+t*z2[i+1];
               uvs  [ 3 * 2 + 0 ] = 0.0;
               uvs  [ 3 * 2 + 1 ] = 1.0;

               if ( !MAGIC_TEST(z1[i]  ) &&
                    !MAGIC_TEST(z2[i]  ) &&
                    !MAGIC_TEST(z1[i+1]) &&
                    !MAGIC_TEST(z2[i+1]) )
               {
                   GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
                                   GEOM_NOT_SHARED, GEOM_COPY_DATA);
                   /***
                   GEOMadd_int_colors(obj, cols, 4, GEOM_COPY_DATA);
                    ***/
                   GEOMadd_uvs (obj, uvs, 4, GEOM_COPY_DATA);
               }
           }
       }

}

/******************************************************************************/

