/****************************************************************************
                          INTERNATIONAL AVS CENTRE
           (This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Centre, University of Manchester, the AVS Consortium and
the individual  submitting the module and files associated with said module
provide absolutely NO WARRANTY OF ANY KIND with respect to this software.  The
entire risk as to the quality and performance of this software is with the
user.  IN NO EVENT WILL The International AVS Centre, University of Manchester,
the AVS Consortium and the individual submitting the module and files
associated with said module BE LIABLE TO ANYONE FOR ANY DAMAGES ARISING FROM
THE USE OF THIS SOFTWARE, INCLUDING, WITHOUT LIMITATION, DAMAGES RESULTING FROM
LOST DATA OR LOST PROFITS, OR ANY SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@iavsc.org.

Please send AVS module bug reports to avs@iavsc.org.

***************************************************************************/
/*********************************************************************

Copyright (C) 1995, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus it is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory
		Berkeley, California

  "this software is 100% hand-crafted by a human being in the USA"

//\/\\//\\//\\/\//\\/\\//\\//\\/\\/\//\\//\\//\\//\\//\\//\/\\//\\//\\/\\

  This module may be used to apply geometric transformations to AVS geometry
  types from events generated from VR input devices.

  Mon Jan  9 09:40:05 PST 1995
  
  LIMITATIONS/BUGS:
  1. due to bugs in the implementation of the AVS-side of CLI, in a
  multi-camera environment, this module will obtain the view matrix from
  camera1 ONLY.  when CLI supports getting the view matrix from whatever
  is the "current" camera (like the documentation says it's supposed to
  do), this routine will work properly.

  2. due to limitations in the AVS-side of CLI, we support only two-level
  object heirachies.  in other words, we can transform %top correctly, and
  we can transform the children of %top correctly.  intermediate objects,
  if the tree depth is 3 or greater,  are not guaranteed to be properly
  transformed.  


*********************************************************************/


#include <stdio.h>
#include <string.h>
#include <avs/avs.h>
#include <avs/field.h>
#include <avs/geomdata.h>
#include <avs/udata.h>
#include "matrix.h"
#include "vr_util.h"

#define CHILL 1
#define WHACKED 0


static char DEVICE_INPUT_NAME[] = {"input_device_event_field"};   

static matrix4x4 p_view,view_inverse,p_parent_object,parent_object_inverse;
static matrix4x4 p_objxform,p_worldxform;

typedef struct ui_state
{
    int have_input_device_data;
} ui_state;

static char cur_obj[256]={"%top"},last_cur_obj[256]={"empty"};

static void build_rot_matrix();
static void build_trans_matrix();
static void copy_avsmatrix_to_matrix();
static void build_object_xform_matrix();
static void grab_view_matrix();
static void grab_obj_matrix();


AVSinit_modules()
{
    int vr_xfrm_obj();
    AVSmodule_from_desc(vr_xfrm_obj);
}

int
vr_xfrm_obj()
{
    int vr_xfrm_obj_compute(),end_func();
    int p;

    AVSset_module_name("VR Transform Object",MODULE_MAPPER);

    AVScreate_input_port(DEVICE_INPUT_NAME,VR_FIELD_STRING,REQUIRED);

    AVScreate_output_port("XformList","geom");

    p = AVSadd_parameter("dummy","string","Sensitivity Control","","");
    AVSadd_parameter_prop(p,"width","integer",4);
    AVSconnect_widget(p,"text");
    
    p = AVSadd_float_parameter("Translate Scale",1.0,0.1,100.);
    AVSconnect_widget(p,"typein_real");

    p = AVSadd_float_parameter("Rotate Scale",1.0,0.1,5.);
    AVSconnect_widget(p,"typein_real");

    p = AVSadd_parameter("dummy2","string","Current Object","","");
    AVSadd_parameter_prop(p,"width","integer",4);
    AVSconnect_widget(p,"text");

    /* make the parm that reports the name of the object this module
       is working on read-only so that the user doesn't get the idea
       that they can do object selection from here. */
    
    p = AVSadd_parameter("Object Name","string","%top",NULL,NULL);
    AVSconnect_widget(p,"text");
    AVSadd_parameter_prop(p,"width","integer",4);

    AVSset_compute_proc(vr_xfrm_obj_compute);
}

int
vr_xfrm_obj_compute(inf,
		    iogeom,
		    dummy,
		    tscale,
		    rscale,
		    dummy2,
		    curobj_ui)
AVSfield *inf;
GEOMedit_list *iogeom;
char *dummy;
float *tscale;
float *rscale;
char *dummy2;
char *curobj_ui;
{
    ui_state new_state;
    GEOMobj *parent;
    char *c,cb[32],cli_buf[256];
    char work[1024];
    char *outbuf,*errbuf,*outbuf_view;
    matrix4x4 temp;
    float *tv,sb[9];
    int i,j;
    matrix4x4 new_xfrm;

    *iogeom = GEOMinit_edit_list(*iogeom);

    /* use CLI to get the name of the currently selected object */
    
    sprintf(cli_buf,"geom_get_cur_obj_name");
    AVScommand("kernel",cli_buf,&outbuf_view,&errbuf);
    strcpy(cur_obj,outbuf_view);
    c = strtok(cur_obj," ");
    if (c == NULL)
	return(0);		/* there must be no geometry viewer */
    cur_obj[strlen(c)-1] = '\0';

    /* update control panel widget to reflect this module's notion
       of "current object" */
    if (strcmp(cur_obj,last_cur_obj) != 0)
    {
	strcpy(last_cur_obj,cur_obj);
	AVSmodify_parameter("Object Name",AVS_VALUE,cur_obj);
    }

    grab_view_matrix(&p_view);
    grab_obj_matrix(cur_obj,&p_parent_object);

    invert_4x4matrix(&p_view,&view_inverse);
    invert_4x4matrix(&p_parent_object,&parent_object_inverse);

    /* call routine to check for a change in state. */
    do_state(&new_state);

    if (new_state.have_input_device_data)
    {
	tv = inf->field_union.field_data_float_u;

	/* we have an event from the device.  extract the translational
	   and rotational components. */
	VR_GET_TRANS_VECTOR(sb[0],sb[1],sb[2],inf);
	VR_GET_ROT_VECTOR(sb[3],sb[4],sb[5],inf);
	
	/* build a new transformation matrix. */
	identity_4x4(&new_xfrm);
	build_object_xform_matrix(&new_xfrm,sb,tscale,rscale);

	work[0] = '%';
	work[1] = '\0';
	strcat(work,cur_obj);
	
	GEOMedit_concat_matrix(*iogeom,work,&(new_xfrm.m[0][0]));
    }
    return(1);
}

int
do_state(this_state)
ui_state *this_state;
{
    /**
      * this routine is a throwback to a prototype which was built as a
      * coroutine.  by making the module a subroutine, fewer states in
      * the software are necessary, making for simplified code.
      * at present, all we do here is check to see if there's any new
      * data from the input device.
    **/
    this_state->have_input_device_data = AVSinput_changed(DEVICE_INPUT_NAME,0);
}

static void
copy_avsmatrix_to_matrix(am,m)
float *am;
matrix4x4 *m;
{
    memcpy((char *)(&m->m[0][0]),(char *)(am),sizeof(matrix4x4));
}

static void
build_object_xform_matrix(m,sb,tscale,rscale)
matrix4x4 *m;
float *sb;
float *tscale,*rscale;
{
    matrix4x4 t,t2;
    matrix4x4 rot;

    identity_4x4(&rot);
    build_rot_matrix(&rot,&p_view,&view_inverse,&parent_object_inverse,sb,rscale);
    build_trans_matrix(&t,&view_inverse,&parent_object_inverse,sb,tscale);

    identity_4x4(m);
    mmul_4x4(&rot,&t);
    matrix_4copy(&rot,m);
}

static void
build_trans_matrix(t,view_inverse,parent_object_inverse,sb,tscale)
matrix4x4 *t,*view_inverse,*parent_object_inverse;
float *sb,*tscale;
{
    vector4 v,r;
    matrix4x4 loc_inverse;
	
    v.v[0] = sb[0] * *tscale;
    v.v[1] = sb[1] * *tscale;
    v.v[2] = -1.*sb[2] * *tscale;  /* flip handed-ness of
				      the coord system. */
    v.v[3] = 0.;		/* ignore W component */

    matrix_4copy(view_inverse,&loc_inverse);
    mmul_4x4(&loc_inverse,parent_object_inverse);
    
    vector_matrix_mult_4(&v,&loc_inverse,&r);

    identity_4x4(t);
    
    t->m[3][0] = r.v[0];
    t->m[3][1] = r.v[1];
    t->m[3][2] = r.v[2];

}

static void
build_rot_matrix(rot,view,view_inverse,object_inverse,sb,rscale)
matrix4x4 *rot,*view_inverse,*object_inverse,*view;
float *sb;
float *rscale;
{
    /**
      * in building the rot matrix, we'll apply the rotations in order of
      * decreasing magnitude.  in other words, if the user is tweaking hard
      * on a y-axis rotation, and just barely on x & z rotations, we'll apply
      * the y-axis rotation first, then the x & z will be done in decreasing
      * order of magnitude.
      *
      * what we want is to figure out how much we have to rotate the object
      * in object space; angle size in each of the axes, in order to achieve
      * the user-specified rotation.  the problem is that the user specifies
      * the rotation in NPC space, yet we rotate in object space.
      *
      * a vector is formed in which the X component is the X axis rotation
      * component from the device, etc.  we transform this vector through
      * the inverse view/object transformation matrices.  this gives us
      * rotation angles about the principal axes in modeling space.
    **/
    int order[3];  /* boolean to indicate order.  order[0]=0 means x axis
		      is first, order[0]=1, x axis is 2nd, order[0]=2,
		      x axis is third. etc. */
    double fx,fy,fz,t,c,s,xr,yr,zr;
    extern double fabs(),cos(),sin();
    matrix4x4 rx,ry,rz,loc_inverse,temp,loc_obj_inverse;
    matrix4x4 *mp[3];
    vector4 r1,r2;
    int i;

    
    matrix_4copy(view_inverse,&temp);
    mmul_4x4(&temp,object_inverse);
    matrix_4copy(&temp,&loc_inverse);

    r1.v[0] = sb[3] * *rscale;
    r1.v[1] = sb[4] * *rscale;
    r1.v[2] = -1. * sb[5] * *rscale;
    r1.v[3] = 0;

    vector_matrix_mult_4(&r1,&loc_inverse,&r2); /* xform rot vector
						   thru invs.*/
    xr = r2.v[0];
    yr = r2.v[1];
    zr = r2.v[2];

    t = r2.v[0]; 
    fx = fabs(t);
    t = r2.v[1]; 
    fy = fabs(t);
    t = r2.v[2]; 
    fz = fabs(t);

    mp[0] = mp[1] = mp[2] = NULL;

    order[0] = ((fx >= fy) && (fx >= fz)) ? 0 :
	(((fx >= fy) || (fx >= fz)) ? 1 : 2);
    order[2] = ((fz >= fy) && (fz >= fx)) ? 0 :
	(((fz >= fy) || (fz >= fx)) ? 1 : 2);
    order[1] = ((fy >= fx) && (fy >= fz)) ? 0 :
	(((fy >= fx) || (fy >= fz)) ? 1 : 2);

    identity_4x4(&rx);
    t = xr;			/* assume device events are in radians  */
    c = cos(t);
    s = sin(t);
    rx.m[1][1] = rx.m[2][2] = c;
    rx.m[1][2] = s;
    rx.m[2][1] = -1. * s;
    
    identity_4x4(&ry);
    t = yr;			/* assume device events are in radians */
    c = cos(t);
    s = sin(t);
    ry.m[0][0] = ry.m[2][2] = c;
    ry.m[0][2] = -1 * s;
    ry.m[2][0] = s;
    
    identity_4x4(&rz);
    t = zr;			/* assume device events are in radians */
    c = cos(t);
    s = sin(t);
    rz.m[0][0] = rz.m[1][1] = c;
    rz.m[0][1] = s;
    rz.m[1][0] = -1. * s;
    
    identity_4x4(rot);

    if (fx != 0)
	mp[order[0]] = &rx;
    if (fy != 0)
	mp[order[1]] = &ry;
    if (fz != 0)
	mp[order[2]] = &rz;

    for (i=0;i<3;i++)
    {
	if (mp[i] != NULL)
	    mmul_4x4(rot,mp[i]);
    }
}

static void
grab_view_matrix(v)
matrix4x4 *v;
{
    int i,j;
    char *c,work[1024],cli_buf[1024],*outbuf_view,*errbuf;

    /**
      * it would be nice to just get the matrix for whatever
      * is the current camera.  however, the CLI command has a bug
      * whereby it won't return the matrix for the "current"
      * camera if an index is not specified.  thus, a limitation of
      * this module is that the inverse view is computed ALWAYS
      * using the view of camera 1.
    **/
    sprintf(cli_buf,"geom_get_matrix -camera 1");
    AVScommand("kernel",cli_buf,&outbuf_view,&errbuf);
    strcpy(work,outbuf_view);

    c = strtok(work," \n");
    for (j=0;j<4;j++)
	for (i=0;i<4;i++)
	{
	    sscanf(c,"%f",&(v->m[j][i]));
	    c = strtok(NULL," \n\0");
	}
}

static void
grab_obj_matrix(name,m)
char *name;
matrix4x4 *m;
{
    int i,j;
    char *c,work[1024],cli_buf[1024],*outbuf_view,*errbuf;
    
    sprintf(cli_buf,"geom_get_matrix -object %s",name);
    AVScommand("kernel",cli_buf,&outbuf_view,&errbuf);
    strcpy(work,outbuf_view);

    c = strtok(work," \n");
    for (j=0;j<4;j++)
	for (i=0;i<4;i++)
	{
	    sscanf(c,"%f",&(m->m[j][i]));
	    c = strtok(NULL," \n\0");
	}
    
    if (strcmp(name,"top") != 0)
    {
	/**
	  * if the selected object isn't "top", grab the "top matrix",
	  * and concatenate it's matrix on with this one.  this will allow
	  * for two-level heirarchies, but will fail for deeper heirarchies.
	  * avs does not permit access to  information such as "who's my
	  * parent" or "who are my children".  this information is required
	  * in order to build transformation trees of an arbitrary depth.
	**/
	matrix4x4 parent;
	
	sprintf(cli_buf,"geom_get_matrix -object top");
	AVScommand("kernel",cli_buf,&outbuf_view,&errbuf);
	strcpy(work,outbuf_view);

	/**
	  * matrices are applied in order of:
	  * A=child object,
	  * B= parent object, so:
	  *   A*B.
	  * what we'll eventually want is B**(-1) * A**(-1), which is
	  * by definition (AB)**(-1).
	**/

	c = strtok(work," \n");
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		sscanf(c,"%f",&(parent.m[j][i]));
		c = strtok(NULL," \n\0");
	    }
	matrix_4copy(&parent,m);
    }
    else
    {
	identity_4x4(m);
    }
}
