/********************************************************************************/
/*										*/
/*	Project  : P A P Y R U S  Toolkit					*/
/*	File     : PapyRead3.c							*/
/*	Function : contains all the reading functions				*/
/*	Authors  : Matthieu Funk						*/
/*		   Christian Girard						*/
/*		   Jean-Francois Vurlod				     		*/
/*		   Marianne Logean					        */
/*										*/
/*	History  : 12.1990	version 1.0				 	*/
/*		   04.1991	version 1.1					*/
/*		   07.1992	version 1.2					*/
/*		   03.1993	version 2.0					*/
/*		   07.1994	version 3.0					*/
/*		   06.1995	version 3.1					*/
/*								   		*/
/* (C) 1990, 1991, 1992, 1993, 1994, 1995, 1996					*/
/* The University Hospital of Geneva						*/
/* All Rights Reserved								*/
/*										*/
/********************************************************************************/

#ifdef Mac
#pragma segment papy3
#endif

/* ------------------------- includes ---------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "setjmp.h"

#include "JPEGLess.h"	/* interface for JPEG lossless decompressor */

#include "jpeglib.h"	/* interface for JPEG lossy decompressor */

#ifndef Papyrus3H 
#include "Papyrus3.h"
#endif


/* ------------------------- forward declarations ----------------------------- */

PapyShort
PutBufferInGroup3 (PapyShort, unsigned char *, ELEMENT *, PapyUShort, PapyULong, 
		   PapyULong *, PapyLong);
		  
extern PapyShort read_group3 (PapyShort, PapyUShort *, unsigned char **,
			      PapyULong *, PapyULong *);		  
extern PapyULong ComputeUndefinedGroupLength3 	 (PapyShort, PapyLong);
extern PapyShort ComputeUndefinedSequenceLength3 (PapyShort, PapyULong *);
extern PapyShort ComputeUndefinedItemLength3	 (PapyShort, PapyULong *);
extern PapyUShort Extract2Bytes (unsigned char *, PapyULong *);
extern PapyULong  Extract4Bytes (unsigned char *, PapyULong *);
	  



/********************************************************************************/
/*										*/
/*	Papy3GetElement : gets the value(s) of the specified element		*/
/* 	return : the value(s) of the element					*/
/*										*/
/********************************************************************************/

VALUE_T * CALLINGCONV
Papy3GetElement (ELEMENT *grOrMod, int element, PapyULong *nbValue, int *elemType)

/*ELEMENT 	*grOrMod;		     ptr on the group or the module */
/*PapyShort	element;	   the position of the element in the group */
/*PapyULong 	*nbValue;		       the number of values to read */
/*PapyShort	*elemType;		    what is the type of the element */
{
  ELEMENT *pElem;	       /* work pointer on the elements of the group */
  VALUE_T *value_t;			    /* the value we are looking for */
  
  
  if (grOrMod == NULL) return NULL;
  
  pElem  = grOrMod;
  pElem += element;			   /* points on the desired element */
  
  *elemType = pElem->vr;	 /* is it a short a long or an ASCII char ? */
  
  if (pElem->nb_val > 0L)		    /* there is an introduced value */
  {
    *nbValue = pElem->nb_val;
    value_t = pElem->value;
  } /* then */
  
  else
  {
     *nbValue = 0L;
     value_t = NULL;
  } /* else */
  
  return value_t;
  
} /* endof Papy3GetElement */



/********************************************************************************/
/*									 	*/
/*	ExtractJPEGlossless : gets and decode JPEG lossless pixel data		*/
/*	Nota : the PAPYRUS toolkit JPEG utility is based in part on the work of	*/
/*	the Independent JPEG Group (see copyright file included)		*/
/* 	return : the image							*/
/*										*/
/********************************************************************************/

PapyShort
ExtractJPEGlossless (PapyShort fileNb, PapyUHChar *buffer, PapyULong pixelStart,
		     PapyULong *offsetTable, int imageNb)
{
  PapyUHChar	tmpBuf [256];
  PapyUHChar	*tmpBufPtr;
  PapyShort	err;
  PapyUShort	group, element;
  PapyULong	i, pos, length;
  
  
  /* position the file pointer at the begining of the pixel datas */
  Papy3FSeek (Papy_file [fileNb], SEEK_SET, (PapyLong) (pixelStart + offsetTable [imageNb - 1]));
  
  /* read 8 chars from the file */
  tmpBufPtr = (PapyUHChar *) &tmpBuf [0];
  i = 8L; 					/* grNb, elemNb & elemLength */
  if ((err = (PapyShort) Papy3FRead (Papy_file [fileNb], &i, 1L, tmpBufPtr)) < 0)
  {
    Papy3FClose (&Papy_file [fileNb]);
    RETURN (err);
  } /* if */
    
  pos = 0L;
  group   = Extract2Bytes (tmpBufPtr, &pos);
  element = Extract2Bytes (tmpBufPtr, &pos);
    
  /* extract the element length */
  length = Extract4Bytes (tmpBufPtr, &pos);
  
  /* if length is 0xFFFFFFFF (undefined) we have to extract it HERE !!! */
  
  /* Pixel data fragment not found when expected */
  if ((group != 0xFFFE) || (element != 0xE000)) RETURN (papBadArgument);
    
  /* Get ready to receive decompressed rows */
  JPEGLosslessDecodeImage (Papy_file [fileNb], (PapyUHShort *) buffer, x0028BitsAllocated [fileNb], length);
  
  return 0;
  
} /* endof ExtractJPEGlossless */



/********************************************************************************/
/*									 	*/
/*	Needed for the error manager of the JPEG lossy library			*/
/*										*/
/********************************************************************************/

struct my_error_mgr {
  struct jpeg_error_mgr pub;	/* "public" fields */

  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/********************************************************************************/
/*									 	*/
/* Here's the routine that will replace the standard error_exit method: 	*/
/* for JPEG lossy								*/
/*									 	*/
/********************************************************************************/

METHODDEF void
my_error_exit (j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  /* Always display the message. */
  /* We could postpone this until after returning, if we chose. */
  (*cinfo->err->output_message) (cinfo);

  /* Return control to the setjmp point */
#ifdef Mac
  longjmp(myerr->setjmp_buffer, 1);
#endif

} /* endofunction my_error_exit */



/********************************************************************************/
/*									 	*/
/*	ExtractJPEGlossy : gets and decode JPEG lossy pixel data		*/
/*	Nota : the PAPYRUS toolkit JPEG utility is based in part on the work of	*/
/*	the Independent JPEG Group (see copyright file included)		*/
/* 	return : the image							*/
/*										*/
/********************************************************************************/

PapyShort
ExtractJPEGlossy (PapyShort fileNb, PapyUHChar *image8, PapyULong pixelStart,
		  PapyULong *offsetTable, int imageNb)
{
  struct my_error_mgr		jerr;		 /* the JPEG error manager var */
  struct jpeg_decompress_struct	cinfo;
  PapyUHChar			tmpBuf [256];
  PapyUHChar			*tmpBufPtr;
  PapyUShort			group, element;
  PapyShort			err;
  PapyULong			i, pos;
  unsigned char			*buffer;
  int 				row_stride;	 /* physical row width in output buffer */
  int				loop;
  PapyUHChar			*wrkCh; /* ptr to the image */
    
    
  /* position the file pointer to the begining of the image */
  Papy3FSeek (Papy_file [fileNb], SEEK_SET, (PapyLong) (pixelStart + offsetTable [imageNb - 1]));
  
  /* read 8 chars from the file */
  tmpBufPtr = (PapyUHChar *) &tmpBuf [0];
  i = 8L; 					/* grNb, elemNb & elemLength */
  if ((err = (PapyShort) Papy3FRead (Papy_file [fileNb], &i, 1L, tmpBufPtr)) < 0)
  {
    Papy3FClose (&Papy_file [fileNb]);
    RETURN (err);
  } /* if */
    
  pos = 0L;
  group   = Extract2Bytes (tmpBufPtr, &pos);
  element = Extract2Bytes (tmpBufPtr, &pos);
    
  /* Pixel data fragment not found when expected */
  if ((group != 0xFFFE) || (element != 0xE000)) RETURN (papBadArgument);
  
  /* We set up the normal JPEG error routines, then override error_exit. */
  cinfo.err = jpeg_std_error (&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  /* Establish the setjmp return context for my_error_exit to use. */
#ifdef Mac
  if (setjmp (jerr.setjmp_buffer)) 
  {
    jpeg_destroy_decompress (&cinfo);
    return NULL;
  }/* if */
#endif
    
  /* initialize the JPEG decompression object */
  jpeg_create_decompress (&cinfo);
    
  /* specify the data source */
  jpeg_stdio_src (&cinfo, Papy_file [fileNb]);
    
  /* read file parameter */
  (void) jpeg_read_header (&cinfo, TRUE);
    
  /* start the decompressor (set the decompression default params) */
  (void) jpeg_start_decompress (&cinfo);
    
  /* JSAMPLEs per row in output buffer */
  row_stride = cinfo.output_width * cinfo.output_components;
    
  /* allocate a one-row-high sample array that will go away when done with image */
  buffer = (unsigned char *) emalloc3 ((PapyULong) row_stride);
    
  wrkCh = image8;
  /* decompress the image line by line */
  while (cinfo.output_scanline < cinfo.output_height) 
  {
    (void) jpeg_read_scanlines (&cinfo, (JSAMPARRAY) &buffer, 1);
      
    /* put the scanline in the image */
    for (loop = 0; loop < cinfo.output_width; loop ++, wrkCh++)
      *wrkCh = buffer [loop]; 
  } /* while ...line by line decompression of the image */
    
  /* frees the row used by the decompressor */
  efree3 ((void **) &buffer);
    
  /* tell the JPEG decompressor we have finish the decompression */
  (void) jpeg_finish_decompress (&cinfo);

} /* endof ExtractJPEGlossy */



/********************************************************************************/
/*									 	*/
/*     				DecodeRLESegment				*/
/*									 	*/
/********************************************************************************/

void
DecodeRLESegment (PapyUHShort *image, PapyUHChar *rle, PapyULong length, 
		  int segtot, int segnb)
/* decode a RLE segment							*/
/* image  : pointer on real image (8 or 16 bits)			*/
/* rle    : pointer on rle buffer (8bits)				*/
/* length : length of rle buffer					*/
/* segtot : total number of segments (1 or 2)				*/
/* segnb  : number of current segment (1 or 2) (only if segtot == 2)	*/
{
  PapyLong		j, indj;
  PapyUHChar		*pix;
  PapyUHChar		val;
  char			code;
  PapyShort		ii, iimax;


  /* *** single segment *** */
  /* ********************** */
  
  if (segtot == 1) 
  { 
    /* convert rle into real image */
    pix = (PapyUHChar *) image;
    indj = 0L;
    for (j = 0L; j < length; ) 
    {
      code = (char) rle [j];
      j++; /* yes, I know but do not move it */
      /* sequence of different bytes */
      if (code == 0) 
      {
	if (j < length-1) pix [indj++] = rle [j++];
      }/* if */
	  
      else if ((code > 0) && (code <= 127)) 
      {
	for (ii = 0; ii < (code + 1); ii++) pix [indj++] = rle [j++];
      } /* if */
	  
      /* repetition of the same byte */
      else if ((code <= -1) && (code >= -127)) 
      {
        val = rle [j++];
        iimax = -code;
        for (ii  =0; ii <= iimax; ii++) pix [indj++] = val;
      } /* if */
	  
    } /* for */
  } /* if ...single segment */
  
  /* *** two segments *** */
  /* ******************** */
  
  else if (segtot == 2) 
  {
    /* we assume it is a 16 bit image	*/
    /* convert rle into real image	*/
    pix = (PapyUHChar *) image;
    indj = 0L;
    if (segnb == 2) indj++;
    for (j = 0L; j < length; ) 
    {
      code = (char) rle [j];
      j++; /* yes, I know but do not move it */
      /* sequence of different bytes */
      if (code == 0) 
      {
	if (j < length - 1) pix [indj] = rle [j++];
	indj = indj + 2;
      } /* if */
	  
      else if ((code > 0) && (code <= 127)) 
      {
	for (ii = 0; ii < (code + 1); ii++) 
	{
	  pix [indj] = rle [j++];
  	  indj = indj + 2;
	} /* for */
      } /* if */
	   
      /* repetition of the same byte */
      else if ((code <= -1) && (code >= -127)) 
      {
        val = rle [j++];
        iimax = -code;
        for (ii = 0; ii <= iimax; ii++) 
        {
	  pix [indj] = val;
  	  indj = indj + 2;
	} /* for */
      } /* if */

    } /* for */
  } /* if ...two segments */
  
} /* endof DecodeRLESegment */



/********************************************************************************/
/*									 	*/
/*	ExtractRLE : gets and decode a RLE pixel data element			*/
/* 	return : the image							*/
/*										*/
/********************************************************************************/

PapyShort
ExtractRLE (PapyShort fileNb, PapyUHShort *image16, PapyULong pixelStart,
	    PapyULong *offsetTable, int imageNb)
{
  PapyUHChar	tmpBuf [256];
  PapyUHChar	*tmpBufPtr;
  PapyUShort	group, element;
  PapyShort	err;
  PapyULong 	numberSegments, i, pos, length;
  PapyUHChar	*rle;
  long		offset1, offset2, rlelen;
    
    
  /* for each image						*/
  /* FFFE E000 length RLE_header RLE_segment1 RLE_segment2 ...	*/
  /* length is 4 bytes, in the case of a single image		*/
  Papy3FSeek (Papy_file [fileNb], SEEK_SET, (PapyLong) (pixelStart + offsetTable [imageNb - 1]));
  
  /* read 8 chars from the file */
  tmpBufPtr = (PapyUHChar *) &tmpBuf [0];
  i = 8L; 					/* grNb, elemNb & elemLength */
  if ((err = (PapyShort) Papy3FRead (Papy_file [fileNb], &i, 1L, tmpBufPtr)) < 0)
  {
    Papy3FClose (&Papy_file [fileNb]);
    RETURN (err);
  } /* if */
    
  pos = 0L;
  group   = Extract2Bytes (tmpBufPtr, &pos);
  element = Extract2Bytes (tmpBufPtr, &pos);
  length  = Extract4Bytes (tmpBufPtr, &pos);
    
  /* Pixel data fragment not found when expected */
  if ((group != 0xFFFE) || (element != 0xE000)) RETURN (papBadArgument);
  
  /* read 4 chars from the file = number of segments */
  tmpBufPtr = (PapyUHChar *) &tmpBuf [0];
  i = 4L;
  pos = 0L;					/* grNb, elemNb & elemLength */
  if ((err = (PapyShort) Papy3FRead (Papy_file [fileNb], &i, 1L, tmpBufPtr)) < 0)
  {
    Papy3FClose (&Papy_file [fileNb]);
    RETURN (err);
  } /* if */
  numberSegments = Extract4Bytes (tmpBufPtr, &pos);
  if (numberSegments > 2L) RETURN (papWrongValue); /* we allow to read 8 and 16 bit images */
    
  /* read offset1 and offset2 and skip 52 bytes */
  tmpBufPtr = (PapyUHChar *) &tmpBuf [0];
  i = 8L;
  pos = 0L;					/* grNb, elemNb & elemLength */
  if ((err = (PapyShort) Papy3FRead (Papy_file [fileNb], &i, 1L, tmpBufPtr)) < 0)
  {
    Papy3FClose (&Papy_file [fileNb]);
    RETURN (err);
  } /* if */
  offset1 = Extract4Bytes (tmpBufPtr, &pos);
  offset2 = Extract4Bytes (tmpBufPtr, &pos);
  Papy3FSeek (Papy_file [fileNb], SEEK_CUR, (PapyLong) 52L);
    
  if (numberSegments == 1) 
  {
    /* read rle image */
    rlelen = length - 64L;
    rle = (PapyUHChar *) emalloc3 ((PapyULong) (rlelen + 10L));
    /* extract the image from the file */
    err = Papy3FRead (Papy_file [fileNb], (PapyULong *) &rlelen, 1L, (void *) rle);

    DecodeRLESegment (image16, rle, rlelen, numberSegments, 1);
    /* delete rle image */
    efree3 ((void **) &rle);
  }/* if ...single segment */
    
  else if (numberSegments == 2) 
  {
    /* deal with first segment */
    rlelen = offset2 - 64L;
    rle = (PapyUHChar *) emalloc3 ((PapyULong) (rlelen + 10L));
    /* extract the image from the file */
    err = Papy3FRead (Papy_file [fileNb], (PapyULong *) &rlelen, 1L, (void *) rle);
    DecodeRLESegment (image16, rle, rlelen, numberSegments, 2);
    /* delete rle image */
    efree3 ((void **) &rle);
      
    /* deal with second segment */
    rlelen = length - offset2;
    rle = (PapyUHChar *) emalloc3 ((PapyULong) (rlelen + 10L));
    /* extract the image from the file */
    err = Papy3FRead (Papy_file [fileNb], (PapyULong *) &rlelen, 1L, (void *) rle);
    DecodeRLESegment (image16, rle, rlelen, numberSegments, 1);
    /* delete rle image */
    efree3 ((void **) &rle);
      
  }/* if ...two segments */
  
  return 0;
    
} /* endof ExtractRLE */



/********************************************************************************/
/*									 	*/
/*	Papy3GetPixelData : gets the specified image or icon			*/
/* 	return : the image							*/
/*										*/
/********************************************************************************/

PapyUShort * CALLINGCONV
Papy3GetPixelData (PapyShort fileNb, int imageNb, MODULE *module, int moduleId,
		   enum VR_T theVR)
{
  PapyUHChar	 *buf;
  PapyUHChar	 tmpBuf [256];
  PapyUHChar	 *tmpBufPtr;
  PapyUHChar	 *pChar;
  PapyUHChar	 char0;
  PapyUHChar	 char1;
  PapyUHShort	 *pUShort;
  PapyUHShort	 uShort1;
  PapyUHShort	 uShort2;
  PapyShort	 err;
  int		 frameCount = 1, loop, ok;
  PAPY_FILE	 fp;
  PapyULong	 bytesToRead, i, uLong, pos, *offsetTable;
  PapyULong	 refPoint, pixelStart;
  ELEMENT 	 *pElem;	/* work pointer on the element of the module */
  
  
  /* some usefull tests */
  if (imageNb > arrNbImages [fileNb] || moduleId > END_MODULE) return NULL;
  
  offsetTable = NULL;
  
  /* get the file pointer from the file number */
  fp = Papy_file [fileNb];

  /* position the file pointer to the pixel data to read */
  switch (moduleId)
  {
    case IconImage :
      /* only allow to get an icon from a PAPYRUS 3 file */
      if (isPapyFile [fileNb] != 1) return NULL;
      
      /* it is one of the pointer sequence module, so go to the given ptr sequence */
      if (Papy3FSeek (Papy_file [fileNb], (int) SEEK_SET, (PapyLong) offsetToPtrSeq [fileNb] + 8L) != 0)
        return NULL;
      
      /* look for the given item of the ptr seq */
      for (i = 1L; i < (int) imageNb; i++)
      {
        bytesToRead = Papy3ExtractItemLength (fileNb);
        if (Papy3FSeek (Papy_file [fileNb], (int) SEEK_CUR, (PapyLong) bytesToRead) != 0)
          return NULL;
      } /* for */
      
      /* then points to the first element of the item */
      if (Papy3FSeek (Papy_file [fileNb], (int) SEEK_CUR, (PapyLong) 8L) != 0) 
        return NULL;
        
      /* look now for the right group, i.e. image */
      err = Papy3GotoGroupNb (fileNb, 0x7FE0);
      /* ... then the right element */
      err = Papy3GotoElemNb (fileNb, 0x7FE0, 0x0010, &bytesToRead);
        
      /* jump over the description of the element */
      if (Papy3FSeek (Papy_file [fileNb], (int) SEEK_CUR, (PapyLong) 8L) != 0) 
        return NULL;

      pElem = module + papPixelDataII;
      break;
    
    case ImagePixel :
      /* go to the begining of the specified image */
      if (Papy3FSeek (Papy_file [fileNb], (int) SEEK_SET, (PapyLong) *(refPixelOffset [fileNb] + imageNb - 1)) != 0)
        return NULL;
  
       pElem = module + papPixelData;
      break;
    
    default :
      return NULL;
      break;
  } /* switch */
    
  
  /* get the size of the pixel data */
  if (moduleId == IconImage) 
    bytesToRead = pElem->length;
  else
    bytesToRead = (PapyULong) x0028Rows [fileNb] * (PapyULong) x0028Columns [fileNb] * 
    		  (PapyULong) (((x0028BitsAllocated [fileNb] - 1) / 8) + 1L);
    
  /* if it is a RGB image, multiply the bytes to read by 3 */
  if (moduleId == ImagePixel && arrPhotoInterpret [fileNb] == RGB) bytesToRead *= 3L;
  
  /* allocate the memory for the pixel data */
  buf = (PapyUHChar *) emalloc3 ((PapyULong) bytesToRead);
  
  
  /* image reading depending on the image encoding */
  
  /* first test if the images is not encoded */
  if (moduleId == IconImage || 
      (arrCompression [fileNb]     == NONE &&
       (arrPhotoInterpret [fileNb] == MONOCHROME1 ||
        arrPhotoInterpret [fileNb] == MONOCHROME2 ||
        arrPhotoInterpret [fileNb] == PALETTE     ||
        arrPhotoInterpret [fileNb] == RGB)))
  {    
    /* if it is a DICOM file then jump to the right image */
    if (isPapyFile [fileNb] == 0 || isPapyFile [fileNb] == 2)
      err = Papy3FSeek (fp, SEEK_CUR, (PapyLong) (bytesToRead * (imageNb -1)));
    
    /* read bytesToRead bytes from the file */
    if ((err = (PapyShort) Papy3FRead (fp, &bytesToRead, 1L, buf)) < 0)
    {
      err = Papy3FClose (&fp);
      efree3 ((void **) &buf);
      return NULL;
    } /* if */
    
    /* swap the bytes if necessary */
    if (moduleId == ImagePixel && x0028BitsAllocated [fileNb] > 8)
    {
      pUShort = (PapyUHShort *) buf;
      
      for (i = 0L, pChar = buf; i < (bytesToRead / 2); i++, pChar += 2, pUShort++)
      {
        char0     = *pChar;
        char1     = *(pChar + 1);
        *pUShort  = (PapyUHShort) char1;
    	*pUShort  = *pUShort << 8;
    	*pUShort |= (PapyUHShort) char0;
      } /* for */
    } /* if ...VR = OW (16 bits) */
    
  } /* if ...module IconImage or photometric interpretation is monochrome/palette/rgb */
  
  /* *** not IconImage module and the pixels are compressed *** */
  else
  {
    /* check to see if there is an offset table, as expected */
    /* so read 8 chars from the file */
    tmpBufPtr = (unsigned char *) &tmpBuf [0];
    i = 8L; 					/* grNb, elemNb & elemLength */
    if ((err = (PapyShort) Papy3FRead (fp, &i, 1L, tmpBufPtr)) < 0)
    {
      err = Papy3FClose (&fp);
      return NULL;
    } /* if */
    
    pos = 0L;
    uShort1 = Extract2Bytes (tmpBufPtr, &pos);
    uShort2 = Extract2Bytes (tmpBufPtr, &pos);
    
    /* test if the values are correct */
    if (uShort1 != 0xFFFE || uShort2 != 0xE000)
      return NULL;
    
    /* offset table size */
    /* extract the element length according to the little-endian syntax */
    uLong = Extract4Bytes (tmpBufPtr, &pos);
    
    if (uLong > 0)
    {
      /* the offset table size does give the number of frames */
      frameCount = (int) (uLong / 4L);
      
      /* allocate room to store the offset table */
      offsetTable = (PapyULong *) emalloc3 ((PapyULong) (frameCount * sizeof (PapyULong)));
 
      for (loop = 0; loop < frameCount; loop++)
      {
        /* read 4 chars from the file */
        i = 4L;
        pos = 0L;
        tmpBufPtr = (unsigned char *) &tmpBuf [0];
        if ((err = (PapyShort) Papy3FRead (fp, &i, 1L, tmpBufPtr)) < 0)
        {
	  err = Papy3FClose (&fp);
	  efree3 ((void **) &offsetTable);
	  return NULL;
        } /* if */
        offsetTable [loop] = Extract4Bytes (tmpBufPtr, &pos);
      } /* for */
     
    } /* if */
    else
    {
      ok = FALSE;
      frameCount = 0;
      
      /* initialize a file reference point */
      Papy3FTell (fp, (PapyLong *) &refPoint);
      
      /* allocate memory for the offset table */
      offsetTable = (PapyULong *) emalloc3 ((PapyULong) (1000L * sizeof (PapyULong)));
      
      while (!ok)
      {
        /* read fragment information : 0xFFFE, 0xE000, length */
        Papy3FTell (fp, (PapyLong *) &pixelStart);
        
        /* read 8 chars from the file */
        i = 8L;
        pos = 0L;
        tmpBufPtr = (unsigned char *) &tmpBuf [0];
        if ((err = (PapyShort) Papy3FRead (fp, &i, 1L, tmpBufPtr)) < 0)
        {
	  err = Papy3FClose (&fp);
	  efree3 ((void **) &offsetTable);
	  return NULL;
        } /* if */
        
        pos = 0L;
        uShort1 = Extract2Bytes (tmpBufPtr, &pos);
        uShort2 = Extract2Bytes (tmpBufPtr, &pos);
        uLong   = Extract4Bytes (tmpBufPtr, &pos);
        
        /* offset table found ? */
        if ((uShort1 == 0xFFFE) && (uShort2 == 0xE000))
        {
          offsetTable [frameCount] = pixelStart - refPoint;
          frameCount ++;
          Papy3FSeek (fp, SEEK_CUR, uLong);
        } /* if */
        else if ((uShort1 == 0xFFFE) && (uShort2 == 0xE0DD)) ok = TRUE;
      
      } /* while */
      
      /* position the file pointer on the first image */
      Papy3FSeek (fp, SEEK_SET, refPoint);
      
    } /* else */
  
    /* get the position of the first pixel */
    Papy3FTell (fp, (PapyLong *) &pixelStart);            
  
    /* in case of a PAPYRUS file, there should be only one frame. */
    /* The positioning of the file pointer to the right image has already been performed */
    if (isPapyFile [fileNb] == 1) imageNb = 1;
    
    
    /*  *** different ways of reading depending on the compression algorithm *** */

  
    /********************************************************************/
    /*******************     Lossless JPEG     **************************/
    /********************************************************************/
    if (arrCompression [fileNb] == JPEG_LOSSLESS)
    {
      err = ExtractJPEGlossless (fileNb, buf, pixelStart, offsetTable, imageNb);
    } /* if ...JPEG lossless */

  
    /********************************************************************/
    /*******************     Lossy JPEG     *****************************/
    /********************************************************************/
    else if (arrCompression [fileNb] == JPEG_LOSSY)
    {
      err = ExtractJPEGlossy (fileNb, buf, pixelStart, offsetTable, imageNb);
    } /* if ...JPEG lossy */

  
    /********************************************************************/
    /*******************     RLE     ************************************/
    /********************************************************************/
    else if (arrCompression [fileNb] == RLE)
    {
      err = ExtractRLE (fileNb, (PapyUHShort *) buf, pixelStart, offsetTable, imageNb);
    } /* if ...Run Length Encoding */

  
    /********************************************************************/
    /*******************     unknown     ********************************/
    /********************************************************************/
    else
    {
      /* black image, that is better than nothing ... */
      for (i = 0L; i < bytesToRead; i++) buf [i] = 0;
    } /* if ...nothing known */
    
  } /* else ...not icon image or compressed pixel data */
  
  
  /* allocate room in the element in order to put the pixel data in the module */
  pElem->value  = (VALUE_T *) emalloc3 ((PapyULong) sizeof (VALUE_T));
  pElem->nb_val = 1L;
  
  /* extract the pixel data depending on the value representation */
  pElem->vr = theVR;
  switch (theVR)
  {
    case OB :
      pElem->value->a = (char *) buf;
      break;
      
    case OW :
      pElem->value->ow = (PapyUShort *) buf;
      break;
      
  } /* switch ... value representation */
  
  if (offsetTable != NULL) efree3 ((void **) &offsetTable);
    
  return (PapyUShort *) buf;
  
} /* endof Papy3GetPixelData */


/********************************************************************************/
/*										*/
/*	Extract2Bytes : extract a 2-Bytes value (USS, SS or AT) from the buf and*/
/*	increment pos accordingly.						*/
/* 	return : the extracted value						*/
/*										*/
/********************************************************************************/

PapyUShort
Extract2Bytes (unsigned char *buf, PapyULong *pos)

/*unsigned char *buf;				 the buffer to read from */
/*PapyULong 	*pos;			      the position in the buffer */
{
    PapyUShort 		myUShort;
    unsigned char	*charPtr;


    /* points to the right place in the buffer */
    charPtr  = buf;
    charPtr += *pos;
    /* updates the current position in the read buffer */
    *pos += 2;
    
    /* extract the element according to the little-endian syntax */
    myUShort  = (PapyUShort) (*(charPtr + 1));
    myUShort  = myUShort << 8;
    myUShort |= (PapyUShort) *charPtr;
    
    return myUShort;

} /* endof Extract2Bytes */



/********************************************************************************/
/*										*/
/*	Extract4Bytes : extract a 4-Bytes value (UL, SL or FL) of the buf and 	*/
/*	increment pos accordingly.						*/
/* 	return : the extracted value					 	*/
/*										*/
/********************************************************************************/

PapyULong
Extract4Bytes (unsigned char *buf, PapyULong *pos)

/*unsigned char *buf;				 the buffer to read from */
/*PapyULong 	*pos;			      the position in the buffer */
{
    unsigned char	*charPtr;
    PapyULong		myULong = 0L, tmpULong;
    
    
    /* points to the right place in the buffer */
    charPtr  = buf;
    charPtr += *pos;
    /* updates the current position in the read buffer */
    *pos += 4;
    
    /* extract the element according to the little-endian syntax */
    tmpULong  = (PapyULong) (*(charPtr + 3));
    tmpULong  = tmpULong << 24;
    myULong  |= tmpULong;
    tmpULong  = (PapyULong) (*(charPtr + 2));
    tmpULong  = tmpULong << 16;
    myULong  |= tmpULong;
    tmpULong  = (PapyULong) (*(charPtr + 1));
    tmpULong  = tmpULong << 8;
    myULong  |= tmpULong;
    tmpULong  = (PapyULong) *charPtr;
    myULong  |= tmpULong;
    
    return myULong;
    
} /* endof Extract4Bytes */



/********************************************************************************/
/*										*/
/*	Extract8Bytes : extract a 8-Bytes value (FD) of the buf and 		*/
/*	increment pos accordingly.						*/
/* 	return : the extracted value					 	*/
/*										*/
/********************************************************************************/

PapyFloatDouble
Extract8Bytes (unsigned char *buf, PapyULong *pos)

/*unsigned char *buf;				 the buffer to read from */
/*PapyULong 	*pos;			      the position in the buffer */
{
    unsigned char	*charPtr, doublePtr [8], i;
    PapyFloatDouble	*myFloatDouble;
    
    
    /* points to the right place in the buffer */
    charPtr  = buf;
    charPtr += *pos;
    /* updates the current position in the read buffer */
    *pos += 8;
    
    /* extract the element according to the little-endian syntax */
    for (i = 0; i < 4; i++)
    {
      doublePtr [2 * i]       = *charPtr;
      doublePtr [(2 * i) + 1] = *(charPtr + 1);
      charPtr += 2;
    } /* for ...extraction of the value */
    
    myFloatDouble = (PapyFloatDouble *) &doublePtr;
    
    return *myFloatDouble;
    
} /* endof Extract8Bytes */



/********************************************************************************/
/*										*/
/*	ExtractString : extract a string from the buffer and put it in the 	*/
/*	given element. Increment pos accordingly.				*/
/*										*/
/********************************************************************************/

void
ExtractString (ELEMENT *elem, unsigned char *buf, PapyULong *bufPos, 
	       PapyULong elemLength)
{
  char			*string, *p, *char_val, *char_wrk;
  unsigned char		*tmp;
  int			ii, j, stringLength;
  
		  				   /* 1 for the string terminator */
  string = (char *) emalloc3 ((PapyULong) (elemLength + 1));
  p = string;
  tmp = buf;
  /* extract the element from the buffer */
  for (ii = 0L; ii < elemLength; ii++, (*bufPos)++)
    *(p++) = tmp [*bufPos];
    
  string [ii] = '\0';
    
  char_val = string;
 
  stringLength = strlen (char_val); 
          
  elem->nb_val = 1L;     /* number of strings */
  char_wrk = char_val;
          
  /* count the number of strings */
  for (j = 0; j < stringLength; j++, char_wrk ++)
  {
    /* value separator */
    if (*char_wrk == '\\') 
    {
      elem->nb_val++;
      *char_wrk = '\0';
    } /* if */
  } /* for ...counting the number of values */
          
  elem->value = (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val, (PapyULong) sizeof (VALUE_T));
          	
  /* extraction of the strings */	
  for (j = 0, char_wrk = char_val; 
       j < elem->nb_val;
       j ++, char_wrk += stringLength + 1)
  {
    stringLength = strlen (char_wrk);
		    
    /* addition to delete the blank if odd string */
    if (elem->vr == UI)
    {
      /* suppress the blank by shifting all the chars to the left */
      /* old was : char_wrk [stringLength - 1] == '0') */
      if (char_wrk [stringLength - 1] == 0x00) 
	char_wrk [stringLength - 1] = '\0';
    } /* then ...VR = UI */
    else
    {
      if (char_wrk [stringLength - 1] == ' ')
	char_wrk [stringLength - 1] = '\0';
    } /* else ...VR <> UI */
		    
    elem->value [j].a = char_wrk;

  } /* for ...extraction of the strings */
          
} /* endof ExtractString */


										
/********************************************************************************/
/*									 	*/
/*	PutBufferInElement3 : fill_in an element structure (one element) 	*/
/* 	from a buffer made of unsigned chars					*/
/* 	return : standard error message						*/
/*									  	*/
/********************************************************************************/

PapyShort
PutBufferInElement3 (PapyShort fileNb, unsigned char *buff, PapyULong elemLength,
		    ELEMENT *elem, PapyULong *bufPos, PapyLong initFilePos)
{
  ITEM 			*seqItem, *dsItem;
  OBJECT		*object;
  ELEMENT		*seqGroup;
  VALUE_T		*pValueT;
  unsigned char 	*tmp0, tmp1,  *pChar;
  unsigned char 	doublePtr [8], incr;
  PapyLong		currFilePos, initialFilePos = initFilePos;
  PapyULong		ii, i, j, posInSeq, seqSize, seqGrSize, imLength;
  PapyULong 		tmpULong, myULong = 0L;
  PapyUShort	 	seqGrNb, elem_nb, *imOW, *tmpUs;
  char 			*char_val, *char_wrk; 
  char			*string, *p;
  int 			enumSeqNb, stringLength, first, isUndefItemL;
  PapyShort		err;

  /*## Start of IAC Addition */
  PapyULong posSoFar = 0L; /* size of previous items in sequence */
  /*## End of IAC Addition */


  /* extract the element depending on the value representation */
  switch (elem->vr)
  {
    case SS :				/* 16 bits binary signed */
      elem->nb_val = (PapyULong) (elemLength / 2);
      elem->value = (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val,
					  (PapyULong) sizeof (VALUE_T));
      pValueT = elem->value;
      for (j = 0; j < elem->nb_val; j++, pValueT++)
      {
        /* points to the right place in the buffer */
        tmp0  = buff;
        tmp0 += *bufPos;
        /* updates the current position in the read buffer */
        *bufPos += 2L;  
        /* extract the element according to the little-endian syntax */
        pValueT->ss  = (PapyUShort) (*(tmp0 + 1));
        pValueT->ss  = pValueT->ss << 8;
        pValueT->ss |= (PapyUShort) *tmp0;
      } /* for */
	    
      break; /* SS */

	  
    case AT :
    case USS :				/* 16 bits binary unsigned */
      elem->nb_val = (PapyULong) (elemLength / 2);
      elem->value = (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val,
					  (PapyULong) sizeof (VALUE_T));
      pValueT = elem->value;
      for (j = 0; j < elem->nb_val; j++, pValueT++)
      {
        /* points to the right place in the buffer */
        tmp0  = buff;
        tmp0 += *bufPos;
        /* updates the current position in the read buffer */
        *bufPos += 2L;  
        /* extract the element according to the little-endian syntax */
        pValueT->us  = (PapyUShort) (*(tmp0 + 1));
        pValueT->us  = pValueT->us << 8;
        pValueT->us |= (PapyUShort) *tmp0;
      } /* for */

      break; /* USS */
	    
        
    case SL :				/* 32 bits binary signed */
      elem->nb_val = (PapyULong) (elemLength / 4);
      elem->value =  (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val,
					   (PapyULong) sizeof (VALUE_T));
      pValueT = elem->value;
      for (j = 0; j < elem->nb_val; j++, pValueT++)
      {
        /* points to the right place in the buffer */
        tmp0  = buff;
        tmp0 += *bufPos;
        /* updates the current position in the read buffer */
        *bufPos += 4L;
        /* extract the element according to the little-endian syntax */
        tmpULong      = (PapyULong) (*(tmp0 + 3));
        tmpULong      = tmpULong << 24;
        myULong	      = tmpULong;
        tmpULong      = (PapyULong) (*(tmp0 + 2));
        tmpULong      = tmpULong << 16;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) (*(tmp0 + 1));
        tmpULong      = tmpULong << 8;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) *tmp0;
        myULong      |= tmpULong;
        pValueT->sl   = myULong;
      } /* for */

      break; /* SL */
	  
	  
    case UL :				/* 32 bits binary unsigned */
      elem->nb_val = (PapyULong) (elemLength / 4);
      elem->value =  (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val,
					   (PapyULong) sizeof (VALUE_T));
      pValueT = elem->value;
      for (j = 0; j < elem->nb_val; j++, pValueT++)
      {
        /* points to the right place in the buffer */
        tmp0  = buff;
        tmp0 += *bufPos;
        /* updates the current position in the read buffer */
        *bufPos += 4L;
        /* extract the element according to the little-endian syntax */
        tmpULong      = (PapyULong) (*(tmp0 + 3));
        tmpULong      = tmpULong << 24;
        myULong	      = tmpULong;
        tmpULong      = (PapyULong) (*(tmp0 + 2));
        tmpULong      = tmpULong << 16;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) (*(tmp0 + 1));
        tmpULong      = tmpULong << 8;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) *tmp0;
        myULong      |= tmpULong;
        pValueT->ul   = myULong;
      } /* for */

      break; /* UL */
	  
	  
    case FL :				/* 32 bits binary floating */
      elem->nb_val = (PapyULong) (elemLength / 4);
      elem->value =  (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val,
					   (PapyULong) sizeof (VALUE_T));
      pValueT = elem->value;
      for (j = 0; j < elem->nb_val; j++, pValueT++)
      {
        /* points to the right place in the buffer */
        tmp0  = buff;
        tmp0 += *bufPos;
        /* updates the current position in the read buffer */
        *bufPos += 4L;
        /* extract the element according to the little-endian syntax */
        tmpULong      = (PapyULong) (*(tmp0 + 3));
        tmpULong      = tmpULong << 24;
        myULong	      = tmpULong;
        tmpULong      = (PapyULong) (*(tmp0 + 2));
        tmpULong      = tmpULong << 16;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) (*(tmp0 + 1));
        tmpULong      = tmpULong << 8;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) *tmp0;
        myULong      |= tmpULong;
        pValueT->fl   = myULong;
      } /* for */

      break; /* FL */
	  
	  
    case FD :				/* 64 bits binary floating */
      elem->nb_val = (PapyULong) (elemLength / 8);
      elem->value =  (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val,
					   (PapyULong) sizeof (VALUE_T));
      pValueT = elem->value;
      for (j = 0; j < elem->nb_val; j++, pValueT++)
      {
        /* points to the right place in the buffer */
        tmp0  = buff;
        tmp0 += *bufPos;
        /* updates the current position in the read buffer */
        *bufPos += 8L;
    
        /* extract the element according to the little-endian syntax */
        for (incr = 0; incr < 4; incr++)
        {
          doublePtr [2 * incr]       = *tmp0;
          doublePtr [(2 * incr) + 1] = *(tmp0 + 1);
          tmp0 += 2;
        } /* for ...extraction of the value */
    
        pValueT->fd = *((PapyFloatDouble *) &doublePtr);
        
      } /* for */

      break; /* FD */
    
    case OB :				/* 1 byte image  */
      elem->nb_val = (PapyULong) 1L;
      elem->value  = (VALUE_T *) emalloc3 ((PapyULong) sizeof (VALUE_T));
      
      /* allocate room for the element */
      pChar = (unsigned char *) emalloc3 ((PapyULong) elemLength);
      
      /* copy the bits of the image to the value */
      tmp0 = pChar;
      buff += *bufPos;
      for (i = 0L; i < elemLength; tmp0++, buff++, i++)
      {
        *tmp0 = *buff;
      } /* for */
      
      elem->value->a = (char *) pChar;
      *bufPos += elemLength;
      break; /* OB */
    
    case OW :				/* 2 Bytes image */
      elem->nb_val = (PapyULong) 1L;
      elem->value  = (VALUE_T *) emalloc3 ((PapyULong) sizeof (VALUE_T));
      imLength     = elemLength / 2;
      
      /* pixel data */
      if (elem->group == 0x7FE0 && elem->element == 0x0010)
      {
#ifndef __alpha
         /* swap the bytes (little endian) */
        for (i = 0L, pChar = buff + (*bufPos); i < imLength; i++, pChar += 2)
        {
          tmp1 	       = *(pChar + 1);
          *(pChar + 1) = *pChar;
          *pChar       = tmp1;
        } /* for */
#endif
        
        elem->value->ow = (PapyUShort *) (buff + (*bufPos));
      } /* if ...pixel data */
      else /* not pixel data */
      {
        imOW = (PapyUShort *) ecalloc3 ((PapyULong) imLength, 
        			        (PapyULong) sizeof (PapyUShort));
        for (i = 0L, tmpUs = imOW, buff += *bufPos; i < imLength; i++, tmpUs++, buff += 2)
        {
          *tmpUs  = (PapyUShort) (*(buff + 1));
    	  *tmpUs  = *tmpUs << 8;
    	  *tmpUs |= (PapyUShort) *buff;
        } /* for */
        
        elem->value->ow = imOW;
      } /* else ...not pixel data */
      
      *bufPos += elemLength;
      break; /* OW */
	    
	  
    case SQ :				/* sequence */
      /* if not the pointer sequence or the image sequence extract the seq */
      if (!(elem->group == 0x0041 && 
            (elem->element == Papy3EnumToElemNb (elem, papPointerSequenceGr) ||
             elem->element == Papy3EnumToElemNb (elem, papImageSequenceGr))))
      {
        elem->nb_val = 1L;
	elem->value  = (VALUE_T *) emalloc3 ((PapyULong) sizeof (VALUE_T));
	elem->value->sq = NULL;
	      
	      
	/* loop on the items of the sequence */
	posInSeq     = 0L;
	first        = TRUE;
	isUndefItemL = FALSE;
	while (elemLength > posInSeq)
	{
	  /* read the basic info on the item */
	  if ((seqGrNb = Extract2Bytes (buff, bufPos)) != 0xFFFE)
	    RETURN (papGroupNumber);
	  if ((elem_nb = Extract2Bytes (buff, bufPos)) != 0xE000)
	    RETURN (papElemNumber);
	  seqSize = Extract4Bytes (buff, bufPos);
  		
  	  posInSeq += 8L;	/* size of the item delimiter */
	  
	  /* if undefined item length, compute it */
	  if (seqSize == 0xFFFFFFFF)
	  {
	    /* set a boolean for futur computing of the seq length */
	    isUndefItemL = TRUE;
	    
	    /* get the current position of the file pointer */
	    err = Papy3FTell (Papy_file [fileNb], &currFilePos);
	    /* position the file pointer at the begining of the item */
	    err = Papy3FSeek (Papy_file [fileNb], SEEK_SET, (initialFilePos + (PapyLong) (*bufPos)));
	    
	    /* computes the item length from the file */
	    seqSize = 0L;
	    err = ComputeUndefinedItemLength3 (fileNb, &seqSize);
	    
	    /* reset the file pointer to its previous position */
	    err = Papy3FSeek (Papy_file [fileNb], SEEK_SET, currFilePos);
	  } /* if */

	  /* creates an empty object that will point to the list of groups */
	  object = (OBJECT *) emalloc3 ((PapyULong) sizeof (OBJECT));
  	  object->whoAmI        = papItem;
 	  object->item          = NULL;
 	  object->module        = NULL;
 	  object->group         = NULL;
	  object->tmpFileLength = 0L;
  
  	  seqItem = InsertLastInList (&(elem->value->sq), object);
  	  
  	  /* keep track of the place where to insert a new object (group) */
  	  seqItem = object->item;
 		
	  /*## Start of IAC Addition */
 	  /* sequence Size needs to be increased by the total size of previous items */
  	  seqSize += posSoFar;
  	  posSoFar = seqSize;
	  /*## End of IAC Addition */

	  /* loop on the groups of the item */
	  while (seqSize > (posInSeq - 8L))
	  {
	    /* read the basic info on the new group */
	    seqGrNb   = Extract2Bytes (buff, bufPos);
	    elem_nb   = Extract2Bytes (buff, bufPos);
	    
	    /* test if it is the group length element */
	    if (elem_nb == 0x0000)
	    {
	      /* jump over : implicit : the length of the element (1 * 4 bytes) */
	      /*	     explicit : the VR and the length of the element (2 * 2 bytes) */
	      *bufPos += 4L;
	      seqGrSize = Extract4Bytes (buff, bufPos);
	      
	      /* the first elem must be taken into account ... */
	      seqGrSize += 12L;
	      
	      /* reset the buff pos to begining of the group */
	      *bufPos -= 12L;
	    } /* if ...elem = group length */
	    /* else, we have to compute the group length */
	    else
	    {
	      /* reset the buff pos to the begining of the group */
	      *bufPos -= 4L;
	      
	      /* get the current position of the file pointer */
	      err = Papy3FTell (Papy_file [fileNb], &currFilePos);
	      /* position the file pointer at the begining of the item */
	      err = Papy3FSeek (Papy_file [fileNb], SEEK_SET, (initialFilePos + (PapyLong) (*bufPos)));
	    
	      /* computes the group length */
	      seqGrSize = ComputeUndefinedGroupLength3 (fileNb, (PapyLong) seqSize);
	      
	      /* then reset the file pointer to its previous position */
	      err = Papy3FSeek (Papy_file [fileNb], SEEK_SET, currFilePos);
	      
	    } /* else ...compute the group length */
	    
	        
	    /* search the enum group number */
	    enumSeqNb = Papy3ToEnumGroup (seqGrNb);
	        
	    /* create the group */
	    seqGroup = Papy3GroupCreate (enumSeqNb);
	        
	    /* fill the group struct from the content of the buffer */
	    err = PutBufferInGroup3 (fileNb, buff, seqGroup, seqGrNb,
		  				   seqGrSize, bufPos, initialFilePos);
	    if (err < 0)
	    {
	      efree3 ((void **) &buff);
	      RETURN (err);
	    } /* if ...err */
		
	    posInSeq += seqGrSize + LENGTHLENGTH; /* add the grNb and elemNb and elemLength */	
		
	    /* creation of the object that will encapsulate the group */
	    object = (OBJECT *) emalloc3 ((PapyULong) sizeof (OBJECT));
	    object->whoAmI        = papGroup;
	    object->objID         = enumSeqNb;
	    object->group         = seqGroup;
	    object->item          = NULL;
	    object->module 	  = NULL;
	    object->tmpFileLength = 0L;
	    				     
	    /* add the object to the list of this element */
	    dsItem = InsertLastInList ((ITEM **) &seqItem, object);
	    
	    if (first)
	    {
	      first = FALSE;
	      elem->value->sq->object->item = dsItem;
	    } /* if ...first time we are in the loop */
	        
	  } /* while ...loop on the groups of the item */
	  
	  /* if it was an item with undefined length move the buffer further the delimiter */
	  if (isUndefItemL)
	  {
	    posInSeq += 8L;
	    *bufPos  += 8L;
	  } /* if */
	        
	} /* while ...loop on the items of the sequence */
	      
      } /* if ...not pointer or image sequence */
	    
      /* pointer or image sequence group 41 */
      else 
      {
	/* there is a value, but set to NULL */
	elem->nb_val = 1L;		/* CHG */
	elem->value  = NULL;
      } /* else ...pointer or image sequence */
      break;
          
    case AE :
    case AS :
    case CS :
    case DA :
    case DS :
    case DT :
    case IS :
    case LO :
    case LT :
    case PN :
    case SH :
    case ST :
    case TM :
    case UI :				/* all kind of strings */
      /*char_val = ExtractString (buff, bufPos, elemLength);*/
		  				   /* 1 for the string terminator */
      string = (char *) emalloc3 ((PapyULong) (elemLength + 1));
      p = string;
      tmp0 = buff;
      /* extract the element from the buffer */
      for (ii = 0L; ii < elemLength; ii++, (*bufPos)++) 
      {
        *(p++) = tmp0 [(int) *bufPos];
      }
    
      string [ii] = '\0';
    
      char_val = string;
 
      stringLength = strlen (char_val); 
          
      elem->nb_val = 1L;     /* number of strings */
      char_wrk = char_val;
          
      /* count the number of strings */
      for (j = 0; j < stringLength; j++, char_wrk ++)
      {
        /* value separator */
        if (*char_wrk == '\\') 
	{
	  elem->nb_val++;
	  *char_wrk = '\0';
	} /* if */
      } /* for ...counting the number of values */
          
      elem->value = (VALUE_T *) ecalloc3 ((PapyULong) elem->nb_val,
          			          (PapyULong) sizeof (VALUE_T));
          	
      /* extraction of the strings */	
      for (j = 0, char_wrk = char_val; j < elem->nb_val;
           j ++, char_wrk += stringLength + 1)
      {
	stringLength = strlen (char_wrk);
		    
	/* addition to delete the blank if odd string */
        if (elem->vr == UI)
        {
          /* suppress the blank by shifting all the chars to the left */
          /* old was : char_wrk [stringLength - 1] == '0') */
          if (char_wrk [stringLength - 1] == 0x00) 
	    char_wrk [stringLength - 1] = '\0';
        } /* then ...VR = UI */
	else
	  if (char_wrk [stringLength - 1] == ' ')
	    char_wrk [stringLength - 1] = '\0';
		    
	elem->value[j].a = PapyStrDup (char_wrk);

      } /* for ...extraction of the strings */
          
      efree3 ((void **) &string);
          
      break; /* strings */
          
  } /* switch ...value representation */
  
  return 0;
	
} /* endof PutBufferInElement3 */


										
/********************************************************************************/
/*									 	*/
/*	PutBufferInGroup3 : fill_in a group structure (all the elements) 	*/
/* 	from a buffer made of unsigned chars					*/
/* 	return : the enum group number if successfull				*/
/*		 standard error message otherwise 				*/
/*									  	*/
/********************************************************************************/

PapyShort
PutBufferInGroup3 (PapyShort fileNb, unsigned char *buff, ELEMENT *group,
		   PapyUShort papyGrNb, PapyULong bytesToRead, PapyULong *bufPos,
		   PapyLong initFilePos)
{
  ELEMENT		*arrElem;
  PapyULong	 	elemLength;
  PapyULong		j, initialBufPos;
  PapyULong		tmpULong, myULong = 0L;
  PapyLong		initialFilePos, currFilePos;
  PapyUShort	 	gr_nb;
  PapyUShort	 	elem_nb, elemLengthGr2;
  char			foo [3], *fooPtr;
  unsigned char		*charPtr; 
  int 			structPos, enum_gr_nb, i, isOld, isUndefSeqL = FALSE;
  int	 		shadow, enabledShadow [0x00FF], max_elem;
  PapyShort		err, creator;
  
  
  initialBufPos = *bufPos;
  initialFilePos= initFilePos;
  isOld         = TRUE; 
  enum_gr_nb    = Papy3ToEnumGroup (papyGrNb);   /* gr_nb papyrus -> enum */
  if (enum_gr_nb < 0)				 /* unknown group number */
  {
    efree3 ((void **) &buff);
    RETURN (papGroupNumber)
  } /* if */
  
  /* the number of elements of this group */
  max_elem = arrGroup [enum_gr_nb].size;
  


  if (papyGrNb >= 0x6000 && papyGrNb <= 0x6FFF)	/* overlay or UIN overlay */
  {      
    for (j = 0, arrElem = group; j < max_elem; j++, arrElem++)
      arrElem->group = papyGrNb;
    
  } /* if ...overlay or UINOverlay group */
  
  if (papyGrNb % 2 != 0) 			/* is it a shadow group ? */
  {
    shadow = TRUE; 
    
    /* disables all elements (initialisation) */
    for (i = 0; i < 0x00FF; i++) enabledShadow [i] = FALSE;
  } /* then */
  else shadow = FALSE;
    
  
  arrElem = group;
  
  while ((*bufPos - initialBufPos) < bytesToRead)	/* loop on the elements */
  {
    structPos = 0;			  /* pos in the array of elements */
    
    /* points to the right place in the buffer */
    charPtr  = buff;
    charPtr += *bufPos;
    /* extract the group number according to the little-endian syntax */
    gr_nb  = (PapyUShort) (*(charPtr + 1));
    gr_nb  = gr_nb << 8;
    gr_nb |= (PapyUShort) *charPtr;
    /* updates the current position in the read buffer */
    *bufPos += 2L;
    /* points to the right place in the buffer */
    charPtr += 2;
    
    /* extract the element according to the little-endian syntax */
    elem_nb  = (PapyUShort) (*(charPtr + 1));
    elem_nb  = elem_nb << 8;
    elem_nb |= (PapyUShort) *charPtr;
    /* updates the current position in the read buffer */
    *bufPos += 2L;
    /* points to the right place in the buffer */
    charPtr += 2;
    
    /* some special test for the group 2 are necessary */
    if (gr_nb == 0x0002)
    {
      /* test to discover which transfert syntax was used to create the file (implicit or explicit VR) */
      fooPtr     = (char *) &foo [0];
      fooPtr [0] = (char)   *charPtr;
      fooPtr [1] = (char) (*(charPtr + 1));
      fooPtr [2] = '\0';
      
      /* if the VR is unknown assume the group 2 is using implicit VR */
      if (!(strcmp (fooPtr, "AE") == 0 || strcmp (fooPtr, "AS") == 0 || strcmp (fooPtr, "AT") == 0 ||
            strcmp (fooPtr, "CS") == 0 || strcmp (fooPtr, "DA") == 0 || strcmp (fooPtr, "DS") == 0 ||
            strcmp (fooPtr, "DT") == 0 || strcmp (fooPtr, "FL") == 0 || strcmp (fooPtr, "FD") == 0 ||
            strcmp (fooPtr, "IS") == 0 || strcmp (fooPtr, "LO") == 0 || strcmp (fooPtr, "LT") == 0 ||
            strcmp (fooPtr, "OW") == 0 || strcmp (fooPtr, "PN") == 0 || strcmp (fooPtr, "SH") == 0 ||
            strcmp (fooPtr, "SL") == 0 || strcmp (fooPtr, "SQ") == 0 || strcmp (fooPtr, "SS") == 0 ||
            strcmp (fooPtr, "ST") == 0 || strcmp (fooPtr, "TM") == 0 || strcmp (fooPtr, "UI") == 0 || 
            strcmp (fooPtr, "UL") == 0 || strcmp (fooPtr, "US") == 0 || strcmp (fooPtr, "OB") == 0))
        arrTransfSyntax [fileNb] = LITTLE_ENDIAN_IMPL;
      
      /* if there are OB values in group 2 it is a recent version of the toolkit ( >  3.3) */
      /* the isOld variable will be used later in the code of this routine */
      if (strcmp (fooPtr, "OB") == 0) isOld = FALSE;
    } /* if ...group 2 */
    
    
    
    /* test wether the transfert syntax is the little-endian explicit VR one */
    if (arrTransfSyntax [fileNb] == LITTLE_ENDIAN_EXPL)
    {
      /* extract the VR */
      fooPtr     = (char *) &foo [0];
      fooPtr [0] = (char)   *charPtr;
      fooPtr [1] = (char) (*(charPtr + 1));
      fooPtr [2] = '\0';
      /* updates the current position in the read buffer */
      *bufPos += 2L;
      /* points to the right place in the buffer */
      charPtr += 2;
      
      /* extract the element length depending on the extracted VR */
      if (strcmp (fooPtr, "OB") == 0 || strcmp (fooPtr, "OW") == 0 || strcmp (fooPtr, "SQ") == 0)
      {
        /* updates the current position in the read buffer by jumping over the 2 bytes set to 0 */
        *bufPos += 2L;
        /* points to the right place in the buffer */
        charPtr += 2;
        
        /* extract the element length according to the little-endian explicit VR syntax */
        tmpULong      = (PapyULong) (*(charPtr + 3));
        tmpULong      = tmpULong << 24;
        myULong	      = tmpULong;
        tmpULong      = (PapyULong) (*(charPtr + 2));
        tmpULong      = tmpULong << 16;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) (*(charPtr + 1));
        tmpULong      = tmpULong << 8;
        myULong	     |= tmpULong;
        tmpULong      = (PapyULong) *charPtr;
        myULong      |= tmpULong;
        
        elemLength    = myULong;
        
        /* updates the current position in the read buffer */
        *bufPos += 4L;
      } /* if ...VR = OB, OW or SQ */
      else
      {
        /* extract the element length according to the little-endian explicit VR syntax */
        elemLengthGr2  = (PapyUShort) (*(charPtr + 1));
        elemLengthGr2  = elemLengthGr2 << 8;
        elemLengthGr2 |= (PapyUShort) *charPtr;
      
        elemLength     = (PapyULong) elemLengthGr2;
        
        /* updates the current position in the read buffer */
        *bufPos += 2L;
      } /* else ...other VRs */
            
    } /* if ...transfert syntax is little_endian explicit VR */
    /* little_endian implicit VR */
    else
    {
      /* extract the element length according to the little-endian implicit VR syntax */
      tmpULong      = (PapyULong) (*(charPtr + 3));
      tmpULong      = tmpULong << 24;
      myULong	    = tmpULong;
      tmpULong      = (PapyULong) (*(charPtr + 2));
      tmpULong      = tmpULong << 16;
      myULong	   |= tmpULong;
      tmpULong      = (PapyULong) (*(charPtr + 1));
      tmpULong      = tmpULong << 8;
      myULong	   |= tmpULong;
      tmpULong      = (PapyULong) *charPtr;
      myULong 	   |= tmpULong;
      elemLength    = myULong;
    
      /* updates the current position in the read buffer */
      *bufPos += 4L;
    } /* else ...little_endian implicit VR */
    
    
    /* it could be an undefined length, i.e. VR = SQ */
    if (elemLength == 0xFFFFFFFF)
    {
      /* for futur move of the buffer pointer */
      isUndefSeqL = TRUE;
      
      elemLength = 0L;
      if (!(gr_nb == 0x7FE0 && elem_nb == 0x0010))
      {
        /* get the current file position */
        err = Papy3FTell (Papy_file [fileNb], &currFilePos);
        /* position the file pointer to point to the item */
        err = Papy3FSeek (Papy_file [fileNb], SEEK_SET, initialFilePos + (PapyLong) (*bufPos));
        
        if ((err = ComputeUndefinedSequenceLength3 (fileNb, &elemLength)) < 0)
          RETURN (err);
      
        /* reset the file pointer to its previous position */
        err = Papy3FSeek (Papy_file [fileNb], SEEK_SET, currFilePos);
      
      } /* if ...not image pixel */
      else /* just decide it is the last readable thing */
        elemLength = bytesToRead - (*bufPos -initialBufPos);
    
    } /* if ...undefined length */
    
    /* odd element length are forbidden */
    if (elemLength % 2 != 0)
      RETURN (papLengthIsNotEven);

    /* it is a shadow group, so we are looking dynamically for our element range */
    if (shadow && elem_nb >= 0x0010 && elem_nb <= 0x00FF)
    {
      creator = Papy3CheckValidOwnerId (fileNb, buff, bufPos, elem_nb, elemLength, group);
      
      /* look for the position in the enum of the group */
      if (creator)
      {
        while (structPos <= max_elem && elem_nb != arrElem [structPos].element)
          structPos++;
        enabledShadow [arrElem [structPos].element] = TRUE;
      } /* if */
    	    
    } /* if */

    else
    {
      if (!shadow ||
          (shadow && elem_nb <  0x0010) ||
    	  (shadow && elem_nb >= 0x1000 && enabledShadow [elem_nb >> 8]))
      {    
        /* search the element in the array */
        while (structPos <= max_elem && elem_nb != arrElem [structPos].element)
          structPos++;
      
        /* element number out of range */
        if (structPos >= max_elem)
        {
          /* it could be an unknown element (who knows with DICOM ?) */
          /* so we just skip the element */
	  *bufPos += elemLength;
        } /* if */
      
        else 
        {
          arrElem [structPos].length = elemLength;
        
          /* there has been a change in the dictionary. */
          /* This helps the new version to read the old files ( < 3.3) */
          if (isOld && gr_nb == 0x0002 && (elem_nb == 0x0001 || elem_nb == 0x0102))
            arrElem [structPos].vr = USS;
 
      
          /* extract the value of the element from the buffer */      
          if (elemLength > 0 && 
      	      !(arrElem [structPos].group   == 0x7FE0 && 
      	        arrElem [structPos].element == 0x0010))
          {
            /* extract the element depending on the value representation */
	    if ((err = PutBufferInElement3 (fileNb, buff, elemLength, 
				            &arrElem [structPos], bufPos, initialFilePos)) < 0)
              RETURN (err);  
            
            /* if it was a sequence with an undefined length, move the buffer accordingly */
            if (isUndefSeqL) 
              *bufPos += 8L;
            
          } /* if ...elemLength > 0 */
        } /* else ...element found */
	
      } /* if 				...we can read this element */
	
      else 		        /* we dont know how to read this element... */
        *bufPos += elemLength;	 			/* ...so we skip it */
    } /* else ...not creator of a private data element */
    
  } /* while ... loop on the elements */
  
  return (PapyShort) enum_gr_nb;

} /* endof PutBufferInGroup3 */


										
/********************************************************************************/
/*									 	*/
/*	Papy3GroupRead : read the group from the file in a buffer then fill_in 	*/
/*	the group structure (all the elements) from the buffer.			*/
/* 	return : the group number (in the enum_type) if able to fill it		*/
/*		 standard error message otherwise 				*/
/*									  	*/
/********************************************************************************/

PapyShort CALLINGCONV
Papy3GroupRead (PapyShort fileNb, ELEMENT **group)
{

  unsigned char 	*buff;
  PapyULong		bytesToRead, grLength, bufPos;
  PapyLong		initFilePos;
  PapyUShort	 	papyGrNb;
  PapyShort		err;
  int 			enum_gr_nb;
  enum TRANSF_SYNTAX	prevSyntax; 
  
  
  /* get the position in the file for any eventual unknown seq length */
  err = Papy3FTell (Papy_file [fileNb], &initFilePos);
  
  bufPos = 0L;
  /* read the buffer from the file */
  if (read_group3 (fileNb, &papyGrNb, &buff, &bytesToRead, &grLength) < 0)
  {
    efree3 ((void **) &buff);
    RETURN (papReadGroup)
  } /* if */
  
  /* makes sure we keep the right syntax and set the default syntax instead */
  if (papyGrNb == 0x0002)
  {
    prevSyntax = arrTransfSyntax [fileNb];
    arrTransfSyntax [fileNb] = LITTLE_ENDIAN_EXPL;
  } /* if */
    
  enum_gr_nb = Papy3ToEnumGroup (papyGrNb);     /* gr_nb papyrus -> enum */
  if (enum_gr_nb < 0)				 /* unknown group number */
  {
    efree3 ((void **) &buff);
    RETURN (papGroupNumber)
  } /* if */
 
  /* allocates the structure of the given group */
  *group = Papy3GroupCreate (enum_gr_nb);
  
  /* if the group do not have the group length element, fill it ... */
  if (grLength != 0)
  {
    (*group)->nb_val    = 1L;
    (*group)->value     = (VALUE_T *) emalloc3 ((PapyULong) sizeof (VALUE_T));
    (*group)->value->ul = grLength;
  } /* if ...undefined group length */
  
  /* extract the elements of the buffer and put them in the group structure */
  bufPos = 0L;
  err = PutBufferInGroup3 (fileNb, buff, *group, papyGrNb, bytesToRead, &bufPos, initFilePos);
  if (err < 0)
  {
    efree3 ((void **) &buff);
    RETURN (err);
  } /* if */

  /* frees the read buffer */
  efree3 ((void **) &buff);
  
  /* restore any previous transfert syntax */
  if (papyGrNb == 0x0002)
    arrTransfSyntax [fileNb] = prevSyntax;
  
  RETURN ((PapyShort) papyGrNb);
  
} /* endof Papy3GroupRead */
