/*
   Write UCD file

   This module takes triangle cell sets from the input
   field structure, and writes them out to a file in
   AVS UCD (Unstructured Cell Data) ASCII file format.

   Inputs:
	  filename - string for file to write
	  in: field with on triangle cell_set and two node_data.
		  primary design goal is isosurface module output

   Outputs:
	  err: integer indicating read error (0 = no error, 1 = error)
	  err_str: string indicating error condition, or "" for no error.

	  ASCII file written.

   The V user interface macro maps this to a label area
   where visibility is controlled with the err int.

   Author: I. Curington 28 January, 2000
   Advanced Visual Systems, Inc.
   ianc@avs.com



  Conditions:
	 INTERNATIONAL AVS CENTRE - WARRANTY DISCLAIMER
	 Please read the file DISCLAIMER for conditions associated with this file.
	 avs@iavsc.org, www.iavsc.org

   This module is implemented in C++.

Ian Curington (ianc@avs.com)
   Updated 20 July 99, derived from WriteMAYA module
   Updated 22 July 99, additional cell sets, group per set
   Updated  3 Aug 99, added degenerate triangle cull
   Updated 28 Jan 00, extended format from Maya to UCD

Andrew Dodd (andrew.dodd@mcc.ac.uk)
   Updated	 7 Sept 00, corrected output of polytri and polyline data
					   node_connect_list is now used for poly data
					   revised node_data output loop
   Updated	15 Sept 00, changed node data so that it can output an
						aribitary number of components and vector lengths.
   Updated	18 Sept 00, fixed bug where comments could be written into the middle of files.

Sudhir Sangappa (sudhir@avs.com)
   Updated	April 25 2001, Implemented the functionality to  handle cell data if present
   Updated	May   02 2001, Fixed handling cell sets of same cell_name
   Updated	May   30 2001, Implemented binary file format.

*/

#include <stdio.h>
#include <stdlib.h>
#include <avs/port.h>
#include <avs/math.h>
#include <avs/f_utils.h>
#include <avs/gd_def.h>
#include "iac_proj/wr_ucd/gen.hxx"

#define CELL_TRI       4
#define CELL_QUAD      5
#define CELL_POLYTRI  10
#define CELL_POLYLINE  3
#define CELL_LINE      2
#define CELL_TET       6
#define CELL_HEX       7
#define CELL_PRISM     8
#define CELL_PYR       9

#define LINES_PER_STATUS 500

/*
 * uncomment the following line to enable the non-functional
 * polyline and polytri code
 * #define OUTPUT_POLY_TEST
 */

/*
 * uncomment the following lines for debug messages
 * #define DEBUG
 * #define DEBUG_BINARY
 */


#ifdef OUTPUT_POLY_TEST
static int check_degen(int i, float *in_coords, int nspace);
#endif


   /***********************/
   /* OM Entry Point	  */
   /***********************/
int WriteUCD_WriteUCDMods_WriteUCDCore::update(OMevent_mask event_mask, int seq_num) {
   // filename (OMXstr read req notify)
   // in (Field read)

   FILE  *outfile;
   unsigned long int npolys;
   int	 i, j, k, id;
   float *in_coords;
   int	 interrupt;
   float raw[20];
   int	 nnodes;
   int	 nspace;
   int	 cs, cell_type[100];
   int	 veclen;
   int   nnode_data,         /* number of node data components */
         total_nnode_data;   /* total node data comps (including veclens) */
   int   ncell_data,         /* number of node data components */
         total_ncell_data;   /* total cell data comps (including veclens) */
   int	 *in_ncl;
   float *xfm, *xlate;
   int	 size;
   char  topname[80];
   int	 nf;
   int	 gcount;
   int	 gncells, gcell;
   int   UCD_BINARY = binary;
   int   num_model_data = 0;
   int   cell_id = 0;

   typedef struct Info {
	   int id, mat, n, cell_type;
   } Cell_Info;

   Cell_Info cell_info ;

   /***********************/
   /* Function's Body	  */
   /***********************/

#ifdef DEBUG
   ERRverror("", ERR_PRINT|ERR_NO_HEADER, "starting write ucd");
#endif
   strcpy(topname,top_name); // from input parameter block


   if (UCD_BINARY) {
      // *********************
      // WRITE AVS UCD BINARY
      // *********************

      // Open the file
      if( (outfile = (FILE *)FILEfopen(filename, SIO_W_BIN)) == (FILE *)NULL ) {
         err = 1;
	 err_str = "write_ucd: could not open file for ucd BINARY write";
	 return(1);
      }

      // Write the magic number at start of outfile.
      char c = 7;
      fwrite(&c, sizeof(char), 1, outfile);
   }

   else {
      // *********************
      // WRITE AVS UCD HEADER
      // *********************
      if( (outfile = (FILE *)FILEfopen(filename, SIO_W_TXT)) == (FILE *)NULL ) {
	 err = 1;
         err_str = "write_ucd: could not open file for ucd write";
	 return(1);
      }
      //
      // write header
      //
      fprintf(outfile,"# AVS UCD File\n");
      fprintf(outfile,"# Generated by AVS/Express write_ucd: http://www.avs.com\n");
      fprintf(outfile,"# OBJECT name=\"%s\"\n", topname);
      fprintf(outfile,"# num_nodes num_cells num_node_data num_cell_data num_model_data\n");
   }


   /*
    *  Loop over all input fields
    */

   gcount = 0; // keep track of global node count in output stream

   for (nf=0; nf < nobjs; nf++) {
#ifdef DEBUG
      ERRverror("", ERR_PRINT|ERR_NO_HEADER, "### starting field ARR loop");
#endif

      interrupt = 0;
      nnodes = in[nf].nnodes;
      nspace = in[nf].nspace;

#ifdef DEBUG
      ERRverror("", ERR_PRINT|ERR_NO_HEADER, "### got nnodes and nspace");
#endif

      // check we have found some facets
      if (nnodes == 0 ) {
	 if (!UCD_BINARY) {
	    fprintf(outfile, "# end of object\n");
	    fprintf(outfile, "# End of File\n");
	 }
	 fclose(outfile);
	 err = 0;
	 return(1);
      }

#ifdef DEBUG
      ERRverror("",ERR_PRINT|ERR_NO_HEADER,"### nnodes not zero");
#endif

      // out.coordinates.values
      // get coords array
      in_coords = (float*)in[nf].coordinates.values.ret_array_ptr(OM_GET_ARRAY_RD);
      if (in_coords == NULL) {
	 err = 1;
	 err_str = "write ucd: problem assigning coordinates array";
	 if (!UCD_BINARY) fprintf(outfile, " *** FILE WRITE STOPPED EARLY, COORD ACCESS ERROR! ***\n");
	 fclose(outfile);
	 return(1);
      }

#ifdef DEBUG
      ERRverror("", ERR_PRINT|ERR_NO_HEADER, "alloced coords");
#endif

      // check if node data is present.
      nnode_data = in[nf].nnode_data;
      total_nnode_data = 0;

      // count up total number of components, including veclens.
      if (nnode_data > 0) {
	 for (i=0; i<nnode_data; i++) {
	    veclen = in[nf].node_data[i].veclen;
	    if (veclen > 0) {
	       total_nnode_data += veclen;
	    }
	 }
      }

      // get the transformation for the field
      xfm   = (float*)in[nf].xform.mat.ret_array_ptr(OM_GET_ARRAY_RD);
      xlate = (float*)in[nf].xform.xlate.ret_array_ptr(OM_GET_ARRAY_RD);

      // supports only cell_sets of specific types,
      // so scan structure for exact matches.
      // after that, we can simply check the cell_type.

      cs = -1;
      gncells = 0;

      for (i=0; i< in[nf].ncell_sets; i++){
	 // npolys = in[nf].cell_set[i].ncells;

	 // check for cell_set of type "Tri",
	 if (in[nf].cell_set[i].ncells > 0       &&
	     in[nf].cell_set[i].cell_ndim   == 2 &&
	     in[nf].cell_set[i].cell_nnodes == 3 &&
	     in[nf].cell_set[i].cell_order  == 1 &&
	     in[nf].cell_set[i].poly_flag   == 0 ) {
	    cell_type[i] = CELL_TRI;
	    cs = i;
	    gncells += in[nf].cell_set[i].ncells;
	 }
	 // check for cell_set of type "Polytri",
	 else if ( in[nf].cell_set[i].ncells > 0       &&
		   in[nf].cell_set[i].cell_ndim	  == 2 &&
		   in[nf].cell_set[i].cell_nnodes == 3 &&
		   in[nf].cell_set[i].cell_order  == 1 &&
		   in[nf].cell_set[i].poly_flag	  == 1 ) {
	    cell_type[i] = CELL_POLYTRI;
	    cs = i;

#ifndef OUTPUT_POLY_TEST
	    gncells += in[nf].cell_set[i].ncells;
#else
	    for (int p = 0; p < (int)npolys; p++){
	       for (i=in_ncl[p*2]; i<=in_ncl[p*2+1]; i++){
		  gncells++;
	       }
	    }
#endif

	 }
	 // check for cell_set of type "Polyline",
	 else if ( in[nf].cell_set[i].ncells > 0       &&
		   in[nf].cell_set[i].cell_ndim	  == 1 &&
		   in[nf].cell_set[i].cell_nnodes == 2 &&
		   in[nf].cell_set[i].cell_order  == 1 &&
		   in[nf].cell_set[i].poly_flag	  == 1 ) {
	    cell_type[i] = CELL_POLYLINE;
	    cs = i;

#ifndef OUTPUT_POLY_TEST
	    gncells += in[nf].cell_set[i].ncells;
#else
	    for (int p = 0; p < (int)npolys; p++){
	       for (i=in_ncl[p*2]; i<=in_ncl[p*2+1]; i++){
		  gncells++;
	       }
	    }
#endif
	 }
	 // check for cell_set of type "Quad",
	 else if ( in[nf].cell_set[i].ncells > 0       &&
		   in[nf].cell_set[i].cell_ndim	 == 2  &&
		   in[nf].cell_set[i].cell_nnodes == 4 &&
		   in[nf].cell_set[i].cell_order  == 1 &&
		   in[nf].cell_set[i].poly_flag	 == 0 ) {
	   cell_type[i] = CELL_QUAD;
	   cs = i;
	   gncells += in[nf].cell_set[i].ncells;
	 }
	 // check for cell_set of type "Line",
	 else if ( in[nf].cell_set[i].ncells > 0       &&
		   in[nf].cell_set[i].cell_ndim	 == 1  &&
		   in[nf].cell_set[i].cell_nnodes == 2 &&
		   in[nf].cell_set[i].cell_order  == 1 &&
		   in[nf].cell_set[i].poly_flag	 == 0 ) {
	    cell_type[i] = CELL_LINE;
	    cs = i;
	    gncells += in[nf].cell_set[i].ncells;
	 }
	 // check for cell_set of type "Tet",
	 else if ( in[nf].cell_set[i].ncells > 0        &&
		   in[nf].cell_set[i].cell_ndim	  == 3  &&
		   in[nf].cell_set[i].cell_nnodes == 4  &&
		   in[nf].cell_set[i].cell_order  == 1  &&
		   in[nf].cell_set[i].poly_flag	  == 0 ) {
	    cell_type[i] = CELL_TET;
	    cs = i;
	    gncells += in[nf].cell_set[i].ncells;
	 }
	 // check for cell_set of type "Hex",
	 else if ( in[nf].cell_set[i].ncells > 0        &&
		   in[nf].cell_set[i].cell_ndim	  == 3  &&
		   in[nf].cell_set[i].cell_nnodes == 8  &&
		   in[nf].cell_set[i].cell_order  == 1  &&
		   in[nf].cell_set[i].poly_flag	  == 0 ) {
	    cell_type[i] = CELL_HEX;
	    cs = i;
	    gncells += in[nf].cell_set[i].ncells;
	 }
	 // check for cell_set of type "Prism",
	 else if ( in[nf].cell_set[i].ncells > 0        &&
		   in[nf].cell_set[i].cell_ndim	  == 3  &&
		   in[nf].cell_set[i].cell_nnodes == 6  &&
		   in[nf].cell_set[i].cell_order  == 1  &&
		   in[nf].cell_set[i].poly_flag	  == 0 ) {
	    cell_type[i] = CELL_PRISM;
	    cs = i;
	    gncells += in[nf].cell_set[i].ncells;
	 }
	 // check for cell_set of type "Pyramid",
	 else if ( in[nf].cell_set[i].ncells > 0        &&
		   in[nf].cell_set[i].cell_ndim	  == 3  &&
		   in[nf].cell_set[i].cell_nnodes == 5  &&
		   in[nf].cell_set[i].cell_order  == 1  &&
		   in[nf].cell_set[i].poly_flag	  == 0 ) {
	    cell_type[i] = CELL_PYR;
	    cs = i;
	    gncells += in[nf].cell_set[i].ncells;
	 }
	 else {
	    cell_type[i] = -1;
	    // skip to next, no type recognized
	 }
      }
      if ( cs == -1 ) {
	 err = 1;
	 err_str = "write ucd: no cell sets of the right type";
	 if (!UCD_BINARY)
	    fprintf(outfile, " *** FILE WRITE STOPPED EARLY! ***\n");
	 else
	    fprintf(stderr, " *** FILE WRITE STOPPED EARLY! ***\n");
	 fclose(outfile);
	 return(1);
      }


#ifdef DEBUG
      ERRverror("", ERR_PRINT|ERR_NO_HEADER, "established valid cell set");
#endif

      // now that we have the data structure set up,
      // write the file contents

      // header line with counts
      //   nnodes, ncells, node_comp, cell_comp, 0

      OMobj_id ncell_data_id;
      ncell_data_id = OMfind_subobj(in[nf].cell_set[0].obj_id(),
				    OMstr_to_name("ncell_data"), OM_OBJ_RW);

      total_ncell_data = 0;
      if( !OMis_null_obj(ncell_data_id) ) {
         // I think we can assume all cell sets have the same kind
         // of cell data (if cell data is present).
         ncell_data = in[nf].cell_set[0].ncell_data;
         // Count up total number of cell data components including
         // the veclens.
	 for(i=0; i < ncell_data; i++ ) {
            veclen = in[nf].cell_set[0].cell_data[i].veclen;
            total_ncell_data += veclen;
         }
      }
      else ncell_data = 0;

      if(!UCD_BINARY)
         fprintf(outfile, "%d %d %d %d %d\n",
                 nnodes,             /* number nodes */
                 gncells,            /* number cells */
                 total_nnode_data,   /* number of node data */
                 total_ncell_data,   /* number cell data */
                 num_model_data);    /* number of model data */
      else {
         fwrite(&nnodes, sizeof(int), 1, outfile);
         fwrite(&gncells, sizeof(int), 1, outfile);
         fwrite(&total_nnode_data, sizeof(int), 1, outfile);
         fwrite(&total_ncell_data, sizeof(int), 1, outfile);
         fwrite(&num_model_data, sizeof(int), 1, outfile);

#ifdef DEBUG_BINARY
         fprintf(stderr, "nodes:%d, cells:%d %d %d %d", nnodes, gncells,
                 total_node_data, total_ncell_data, num_model_data);
#endif
      };


      // Calculate num_nlist_nodes ( UCD binary number )
      // of nlist nodes (for cell topology)
      int num_nlist_nodes = 0;
      if (UCD_BINARY) {
	 for(i=0; i < in[nf].ncell_sets; i++ ) {
	    num_nlist_nodes += in[nf].cell_set[i].ncells * in[nf].cell_set[i].cell_nnodes;
	 }

	 // Write num_nlist_nodes
#ifdef DEBUG_BINARY
	 fprintf(stderr, " %d\n", num_nlist_nodes);
#endif
	 fwrite(&num_nlist_nodes, sizeof(int), 1, outfile);
      }
      else {
	 // ***********************
	 //  WRITE AVS UCD CONTENTS
	 // ***********************

	 //  loop over vertices
	 for (i = 0; i < nnodes; i++) {
	    raw[0] = in_coords[i*nspace+0];
	    raw[1] = in_coords[i*nspace+1];
	    if (nspace == 3)
	       raw[2] = in_coords[i*nspace+2];
	    else
	       raw[2] = 0.0;
	    MATvec3_mat4_multiply(raw, (Matr4 *)xfm);
	    VEC_ADD(raw, raw, xlate);

	    fprintf(outfile,"%d %f %f %f\n",
		    i+1, raw[0], raw[1], raw[2]);
	 }
      }

      // Allocate memory for cell topology lists (only for UCD_BINARY)
      int *cell_topology_lists = NULL;
      int ctl_id = 0;
      if (UCD_BINARY) {
	 cell_topology_lists = (int *) malloc (sizeof(int) * num_nlist_nodes );
	 if (cell_topology_lists == NULL) {
	    err = 1;
	    err_str = "Could not allocate memory for cell topology lists";
	    fclose(outfile);
	    return(1);
	 }
      }

      gcell = 1;
      // loop over all cell sets, processing for output
      for (cs=0; cs < in[nf].ncell_sets; cs++){
	 npolys = in[nf].cell_set[cs].ncells;

	 // in.cell_set[cs].node_connect_list
	 // get node connect list

#ifndef OUTPUT_POLY_TEST
	 in_ncl = (int*)in[nf].cell_set[cs].node_connect_list.ret_array_ptr(OM_GET_ARRAY_RD);
#else
	 if (cell_type[cs]==CELL_POLYTRI || cell_type[cs]==CELL_POLYLINE)
	    in_ncl = (int*)in[nf].cell_set[cs].poly_connect_list.ret_array_ptr(OM_GET_ARRAY_RD);
	 else
	    in_ncl = (int*)in[nf].cell_set[cs].node_connect_list.ret_array_ptr(OM_GET_ARRAY_RD);
#endif

	 if (in_ncl == NULL) {
	    if (in_coords) ARRfree(in_coords);
	    ARRfree(xfm);
	    ARRfree(xlate);
	    fclose(outfile);
	    err = 1;
	    err_str = "write_ucd: problem allocating node connect list";
	    return(1);
	 }

#ifdef DEBUG
	 ERRverror("", ERR_PRINT|ERR_NO_HEADER,"number of polys = %d",npolys);
#endif

	 int cell_group;

	 // Find material id for each cell (Required for only UCD_BINARY)
	 int mat;
	 float *props = NULL;
	 int size, returned_val;
	 OMobj_id cell_props_id;

	 cell_props_id = OMfind_subobj(in[nf].cell_set[cs].obj_id(),
				       OMstr_to_name("nprops"), OM_OBJ_RD);

	 if (OMget_int_val(cell_props_id, &returned_val) == 1) {
	    FLDget_cell_props(in[nf].cell_set[cs].obj_id(),
			      &props, &size, OM_GET_ARRAY_RD);
	    mat = (int)props[0];
	 }
	 else mat = 0;

	 // Cell group for ASCII is same as material for binary
	 cell_group = mat;

	 // process each cell type in turn:
	 switch (cell_type[cs]) {
	 case CELL_LINE:
	 case CELL_TRI:
	 case CELL_QUAD:
	 case CELL_TET:
	 case CELL_HEX:
	 case CELL_PYR:
	 case CELL_PRISM:
#ifndef OUTPUT_POLY_TEST
	 case CELL_POLYLINE:
	 case CELL_POLYTRI:
#endif
	    // loop over geometric primitives within this cell set
	    for (i = 0; i < (int)npolys; i++) {

#ifndef OUTPUT_POLY_TEST
	       if ((cell_type[cs]==CELL_TRI) || (cell_type[cs]==CELL_POLYTRI))
#else
	       if (cell_type[cs]==CELL_TRI)
#endif
		  if (UCD_BINARY) {
		     cell_info.id = cell_id++;
		     cell_info.mat = mat;
		     cell_info.n = 3;
		     cell_info.cell_type = 2;
		     fwrite(&cell_info, sizeof(Cell_Info), 1, outfile);
#ifdef DEBUG_BINARY
		     fprintf(stderr, "%d %d %d %d\n", cell_info.id, cell_info.mat, cell_info.n, cell_info.cell_type);
#endif
		     cell_topology_lists[ctl_id++] = in_ncl[i*3]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*3+1]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*3+2]+1;
		  }
		  else {
		     fprintf(outfile, "%d %d tri %d %d %d\n",
			     gcell++, cell_group,
			     (in_ncl[i*3]+1),
			     (in_ncl[i*3+1]+1),
			     (in_ncl[i*3+2]+1));
		  }

	       else if (cell_type[cs]==CELL_QUAD)
		  if (UCD_BINARY) {
		     cell_info.id = cell_id++;
		     cell_info.mat = mat;
		     cell_info.n = 4;
		     cell_info.cell_type = 3;
		     fwrite(&cell_info, sizeof(Cell_Info), 1, outfile);
#ifdef DEBUG_BINARY
		     fprintf(stderr, "%d %d %d %d\n",
			     cell_info.id, cell_info.mat,
			     cell_info.n, cell_info.cell_type);
#endif
		     cell_topology_lists[ctl_id++] = in_ncl[i*4]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*4+1]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*4+2]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*4+3]+1;
		  }
		  else {
		     fprintf(outfile, "%d %d quad %d %d %d %d\n",
			     gcell++, cell_group,
			     (in_ncl[i*4]+1),
			     (in_ncl[i*4+1]+1),
			     (in_ncl[i*4+2]+1),
			     (in_ncl[i*4+3]+1));
		  }
#ifndef OUTPUT_POLY_TEST
	       else if ((cell_type[cs]==CELL_LINE) || (cell_type[cs]==CELL_POLYLINE))
#else
	       else if (cell_type[cs]==CELL_LINE)
#endif
		  if (UCD_BINARY) {
		     cell_info.id = cell_id++;
		     cell_info.mat = 0;
		     cell_info.n = 2;
		     cell_info.cell_type = 1;
		     fwrite(&cell_info, sizeof(Cell_Info), 1, outfile);
		     cell_topology_lists[ctl_id++] = in_ncl[i*2]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*2+1]+1;
		  }
		  else {
		     fprintf(outfile, "%d %d line %d %d\n",
			     gcell++, cell_group,
			     (in_ncl[i*2]+1),
			     (in_ncl[i*2+1]+1));
		  }

	       else if (cell_type[cs]==CELL_TET)
		  if (UCD_BINARY) {
		     cell_info.id = cell_id++;
		     cell_info.mat = mat;
		     cell_info.n = 4;
		     cell_info.cell_type = 4;
		     fwrite(&cell_info, sizeof(Cell_Info), 1, outfile);
		     cell_topology_lists[ctl_id++] = in_ncl[i*4]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*4+1]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*4+2]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*4+3]+1;
		  }
		  else {
		     fprintf(outfile, "%d %d tet %d %d %d %d\n",
			     gcell++, cell_group,
			     (in_ncl[i*4]+1),
			     (in_ncl[i*4+1]+1),
			     (in_ncl[i*4+2]+1),
			     (in_ncl[i*4+3]+1));
		  }

	       else if (cell_type[cs]==CELL_HEX)
		  if (UCD_BINARY) {
		     cell_info.id = cell_id++;
		     cell_info.mat = mat;
		     cell_info.n = 8;
		     cell_info.cell_type = 7;
		     fwrite(&cell_info, sizeof(Cell_Info), 1, outfile);

		     cell_topology_lists[ctl_id++] = in_ncl[i*8]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*8+1]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*8+2]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*8+3]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*8+4]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*8+5]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*8+6]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*8+7]+1;
		  }
		  else {
		     fprintf(outfile, "%d %d hex %d %d %d %d %d %d %d %d\n",
			     gcell++, cell_group,
			     (in_ncl[i*8]+1),
			     (in_ncl[i*8+1]+1),
			     (in_ncl[i*8+2]+1),
			     (in_ncl[i*8+3]+1),
			     (in_ncl[i*8+4]+1),
			     (in_ncl[i*8+5]+1),
			     (in_ncl[i*8+6]+1),
			     (in_ncl[i*8+7]+1));
		  }
	       else if (cell_type[cs]==CELL_PRISM)
		  if (UCD_BINARY) {
		     cell_info.id = cell_id++;
		     cell_info.mat = mat;
		     cell_info.n = 6;
		     cell_info.cell_type = 6;
		     fwrite(&cell_info, sizeof(Cell_Info), 1, outfile);

		     cell_topology_lists[ctl_id++] = in_ncl[i*6]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*6+1]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*6+2]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*6+3]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*6+4]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*6+5]+1;
		 }
		 else {
		    fprintf(outfile, "%d %d prism %d %d %d %d %d %d\n",
			    gcell++, cell_group,
			    (in_ncl[i*6]+1),
			    (in_ncl[i*6+1]+1),
			    (in_ncl[i*6+2]+1),
			    (in_ncl[i*6+3]+1),
			    (in_ncl[i*6+4]+1),
			    (in_ncl[i*6+5]+1));
		 }
	       else if (cell_type[cs]==CELL_PYR)
		  if (UCD_BINARY) {
		     cell_info.id = cell_id++;
		     cell_info.mat = mat;
		     cell_info.n = 5;
		     cell_info.cell_type = 5;
		     fwrite(&cell_info, sizeof(Cell_Info), 1, outfile);
		     cell_topology_lists[ctl_id++] = in_ncl[i*5]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*5+1]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*5+2]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*5+3]+1;
		     cell_topology_lists[ctl_id++] = in_ncl[i*5+4]+1;
		  }
		  else {
		     fprintf(outfile, "%d %d pyr %d %d %d %d %d\n",
			     gcell++, cell_group,
			     (in_ncl[i*5]+1),
			     (in_ncl[i*5+1]+1),
			     (in_ncl[i*5+2]+1),
			     (in_ncl[i*5+3]+1),
			     (in_ncl[i*5+4]+1));
		 }

	       // status and interrupt check
	       if ( (i%LINES_PER_STATUS)==0) {
		  OMstatus_check((int)(100*i/(float)npolys),
				 "write ucd", &interrupt);
		  if (interrupt) {
#ifdef DEBUG
		     ERRverror("", ERR_PRINT|ERR_NO_HEADER, 
                               "got an interrupt, stopping");
#endif
		     ARRfree(in_coords);
		     ARRfree(in_ncl);
		     ARRfree(xfm);
		     ARRfree(xlate);
		     fprintf(outfile," *** FILE WRITE INTERRUPTED! ***\n");
		     fclose(outfile);
		     return(1);
		  }
	       }
	    } // end of primitive loop
	    break;

#ifdef OUTPUT_POLY_TEST
	 case CELL_POLYTRI:
	 case CELL_POLYLINE:
         {
	    npolys = in[nf].cell_set[cs].npolys;
	    // loop over geometric primitives within this cell set
	    for (int p = 0; p < (int)npolys; p++) {
	       for (i=in_ncl[p*2]; i<=in_ncl[p*2+1]; i++)
	       {
		  // write triangle or polyline connectivity
		  // NOTE: will contain degenerates, no checking here...
		  if (i <= in_ncl[p*2+1] - 2) {
		     if (cell_type[cs]==CELL_POLYTRI &&
			 check_degen(i,in_coords,nspace)) {
		        if ((i-in_ncl[p*2])%2==1) // tristrip parity ordering
			   fprintf(outfile, "%d %d tri %d %d %d\n",
				   gcell, 1,
				   i+1+gcount, i+2+gcount, i+3+gcount);
			else
			   fprintf(outfile, "%d %d tri %d %d %d\n",
				   gcell, 1,
				   i+3+gcount, i+2+gcount, i+1+gcount);
		     }
		  }
		  if (i <= in_ncl[p*2+1] - 1) {
		     if (cell_type[cs]==CELL_POLYLINE)
		        fprintf(outfile,"%d %d line %d %d\n",
				gcell, 1,
				i+1+gcount, i+2+gcount);
		  }

		  // status and interrupt check
		  if ( (i%LINES_PER_STATUS)==0) {
		     OMstatus_check((int)(100*i/(float)npolys),
				    "write ucd", &interrupt);
		     if (interrupt) {
#ifdef DEBUG
		        ERRverror("", ERR_PRINT|ERR_NO_HEADER,
                                  "got an interrupt, stopping");
#endif
			ARRfree(in_coords);
			ARRfree(in_ncl);
			ARRfree(xfm);
			ARRfree(xlate);
			fprintf(outfile," *** FILE WRITE INTERRUPTED! ***\n");
			fclose(outfile);
			return(1);
		     }
		  }
	       } // end of vertex loop
	    } // end of primitive loop
	    break;
         }
#endif

	 default:
	    // This should just add a comment to the UCD file to indicate
	    // that we don't understand the current cell-set.
	    // Unfortunately the Read_UCD module crashes if it finds a
	    // comment in the middle of a file.
	    // fprintf(outfile, "# ???? UNKNOWN-POLY\n");
	    break;
	 } // end of switch

	 if (props) ARRfree(props);

      } // end of cell set loop

      // Write Cell topology lists and X, Y, Z coordinates for nodes
      if (UCD_BINARY) {

	 // Write Cell topology lists
	 fwrite(cell_topology_lists, sizeof(int), num_nlist_nodes, outfile);

#ifdef DEBUG_BINARY
	 for( i =0; i < num_nlist_nodes; i++ ) {
	    fprintf(stderr, "%d ", cell_topology_lists[i]);
	}
#endif

	free(cell_topology_lists);

	/*
	 * Write Coordinates
	 */

	 float *xyz = NULL;
	 xyz = (float *) malloc (sizeof(float) * nnodes);
	 if (xyz == NULL) {
	    err = 1;
	    err_str = "Could not allocate memory for cell coordinates";
	    fclose(outfile);
	    return(1);
	 }

	 // Write X coordinates for nodes
	 for (i = 0; i < nnodes; i++) xyz[i] = in_coords[i*nspace+0];
	 fwrite(xyz, sizeof(float), nnodes, outfile);

#ifdef DEBUG_BINARY
	 fprintf(stderr, "\n");
	 for (i = 0; i < nnodes; i++) fprintf(stderr, "%f ", xyz[i]);
#endif

	 // Write Y coordinates for nodes
	 for (i = 0; i < nnodes; i++) xyz[i] = in_coords[i*nspace+1];
	 fwrite(xyz, sizeof(float), nnodes, outfile);

#ifdef DEBUG_BINARY
	 fprintf(stderr, "\n");
	 for (i = 0; i < nnodes; i++) fprintf(stderr, "%f ", xyz[i]);
#endif

	 // Write Z coordinates for nodes
	 for (i = 0; i < nnodes; i++){
	    if(nspace == 3 ) xyz[i] = in_coords[i*nspace+2];
	    else xyz[i]= 0.0;
	 }
	 fwrite(xyz, sizeof(float), nnodes, outfile);

#ifdef DEBUG_BINARY
	 fprintf(stderr, "\n");
	 for (i = 0; i < nnodes; i++) fprintf(stderr, "%f ", binary_z[i]);
	 fprintf(stderr, "\n");
#endif
	 free(xyz);
      }

      //
      // Node Data
      //

      if (total_nnode_data > 0) {

         FLD_Node_Data_node_data_array &node_data = in[nf].node_data;

         // Node data for binary file format
         if (UCD_BINARY) {
	    char node_data_labels[1024], node_data_units[1024];
	    float *node_data_minimums, *node_data_maximums;
	    int *node_active_list;
	    int veclen;

	    node_data_minimums = (float *) malloc (sizeof(float) * total_nnode_data);
	    node_data_maximums = (float *) malloc (sizeof(float) * total_nnode_data);
	    node_active_list = (int *) malloc (sizeof(int) * total_nnode_data);

	    // Write labels for each component
	    int ptr =0;
	    for( i=0; i<nnode_data; i++) {
               int len = strlen((char *)node_data[i].labels);
	       for (j =0; j < len; j++)
		  node_data_labels[ptr++] = node_data[i].labels[j];
	       node_data_labels[ptr++]='.';
	    }
            // blank the rest, backtracking over the trailing last period
	    for(i=ptr-1; i < 1024; i++) node_data_labels[i] = 0;
	    fwrite(node_data_labels, sizeof(char), 1024, outfile);

	    // Write units for each component
	    ptr =0;
	    for( i=0; i<nnode_data; i++) {
               int len = strlen((char *)node_data[i].units);
	       for (j =0; j < len; j++)
		  node_data_units[ptr++] = node_data[i].units[j];
	       node_data_units[ptr++]='.';
	    }
            // blank the rest, backtracking over the trailing last period
	    for(i=ptr-1; i < 1024; i++) node_data_units[i] = 0;
	    fwrite(node_data_units, sizeof(char), 1024, outfile);

	    // Write number of node components
	    fwrite(&nnode_data, sizeof(int), 1, outfile);

#ifdef DEBUG_BINARY
	    fprintf(stderr, "nnode_data %d\n", nnode_data);
#endif

	    // Write node component list
	    for( i=0; i<nnode_data; i++) {
	       veclen = node_data[i].veclen;
	       fwrite(&veclen, sizeof(int), 1, outfile);
	    }
            if( nnode_data < total_nnode_data ) {
              // Just to space it out properly.
              for( i=nnode_data; i<total_nnode_data; i++) {
                  veclen = 0;
                  fwrite(&veclen, sizeof(int), 1, outfile);
               }
            }

	    // Write minimums & maximums for node data

	    // Find min and max of node data
            int ii = 0;
	    for(i=0; i<nnode_data; i++) {
               veclen = in[nf].node_data[i].veclen;
               if( veclen == 1 ) {
                  node_data_minimums[ii] = in[nf].node_data[i].min;
	          node_data_maximums[ii] = in[nf].node_data[i].max;
                  ii++;
               }
               else {
                  float *min_vec = (float *)in[nf].node_data[i].min_vec.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);
                  float *max_vec = (float *)in[nf].node_data[i].max_vec.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);
                  for(j=0; j<veclen; ++j) { 
                     node_data_minimums[ii] = min_vec[j];
                     node_data_maximums[ii] = max_vec[j];
                     ii++;
                  }
                  ARRfree(min_vec);
                  ARRfree(max_vec);
               }
	    }

	    // Write the min and max for node data
	    fwrite(node_data_minimums, sizeof(float), total_nnode_data, outfile);
	    fwrite(node_data_maximums, sizeof(float), total_nnode_data, outfile);

	    // Write Node data
	    float *node_data_raw = NULL;

	    for (i = 0; i<nnode_data; i++) {
	       veclen = in[nf].node_data[i].veclen;
	       if (veclen > 0) {
		  node_data_raw = (float *)
		    in[nf].node_data[i].values.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);
		  fwrite(node_data_raw, sizeof(float), nnodes * veclen, outfile);
                  ARRfree(node_data_raw);
	       }
	    }

	    // Output cell active list.
	    // Blank until meaning of 'cell_active_list' is discovered.
	    for(i=0; i<total_nnode_data; i++) node_active_list[i] = 0;
	    fwrite(node_active_list, sizeof(int), total_nnode_data, outfile);

	    free(node_data_minimums);
	    free(node_data_maximums);
	    free(node_active_list);
	 }
         // Node data for ASCII file format
         else {
	    int* nd_veclen;
	    float** node_data;

	    // Allocate and fill arrays to store veclen and actual node data
	    nd_veclen = (int*)malloc(nnode_data * sizeof(int));
	    node_data = (float**)malloc(nnode_data * sizeof(float*));

	    for (i = 0; i<nnode_data; i++) {
	       nd_veclen[i] = in[nf].node_data[i].veclen;

               if (nd_veclen[i] > 0) {
                  node_data[i] = (float *)
                     in[nf].node_data[i].values.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);
               }
               else {
                  node_data[i] = NULL;
               }
            }

            // Write line describing node data components
            // (number of components, veclen for each component...)
            fprintf(outfile, "%d", nnode_data);
            for (i = 0; i < nnode_data; i++) {
               fprintf(outfile, " %d", nd_veclen[i]);
            }
            fprintf(outfile, "\n");

            // Write out label and units for each component
            for (i = 0; i<nnode_data; i++) {
               fprintf(outfile, "%s, %s\n", 
                       (char*)in[nf].node_data[i].labels,
                       (char*)in[nf].node_data[i].units);
            }

            // Write out node data
            for (i = 0; i < nnodes; i++){
               fprintf(outfile, "%d", i+1);

               for (int nd=0; nd < nnode_data; nd++){
                  if (nd_veclen[nd] > 0) {
                     for (int ndv=0; ndv < nd_veclen[nd]; ndv++) {
                        fprintf(outfile, " %f", 
                                node_data[nd][i*nd_veclen[nd] + ndv]);
                     }
                  }
               }

               fprintf(outfile, "\n");
            }

            // Free allocated memory
            for (i = 0; i<nnode_data; i++) {
               if (node_data[i] != NULL)
                  ARRfree(node_data[i]);
            }

            free(nd_veclen);
            free(node_data);
         }
      } // if node data


      //
      // Write Cell data
      //

      ncell_data_id = OMfind_subobj(in[nf].cell_set[0].obj_id(),
                                    OMstr_to_name("ncell_data"), OM_OBJ_RW);

      int returned_val;

      // Check if Cell Data is present
      if (OMget_int_val(ncell_data_id, &returned_val) == 1) {
         int size;

         // Hmm, doesn't work
         //FLD_Cells_cell_set_array &cell_sets = in[nf].cell_set;

         // Write UCD binary cell data if present
         if (UCD_BINARY) {
            char cell_data_labels[1024], cell_data_units[1024];
            float *cell_data_minimums, *cell_data_maximums;
            int *cell_active_list;

            cell_data_minimums = (float *) malloc (sizeof(float) * total_ncell_data);
            cell_data_maximums = (float *) malloc (sizeof(float) * total_ncell_data);
            cell_active_list = (int *) malloc (sizeof(int) * total_ncell_data);

            // Write labels for each component
            int ptr = 0;
            for( i=0; i<ncell_data; i++) {
               int len = strlen((char *)in[nf].cell_set[0].cell_data[i].labels);
               for (j =0; j < len; j++) {
                  cell_data_labels[ptr++] = 
                     in[nf].cell_set[0].cell_data[i].labels[j];
               }
               cell_data_labels[ptr++]='.';
            }

            for(i=ptr-1; i < sizeof(cell_data_labels); i++)
               cell_data_labels[i] = 0;
            fwrite(cell_data_labels, sizeof(char), 1024, outfile);

            // Write units for each component
            ptr = 0;
            for( i=0; i<ncell_data; i++) {
               int len = strlen((char *)in[nf].cell_set[0].cell_data[i].units);
               for (j =0; j < len; j++) {
                  cell_data_units[ptr++] = 
                    in[nf].cell_set[0].cell_data[i].units[j];
               }
               cell_data_units[ptr++]='.';
            }
            for(i=ptr-1; i < sizeof(cell_data_units); i++)
              cell_data_units[i] = 0;
            fwrite(cell_data_units, sizeof(char), 1024, outfile);

            // Write number of cell components
            fwrite(&total_ncell_data, sizeof(int), 1, outfile);

#ifdef DEBUG_BINARY
            fprintf(stderr, "ncell_data %d\n", ncell_data);
#endif

            // Write cell component list
            for( i=0; i<ncell_data; i++ ) {
               veclen = in[nf].cell_set[0].cell_data[i].veclen;
               fwrite(&veclen, sizeof(int),1,outfile);
            }
            if( ncell_data < total_ncell_data ) {
               // Just to space it out properly.
               for( i=ncell_data; i<total_ncell_data; i++) {
                  veclen = 0;
                  fwrite(&veclen, sizeof(int), 1, outfile);
               }
            }

            // Find min and max
            // (probably a mistake to only consider cell_set[0]).
            int ii = 0;
            for(i=0; i<ncell_data; i++) {
               veclen = in[nf].cell_set[0].cell_data[i].veclen;
               if( veclen == 1 ) {
                  cell_data_minimums[ii] = in[nf].cell_set[0].cell_data[i].min;
                  cell_data_maximums[ii] = in[nf].cell_set[0].cell_data[i].max;
                  ii++;
               }
               else {
                  float *min_vec = (float *)in[nf].cell_set[0].cell_data[i].min_vec.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);
                  float *max_vec = (float *)in[nf].cell_set[0].cell_data[i].max_vec.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);
                  for(j=0; j<veclen; ++j) { 
                     cell_data_minimums[ii] = min_vec[j];
                     cell_data_maximums[ii] = max_vec[j];
                     ii++;
                  }
                  ARRfree(min_vec);
                  ARRfree(max_vec);
               }
            }

            // Write the min and max for cell data
            fwrite(cell_data_minimums, sizeof(float), total_ncell_data, outfile);
            fwrite(cell_data_maximums, sizeof(float), total_ncell_data, outfile);

            // Write Cell data
            float *cell_data = NULL;

            for(i=0;i< ncell_data; i++){
               for(j=0; j<(int)in[nf].ncell_sets; j++) {

                  cell_data = (float *) in[nf].cell_set[j].cell_data[i].values.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);

                  fwrite(cell_data, sizeof(float), (int)in[nf].cell_set[j].ncells * (int)in[nf].cell_set[j].cell_data[i].veclen, outfile);

#ifdef DEBUG_BINARY
                  for (k=0; k< (int)in[nf].cell_set[j].ncells * (int)in[nf].cell_set[j].cell_data[i].veclen; k++)
                     fprintf(stderr, "%f ", cell_data[k]);
                  fprintf(stderr, "\n");
#endif
                  ARRfree(cell_data);
               }
            }

            // Output cell active list (To be implemented)
            // Initialize to zero until what 'cell_active_list' is found
            for(i=0; i<total_ncell_data; i++) cell_active_list[i] = 0;
            fwrite(cell_active_list, sizeof(int), total_ncell_data, outfile);

            free(cell_data_minimums);
            free(cell_data_maximums);
            free(cell_active_list);
         }
         // Write UCD ASCII cell data
         else {
            // Write the number of Components
            fprintf(outfile, "%d ", ncell_data);

	    // Write veclen for each component
	    for( i=0; i< ncell_data; i++)
	       fprintf(outfile, "%d ", (int)in[nf].cell_set[0].cell_data[i].veclen);
	    fprintf(outfile, "\n");

	    // Write labels and units for each component
	    for( i=0; i< ncell_data; i++)
	       fprintf(outfile, "%s, %s\n",
		       (char *)in[nf].cell_set[0].cell_data[i].labels,
		       (char *)in[nf].cell_set[0].cell_data[i].units );

	    //
	    // Write ID (generated), values for each component for every cell
	    //
	    id = 0;
	    float **cell_data;

	    // Loop thru each cell set
	    for (i=0; i<(int)in[nf].ncell_sets; i++) {
	       cell_data = (float **) malloc ((int)in[nf].cell_set[i].ncell_data *sizeof (float*));

	       for (j=0; j<(int)in[nf].cell_set[i].ncell_data; j++) {
		  cell_data[j] = (float *)in[nf].cell_set[i].cell_data[j].values.ret_typed_array_ptr(OM_GET_ARRAY_RD, DTYPE_FLOAT, &size);
	       }

	       // Loop thru each cell
	       for (k =0; k < (int)in[nf].cell_set[i].ncells; k++) {

		  // Generate and print ID
		  id++;
		  fprintf(outfile, "%d ", id);

		  // Write Cell data
		  for ( j=0; j<(int)in[nf].cell_set[i].ncell_data;j++) {
                     /* What if veclen != 1 ? */
		     fprintf(outfile, "%f ", cell_data[j][k]);
		  }
		  fprintf(outfile, "\n");
	       }

	       for( j=0; j< (int)in[nf].cell_set[i].ncell_data; j++) {
		  ARRfree( cell_data[j] );
	       }

               free( cell_data );
	    }
	 }
      } // if cell data


#ifdef DEBUG
      ERRverror("", ERR_PRINT|ERR_NO_HEADER,
                "freeing memory for current field");
#endif

      // free arrays, communicating changes to OM
      ARRfree(in_coords);
      ARRfree(xfm);
      ARRfree(xlate);
      ARRfree(in_ncl);

      // update global node count, across fields
      gcount += nnodes;

   } // end of ARR field loop

   // close the file, finished writing
   fclose(outfile);

#ifdef DEBUG
   ERRverror("", ERR_PRINT|ERR_NO_HEADER, "file closed");
#endif

   // clean up user interface
   err = 0;
   err_str = "";

#ifdef DEBUG
   ERRverror("", ERR_PRINT|ERR_NO_HEADER, "finished, returning\n\n");
#endif

   // return 1 for success
   return(1);
}


#ifdef OUTPUT_POLY_TEST

// function to check for degenerate triangles
//	 checks if any two coordinates are coincident
static int check_degen(int i, float *in_coords, int nspace) {

  float r1, r2, r3;
  int p, q;

  p = 0; q = 1; // first compare
  r1 =	 _ABS( in_coords[(i+p)*nspace+0] - in_coords[(i+q)*nspace+0]) +
         _ABS( in_coords[(i+p)*nspace+1] - in_coords[(i+q)*nspace+1]);
  if (nspace == 3)
     r1 += _ABS( in_coords[(i+p)*nspace+2] - in_coords[(i+q)*nspace+2]);

  p = 1; q = 2; // second compare
  r2 =	 _ABS( in_coords[(i+p)*nspace+0] - in_coords[(i+q)*nspace+0]) +
         _ABS( in_coords[(i+p)*nspace+1] - in_coords[(i+q)*nspace+1]);
  if (nspace == 3)
     r2 += _ABS( in_coords[(i+p)*nspace+2] - in_coords[(i+q)*nspace+2]);

  p = 0; q = 2; // third compare
  r3 =	 _ABS( in_coords[(i+p)*nspace+0] - in_coords[(i+q)*nspace+0]) +
         _ABS( in_coords[(i+p)*nspace+1] - in_coords[(i+q)*nspace+1]);
  if (nspace == 3)
     r3 += _ABS( in_coords[(i+p)*nspace+2] - in_coords[(i+q)*nspace+2]);

  if ( r1 <= ( 1.0e-12 ) ||
       r2 <= ( 1.0e-12 ) ||
       r3 <= ( 1.0e-12 ) )
     return(0);
  else
     return(1);
}

#endif


// =================== end of source file ================

