/*
 * INTERNATIONAL AVS CENTRE - WARRANTY DISCLAIMER
 * Please read the file DISCLAIMER for conditions associated with this file.
 * avs@iavsc.org, www.iavsc.org
 */

/* ----------------------------------------------------------------------
 * FlagRaw Module
 * ----------------------------------------------------------------------
 * Description:
 *   Flag simulation module from AVS XP_DEMO CD.
 *   Converted from Dore' to AVS/Express by I. Curington
 *
 * Author: Ian Curington
 *
 * Modification History:
 *   12 April 1995 - ianc - CREATED express version
 *   16 April 1995 - ianc - fld call reorg
 *   23 May   1995 - ianc - fld call reorg 2
 *    8 June  1995 - ianc - optimization of field creation
 *   28 Sept  1995 - ianc - renamed sqrtl to sqrt_tbl
 *   13 Feb   1996 - ianc - changed to 3-arg omethod
 *   13 May   1996 - ianc - new parameter exposed
 *   13 Dec   1996 - ianc - fixed compile warning
 *   19 Mar   1998 - Paul Lever - modified for IAC repository
 *   30 Oct   2000 - Andrew Dodd - fixed compile warnings on Windows
 * ----------------------------------------------------------------------
 * Note:
 *   The gen.h include file is generated by Express when the module is 
 *   compiled. It avoids including "user.h" or "express.h" directly, so 
 *   that the module may be moved across the processes by changing the V
 *   properties in the library.
 * ----------------------------------------------------------------------
 */

#include "iac_proj/flag/gen.h"

/***************************************************************************
Copyright (C) 1992, 1993
by Kubota Pacific Computer Inc.  All Rights Reserved.
This program is a trade secret of Kubota Pacific Computer Inc. and
it is not to be reproduced, published, disclosed to others, copied,
adapted, distributed, or displayed without the prior authorization
of Kubota Pacific Computer Inc.  Licensee agrees to attach or embed
this Notice on all copies of the program, including partial copies
or modified versions thereof.
***************************************************************************/

#include <stdio.h>
#include <math.h>


#include "avs/err.h"
#include "avs/om.h"
#include "avs/gd_def.h"

#include "flag.h"


/* Search for string 'NOTE' for efficiancy notes */

/* NOTE: Many divs can be omitted on horizontal and vertical (but not diagonal)
** links if we assume a tight enough system that 'length d' ** will always be
** close to 1.  */

    /****************************/
    /***  Defined Parameters  ***/
    /****************************/

/**
#define DBG_MESSAGE 1
 **/

	/* Dimensions of Flag */

#define sizex 20
#define sizey 12

	/* Weight of a point is always 1.0.  Prefered length of horizontal
	** or vertical spring is always 1.0
	**     Strength of springs: ( force = strength * displacement )
	**     Friction: ( force = fric * vel )
	*/

	/* Number of seconds per sim step: */

#define timestep  0.1

/**
 Now Global variable controlled by UI, below
#define strength 20.0
 **/

	/* NOTE: statements involving fric have been commented out on the
	** assumption that friction is 1.0. */

#define fric  1.00
#define G     0.05		/* Gravity */

	/* Resolution of the square-root lookup table.  */

#define sqrtres   4096
#define sqrtscale (sqrtres / 8.0)

	/* Arrays are indexed with 0 in the 'lower left'.  X increments by 1,
	** y increments by sizex.  (Scanline order, bottom to top.) 
	** coords are stored all x's, then all y's, then z's,
	** xxxx, yyyy, zzzz. */

#define len1       (sizex * sizey)
#define len3       (3 * len1)
#define veci(lim)  for (i=0; i < (lim); i++)

#if 0
#   define xoff 0
#   define yoff len1
#   define zoff (len1*2)
#else
    static int xoff = 0;
    static int yoff = len1;
    static int zoff = 2 * len1;
#endif


    /***************************/
    /***  Macro Definitions  ***/
    /***************************/

#define VINDEX(x,y)	(((y) * sizex) + (x))

#define CLAMP(x,lo,hi)  (((x) < (lo)) ? (lo) : (((x) > (hi)) ? (hi) : (x)))

static int    skip[] = { 0, 12, 2, 1 };	/* Number of Steps Between Display */

#define ERR_RETURN(A) { \
   ERRerror("flag demo", 1, ERR_ORIG, A); return(0); \
}


#define RESOLVE_INT_PARM(elem_id,string,flag,ii)             \
  {local_id = OMfind_subobj((elem_id),                       \
              OMstr_to_name((string)),flag);                 \
  if ( OMis_null_obj(local_id))                              \
  {                                                          \
      ERRerror("flag_demo",1,ERR_ORIG,                       \
               "search failed for subobj int param name");   \
      return(0);                                             \
  }                                                          \
  if ( OMget_int_val(local_id,&(ii)) != 1 )                  \
  {                                                          \
      ERRerror("flag_demo",1,ERR_ORIG,                       \
               "search failed for subobj int param value");  \
      return(0);                                             \
  }                                                          \
  }

#define RESOLVE_FLT_PARM(elem_id,string,flag,xx)             \
  {local_id = OMfind_subobj((elem_id),                       \
              OMstr_to_name((string)),flag);                 \
  if ( OMis_null_obj(local_id))                              \
  {                                                          \
      ERRerror("flag_demo",1,ERR_ORIG,                       \
               "search failed for subobj flt param name");   \
      return(0);                                             \
  }                                                          \
  if ( OMget_real_val(local_id,&(xd)) != 1 )                 \
  {                                                          \
      ERRerror("flag_demo",1,ERR_ORIG,                       \
               "search failed for subobj flt param value");  \
      return(0);                                             \
  }                                                          \
  (xx) = xd;                                                 \
  }

    /**************************/
    /***  Global Variables  ***/
    /**************************/

/*
 * User - Interface Control Structure
 */
static Ui ui;

/*
 * Main Loop Counter for application
 */
static int traversal_counter=0;

/*
 * output structures, for graphics, once CFD solution is complete
 */
static float    Vertices[sizex*sizey*3];
static float    NodeData[sizex*sizey*3];
static int      Dim0 = sizex;
static int      Dim1 = sizey;

    /* Point location, velocity, force (= acceleration, as mass is 1.).  */

double pxyz[len3];  /* x, y, z coord of point */
double vxyz[len3];  /* x, y, z velocity vector */
double fxyz[len3];  /* x, y, z force vector */

/*
 * Elastic Strength of each spring
 */
static double strength = 20.0;

    /* Temporary variables.  */

double dxyz[len3];
double txyz[len3];

double ld[len1];    /* Length of 'd' */
double sf[len1];    /* Spring Force */
double ff[len1];    /* Friction Force */

/* The assumption is that the point displacement from spring neutral (a
** distance of 1.0) will never exceed 1.0.  Therefore, the sqrt must produce
** values from 0.0 to 2.0, leaving an input range from 0 to 4.
**
** So, sqrt(r) = sqrt_tbl[(int)(r*((sqrtres-1)/4.))]
**             = sqrt_tbl[(int)(r*sqrtscale)]
**
** NOTE: with the addition of diagonal connections, the range should be
** increased to accomodate the added length.  */

    /* The divisor here should be greater than the square of the
    ** greatest stretched connection length expected.  */

double sqrt_tbl[sqrtres];

	/* Colors */

static float ball_color[3] = { 0.6, 0.6, 0.8 };
static float pole_color[3] = { 0.6, 0.6, 0.8 };
static float flag_color[3] = { 0.7, 0.3, 0.6 };
static float spec_color[3] = { 0.8, 0.8, 0.8 };

/* Texture coords */
static float *Tu = 0;	/* Texture U Coordinates */
static float *Tv = 0;	/* Texture V Coordinates */



/*******************************/
/***  Main Express Entry Pnt ***/
/*******************************/

int
flag_demo(OMobj_id object, OMevent_mask event_mask, int seq_num)

{
   OMobj_id field_id, local_id, traverse_id;
   int      count, vert, num_verts; 
   int      obj, num_obj;
   int      veclen, ndims, dims[3]; 
   float    x, y, z;  
   int      i;
   double   xd;
   float    *points;
   float    *coords;
   int      nnodes;
   float    *node_data;
   int      type, size;
   int      first_time;

   /*
    * Reset all values in ui block
    */

   /* just used for input trigger */
   RESOLVE_INT_PARM(object,"count",OM_OBJ_RD,count);
   RESOLVE_INT_PARM(object,"reset",OM_OBJ_RD,ui.flag_reset);
   RESOLVE_INT_PARM(object,"flag_color",OM_OBJ_RD,ui.flag_color);
   RESOLVE_INT_PARM(object,"flag_motion",OM_OBJ_RD,ui.flag_motion);
   RESOLVE_INT_PARM(object,"release_top",OM_OBJ_RD,
		    ui.flag_release[RELEASE_TOP]);
   RESOLVE_INT_PARM(object,"release_bottom",OM_OBJ_RD,
		    ui.flag_release[RELEASE_BOTTOM]);
   RESOLVE_FLT_PARM(object,"flag_wind_1",OM_OBJ_RD,ui.flag_wind[0]);
   RESOLVE_FLT_PARM(object,"flag_wind_2",OM_OBJ_RD,ui.flag_wind[1]);
   RESOLVE_FLT_PARM(object,"flag_strength",OM_OBJ_RD,ui.flag_strength);

   RESOLVE_INT_PARM(object,"traverse",OM_OBJ_RW,traversal_counter);
   traverse_id = local_id;

   /***********************
    * Run CFD Simulation  *
    ***********************/
#ifdef DBG_MESSAGE
   printf("flag: traversal_counter = %d\n",traversal_counter);
#endif

   strength = ui.flag_strength;

   if ( traversal_counter == 0 )
   {
       make_flag_model_group();
       first_time = 1;
   }
   else
       first_time = 0;

   update_flag_callback();

   /*
    * Update the traversal counter in the OM
    */
   OMset_int_val( traverse_id, traversal_counter );

   /*
    * Get id of field
    */
   field_id = OMfind_subobj(object, OMstr_to_name("out"), OM_OBJ_RW);
   if (OMis_null_obj(field_id)) {
      ERR_RETURN("Error searching for field");
   }

   nnodes = Dim0 * Dim1;
   ndims = 2;
   dims[0] = Dim0;
   dims[1] = Dim1;

   /*
    * set characteristics of output field
    */
   if (ui.flag_color == COLOR_SOLID)    veclen = 0;
   if (ui.flag_color == COLOR_TEXTURE)  veclen = 2;
   if (ui.flag_color == COLOR_VELOCITY) veclen = 3;
   if (ui.flag_color == COLOR_FORCE)    veclen = 3;
   if (ui.flag_color == COLOR_FORCEMAG) veclen = 1;

   num_verts = len1;

#ifdef DBG_MESSAGE
   printf("flag: nnodes = %d\n",nnodes);
   printf("flag: ndims = %d\n",ndims);
   printf("flag: dims = %d %d\n",dims[0], dims[1]);
   for (i=0; i < 10; i++)
   printf("flag: coord %d = %f %f %f\n",i,Vertices[3*i],
				          Vertices[3*i+1],
				          Vertices[3*i+2]);
#endif

   /*
    * Only set up the entire field structure the first time,
    * then after only update the parts that change, like
    * data and coords
    */

   if ( first_time )
   {
       /*
        * set dimensions of mesh
        */
       if (FLDset_ndim(field_id,ndims) != 1) {
               ERR_RETURN("Error setting ndim of output field");
       }
#ifdef DBG_MESSAGE
   printf("flag: set ndim ok\n");
#endif

       if (FLDset_dims(field_id, dims) != 1) {
               ERR_RETURN("Error setting dims array of node data");
       }
#ifdef DBG_MESSAGE
   printf("flag: set dims ok\n");
#endif

       /*
        * Define number of nodes in the field
        */
       if (FLDset_nnodes(field_id, nnodes) != 1) {
          ERR_RETURN("Error setting nnodes");
       }
#ifdef DBG_MESSAGE
   printf("flag: set nnodes ok\n");
#endif

       /*
        * Define rspace/nspace of field
        */
       if (FLDset_nspace(field_id, 3) != 1) {
          ERR_RETURN("Error setting nspace");
       }
#ifdef DBG_MESSAGE
   printf("flag: set nspace ok\n");
#endif

   } /* end of first time condition */

   /*
    * Define coordinates of field
    */
   if (FLDget_coord(field_id, &coords, &size,
                    OM_GET_ARRAY_WR) != 1) {
      ERR_RETURN("Error setting coord");
   }
#ifdef DBG_MESSAGE
   printf("flag: got coord pointer ok\n");
#endif

   /*
    * copy over the coord array
    */
   for (i=0; i < nnodes * 3; i++)
       coords[i] = Vertices[i];
   ARRfree((char *) coords);

#ifdef DBG_MESSAGE
   printf("flag: ARRfree on coords ok\n");
#endif

   /*
    * Any node data to add?
    */


   if ( veclen > 0 )
   {
      /*
       * how many components?
       */
      if (FLDset_node_data_ncomp (field_id, 0) != 1) {
                ERR_RETURN("Error zero-ing nnode_data_ncomp");
      }
      if (FLDset_node_data_ncomp (field_id, 1) != 1) {
                ERR_RETURN("Error setting nnode_data_ncomp");
      }

      /*
       * Define vector length of node data
       */
      if (FLDset_node_data_comp(field_id,0,veclen,"Flag","Flag_Units") != 1) {
          ERR_RETURN("Error setting node_data_comp");
      }

      /*
       * ID for what type of data we have
       */
      if ( veclen == 3 )
      {
          if (FLDset_node_data_id(field_id, 0, GD_COLOR_DATA_ID) != 1) {
                 ERR_RETURN("Error setting 3-vector RGB data id flag");
          }
      }
      else if ( veclen == 2 )
      {
          if (FLDset_node_data_id(field_id, 0, GD_UV_DATA_ID) != 1) {
                 ERR_RETURN("Error setting UV-texture data id flag");
          }
      }
   
      /*
       * load in the node data
       */
      type = OM_TYPE_FLOAT;
      size = veclen*num_verts;
      if (FLDget_node_data(field_id, 0, &type,
                             (char **)&node_data,
                             &size,
                             OM_GET_ARRAY_WR) != 1) {
                ERR_RETURN("cannot set node data on output");
      }

      for (i=0; i < veclen*num_verts; i++)
          node_data[i] = NodeData[i];
      ARRfree((char *) node_data);

   }
   else
   {
      if (FLDset_node_data_ncomp (field_id, 0) != 1) {
                ERR_RETURN("Error setting nnode_data_ncomp to zero");
      }
   }

   return(1);
}

/*******************************/
/***  Main Express Destroy Cleanup Routine ***/
/*******************************/

int
flag_demo_del(OMobj_id object, OMevent_mask event_mask, int seq_num)

{
   traversal_counter = 0;
   ui.flag_reset = TRUE;
   if (Tu) free (Tu);
   if (Tv) free (Tv);
   Tu = Tv = 0;

   return(1);
}

    /*******************************/
    /***  Function Declarations  ***/
    /*******************************/


/**************************************************************************/

int
make_flag_model_group (void)
{
    /*
     * initialize system
     */

#ifdef DBG_MESSAGE
   printf("flag: make_flag_model_group function.\n");
#endif

    init_sqrt ();
    init_flag ();
    calc_wind ();
    createflag();

    ui.flag_reset = FALSE;

    return (1);
}

/**************************************************************************/

void update_flag_callback (void)
{ 
    auto   int    ii;
    static int    prev_color = -1;
    auto   int    move_flag;

#ifdef DBG_MESSAGE
   printf("flag: update_flag_callback function.\n");
#endif

    /* If the flag reset button was pushed, then re-initialize the flag. */

    if (ui.flag_reset)
    {   ui.flag_reset = FALSE;
	init_flag ();
    }

    /* Make several passes over the data before updating the display. */

    for (ii=0;  ii < skip [ui.flag_motion];  ++ii) {
	forceflag ();
	externalforces ();
	moveflag ();
    }

    move_flag = (ui.flag_motion != MOTION_NONE) || (traversal_counter == 0);

    /* If the flag has moved (due to wind, not view changes) or has not yet
    ** been created, then create the flag object. */

    if (move_flag) {
   	createflag();
    }

    /* Execute the flag object and bump the traversal count. */

    /*
       DISPLAY IT?

    DsExecuteObj(flagobject);

     */
    traversal_counter++;
}

/**************************************************************************/

void forceflag (void)
{
    calc_wind();
    /*
     * Start with 0 force.
     */
    veccopyc(0.,fxyz,len3);
    
    /*
     * Essentially, we are calculating
     *
     * (strength * (l-1.)) + (fric * (v dot d)/l)
     * ------------------------------------------ * d
     *                     l
     *
     * where d is the vector between the two points, l is the length of d,
     * and v is the difference between their velocities.  This handles
     * spring force Fs = -k(x-x0), and friction force Ff = -Rv, where
     * k = strength, R = fric, and x0 = 1.
     *
     * Each of the following functions is specialized to calculate the
     * indicated forces by the above method in as vectorizable a fashion
     * as possible.
     */

    force_horizontal ();
    force_vertical ();
    force_diagonal ();
}

/**************************************************************************/

void force_horizontal (void)
{
    int i;

    vecsub3(pxyz,pxyz+1,dxyz,len3-1);	/* d = p1-p (vec from p to p1) */
    vecmul3(dxyz,dxyz,txyz,len3-1);     /* t = d*d */
    vecadd(txyz+yoff,txyz,len1-1);	/* tx += ty */
    vecadd(txyz+zoff,txyz,len1-1);	/* tx += tz (tx=(length d)^2) */

    /* No interaction between ends of flag: */

    for (i=sizex-1; i<len1; i+=sizex)
	txyz[i] = 1.;

    vecmulc(sqrtscale,txyz,len1-1);    /* t *= sqrtscale        */

    veci(len1-1) {            /* ld = length(d)        */
        if ((int)txyz[i] >= sqrtres) {
            printf("force_horizontal: (int)txyz[%d] = %d\n",
                            i, (int)txyz[i]);
            txyz[i] = sqrtres - 1;
        }
        ld[i] = sqrt_tbl[(int)txyz[i]];
    }

    /*
     * Now, ld = length(d), and we are done with txyz.
     */

    /*
     * Using ld, find spring force numberator.  Leave
     * result in sf.
     */
    veccopy(ld,sf,len1-1);        /* sf = ld            */
    vecsubc(1.,sf,len1-1);        /* sf = ld-1.            */
    vecmulc(strength,sf,len1-1);    /* sf = strength * (ld-1.)    */

    /*
     * Using ld, d, and v, find friction
     * force numberator.  Leave result in txyz.
     */
    vecsub3(vxyz,vxyz+1,txyz,len3-1); /* t = v (rel velocity)    */
    vecmul(dxyz,txyz,len3-1);      /* t = v * d            */
    vecadd(txyz+yoff,txyz,len1-1);      /* ...            */
    vecadd(txyz+zoff,txyz,len1-1);      /* t = v dot d        */
/*NOTE    vecmulc(fric,txyz,len1-1);      *//* t = fric * (v dot d)    */
/*NOTE    vecdiv(ld,txyz,len1-1);          *//* t = fric * (v dot d) / l    */

    /*
     * Nullify any effects at the wrap around
     * points. (sides of flag)
     */
    for (i=sizex-1; i<len1; i+=sizex)
        txyz[i] = 0.;

    /*
     * Add forces together...
     */
    vecadd(txyz,sf,len1-1);

/*NOTE    vecdiv(ld,sf,len1-1);        *//* sf = (lots of stuff)/ld    */

    vecmul(sf,dxyz+xoff,len1-1);    /* d  = force vector.        */
    vecmul(sf,dxyz+yoff,len1-1);    /* d  = force vector.        */
    vecmul(sf,dxyz+zoff,len1-1);    /* d  = force vector.        */

    /*
     * Remember, when going from a len1 to
     * a len3, we must pay special attention to
     * what happens at the wrap around points.
     */
    
    dxyz[len1-1] = 0.;
    dxyz[yoff+len1-1] = 0.;

    vecadd(dxyz,fxyz,len3-1);    /* Force pulling right        */
    vecsub(dxyz,fxyz+1,len3-1);    /* Force pulling left        */
}

/**************************************************************************/

/*
 * See force_horizontal for in code comments.
 */
void force_vertical (void)
{
    int i;

    vecsub3(pxyz,pxyz+sizex,dxyz,len3-sizex);
    vecmul3(dxyz,dxyz,txyz,len3-sizex);
    vecadd(txyz+yoff,txyz,len1-sizex);
    vecadd(txyz+zoff,txyz,len1-sizex);
    vecmulc(sqrtscale,txyz,len1-sizex);    /* These two lines just set */
    veci(len1-sizex) {            /* ld = sqrt(txyz)   -NOTE- */
        if ((int)txyz[i] >= sqrtres) {
#ifdef DBG_MESSAGE
            printf("force_vertical: (int)txyz[%d] = %d\n",
                            i, (int)txyz[i]);
#endif
	    txyz[i] = sqrtres - 1;
        }
        ld[i] = sqrt_tbl[(int)txyz[i]];
    }
    veccopy(ld,sf,len1-sizex);
    vecsubc(1.,sf,len1-sizex);
    vecmulc(strength,sf,len1-sizex);
    vecsub3(vxyz,vxyz+sizex,txyz,len3-sizex);
    vecmul(dxyz,txyz,len3-sizex);
    vecadd(txyz+yoff,txyz,len1-sizex);
    vecadd(txyz+zoff,txyz,len1-sizex);
/*NOTE    vecmulc(fric,txyz,len1-sizex); */
/*NOTE    vecdiv(ld,txyz,len1-sizex);        *//* NOTE: divide by 1. */
    vecadd(txyz,sf,len1-sizex);
/*NOTE    vecdiv(ld,sf,len1-sizex);        *//* NOTE: divide by 1. */
    vecmul(sf,dxyz+xoff,len1-sizex);
    vecmul(sf,dxyz+yoff,len1-sizex);
    vecmul(sf,dxyz+zoff,len1-sizex);
    veccopyc(0.,dxyz+(len1-sizex),sizex);
    veccopyc(0.,dxyz+(yoff+(len1-sizex)),sizex);
    vecadd(dxyz,fxyz,len3-sizex);
    vecsub(dxyz,fxyz+sizex,len3-sizex);
}

/**************************************************************************/

/*
 * See force_horizontal for in code comments.
 */

/*
 * Length of a diagonal is NOT 1.
 */
#define dlen 1.414214

void force_diagonal (void)
{
    int i;

    vecsub3(pxyz,pxyz+(sizex+1),dxyz,len3-(sizex+1));
    vecmul3(dxyz,dxyz,txyz,len3-(sizex+1));
    vecadd(txyz+yoff,txyz,len1-(sizex+1));
    vecadd(txyz+zoff,txyz,len1-(sizex+1));
    for (i=sizex-1; i<len1-(sizex+1); i+=sizex)
        txyz[i] = 1.;            /* Anything safe for sqrt */
    vecmulc(sqrtscale,txyz,len1-(sizex+1));    /* These two lines just set */
    veci(len1-(sizex+1)) {            /* ld = sqrt(txyz)   -NOTE- */
        if ((int)txyz[i] >= sqrtres) {
/**
            printf("force_diagonal: (int)txyz[%d] = %d\n",
                            i, (int)txyz[i]);
**/
            txyz[i] = sqrtres - 1;
        }
        ld[i] = sqrt_tbl[(int)txyz[i]];
    }
    veccopy(ld,sf,len1-(sizex+1));
    vecsubc(dlen,sf,len1-(sizex+1));
    vecmulc(strength,sf,len1-(sizex+1));
    vecsub3(vxyz,vxyz+(sizex+1),txyz,len3-(sizex+1));
    vecmul(dxyz,txyz,len3-(sizex+1));
    vecadd(txyz+yoff,txyz,len1-(sizex+1));
    vecadd(txyz+zoff,txyz,len1-(sizex+1));
/*NOTE    vecmulc(fric,txyz,len1-(sizex+1)); */
/*NOTE    vecdiv(ld,txyz,len1-(sizex+1));        *//* NOTE: divide by dlen */
/*NOTE*/vecmulc((1./dlen),txyz,len1-(sizex+1));    /* NOTE: Substitute for above*/
    vecadd(txyz,sf,len1-(sizex+1));
/*NOTE    vecdiv(ld,sf,len1-(sizex+1));        *//* NOTE: divide by dlen */
/*NOTE*/vecmulc((1./dlen),sf,len1-(sizex+1));    /* NOTE: Substitute for above*/
    vecmul(sf,dxyz+xoff,len1-(sizex+1));
    vecmul(sf,dxyz+yoff,len1-(sizex+1));
    vecmul(sf,dxyz+zoff,len1-(sizex+1));
    veccopyc(0.,dxyz+(len1-sizex),sizex-1);
    veccopyc(0.,dxyz+(yoff+(len1-sizex)),sizex-1);
    for (i=sizex-1; i<len3-(sizex+1); i+=sizex)
        dxyz[i] = 0.;
    vecadd(dxyz,fxyz,len3-(sizex+1));
    vecsub(dxyz,fxyz+(sizex+1),len3-(sizex+1));
}

/**************************************************************************/

void externalforces (void)
{
    vecaddc ((-G), fxyz+yoff, len1);	/* Force Due to Gravity */
    force_wind ();			/* Force Due to Wind    */

    /* Zero total force on attatchment points:
    **
    ** To attatch all pole-side points:
    **
    ** for (i=0; i<len3; i+=sizex)
    **    fxyz[i] = 0.0;
    **
    ** To attattch only the corners:
    **/

    /* Release bottom of flag. */

    if (!ui.flag_release [RELEASE_BOTTOM])
    {   fxyz [0]    = 0.0;
        fxyz [yoff] = 0.0;
        fxyz [zoff] = 0.0;
    }

    /* Release top of flag. */

    if (!ui.flag_release [RELEASE_TOP])
    {   fxyz [       (sizex*(sizey-1))] = 0.0;
        fxyz [yoff + (sizex*(sizey-1))] = 0.0;
        fxyz [zoff + (sizex*(sizey-1))] = 0.0; 
    }
}

/***************************************************************************
** This takes the cross product of the connection to the right with the
** connection upward, to get the 'normal' of that node, which is then dotted
** with the relative wind.  (Relative wind is the absolute wind less the
** motion of that node.) That dot product is multiplied by the original
** normal vector, and added as force on that node, and perhaps the connected
** nodes.  (I haven't decided yet.) It should be noted that again assuming a
** connection length of 1., and a 90 degree connection, saves much hassle.
** 
** NOTE that the wind-friction factor is assumed 1., which is to say, the
** force generated by a wind of velocity 1. (in the same units as vxyz)
** hitting a square dead on (perpendicular) will generate a perpendicular
** force of 1..  This can make the flag look dead, but it also helps make it
** look light (very affected by the wind.)  If it looks to weightless, throw
** in a vecmulc of some less than 1. number just before adding this wind
** force to the global fxyz.
***************************************************************************/

static double windx,windy,windz;

void force_wind (void)
{
    static double norm[len3],rel[len3];
    static double a[len3],b[len1];
    int i;

    vecsub3(pxyz,pxyz+1,dxyz,len3-(sizex+1)); /* Rel vectors to right, */
    vecsub3(pxyz,pxyz+sizex,txyz,len3-(sizex+1)); /* and up. */

    /*
     * Cross product: norm = dxyz X txyz
     */
    vecmul3(dxyz+yoff,txyz+zoff,a,len1-(sizex+1));
    vecmul3(dxyz+zoff,txyz+yoff,b,len1-(sizex+1));
    vecsub3(b,a,norm,len1-(sizex+1));
    vecmul3(dxyz+zoff,txyz+xoff,a,len1-(sizex+1));
    vecmul3(dxyz+xoff,txyz+zoff,b,len1-(sizex+1));
    vecsub3(b,a,norm+yoff,len1-(sizex+1));
    vecmul3(dxyz+xoff,txyz+yoff,a,len1-(sizex+1));
    vecmul3(dxyz+yoff,txyz+xoff,b,len1-(sizex+1));
    vecsub3(b,a,norm+zoff,len1-(sizex+1));

    /*
     * Calculate rel wind/4.
     * (Divide by four cause we're gonna
     * add it to all four nodes on that
     * square.)
     */
    vecsubc3(windx,vxyz+xoff,rel+xoff,len1-(sizex+1));
    vecsubc3(windy,vxyz+yoff,rel+yoff,len1-(sizex+1));
    vecsubc3(windz,vxyz+zoff,rel+zoff,len1-(sizex+1));
    vecmulc(-0.25,rel,len3-(sizex+1));

    /*
     * Dot product: a = norm dot rel wind/4.
     */
    vecmul3(norm,rel,a,len3-(sizex+1));
    vecadd(a+yoff,a,len1-(sizex+1));
    vecadd(a+zoff,a,len1-(sizex+1));

    /*
     * Multiply norm by a.
     */
    vecmul(a,norm+xoff,len1-(sizex+1));
    vecmul(a,norm+yoff,len1-(sizex+1));
    vecmul(a,norm+zoff,len1-(sizex+1));

    /*
     * Clean up the wrap crap
     */
    veccopyc(0.,norm+(len1-sizex),sizex-1);
    veccopyc(0.,norm+(yoff+(len1-sizex)),sizex-1);
    for (i=sizex-1; i<len3-(sizex+1); i+=sizex)
        norm[i] = 0.;

    /*
     * Add to the force vector.
     */
    vecadd(norm,fxyz,len3-(sizex+1));
    vecadd(norm,fxyz+1,len3-(sizex+1));
    vecadd(norm,fxyz+sizex,len3-(sizex+1));
    vecadd(norm,fxyz+sizex+1,len3-(sizex+1));
}


/**************************************************************************/

void moveflag (void)
{

    vecmulc(timestep,fxyz,len3);
    vecadd(fxyz,vxyz,len3);

    vecmulc3(timestep,vxyz,dxyz,len3);
    vecadd(dxyz,pxyz,len3);
}

/**************************************************************************/

void init_sqrt (void)
{
    int i;

    fflush(stdout);
    veci(sqrtres)
        sqrt_tbl[i] = sqrt(i/sqrtscale);
}


/**************************************************************************/

void init_flag (void)
{
    int i,xc;
    double x,y;

    xc = 0;
    x = 0.;
    y = 0.;

    veccopyc(0.,vxyz,len3);

    veci(len1) {

        pxyz[xoff+i] = x;
        pxyz[yoff+i] = y;
        pxyz[zoff+i] = y*x*0.01/(sizex*sizey)+x*0.01/sizex;

        if ((++xc)>=sizex) {
            x = 0.;
            xc = 0;
            y++;
        } else
            x++;
    }

    ui.flag_release [RELEASE_TOP]    = 0;
    ui.flag_release [RELEASE_BOTTOM] = 0;
}

/**************************************************************************/

int createflag (void)
{
    auto   int     ix, iy;	/* Flag Vertex Indices */
    auto   int     index;	/* Vertex Location Index */
    auto   float   *vdata;	/* Vertex Data Pointer */
    float  r_data, g_data, b_data;
    float intensity;	/* Frequency Intensity */

    /* If the texture UV coordinate tables are nil pointers, then allocate
    ** and initialize them. */

    if (!Tu) {
	Tu = (float *) malloc (sizex * sizeof(float));
	for (ix=0;  ix < sizex;  ++ix)
	    Tu[ix] = ((float)(ix) / (float)(sizex-1));
    }

    if (!Tv) {
	Tv = (float *) malloc (sizey * sizeof(float));
	for (iy=0;  iy < sizey;  ++iy)
	    Tv[iy] = ((float)(iy) / (float)(sizey-1));
    }


#ifdef DBG_MESSAGE
        index = 0;
	for (iy=0;  iy < sizey;  ++iy) {
	    for (ix=0;  ix < sizex;  ++ix) {
        printf("flag: VINDEX(ix,iy) = %d %d %d\n",ix, iy,VINDEX(ix,iy));
        printf("flag: pxyz = %f %f %f\n",pxyz[index],
					 pxyz [index+yoff],
					 pxyz [index+zoff]);
		++index;
	    }
	}
#endif

    index = 0;

    if (ui.flag_color == COLOR_TEXTURE) {

	for (iy=0;  iy < sizey;  ++iy) {
	    for (ix=0;  ix < sizex;  ++ix) {

                Vertices[3 * VINDEX(ix,iy) + 0] = pxyz [index];
                Vertices[3 * VINDEX(ix,iy) + 1] = pxyz [index+yoff];
                Vertices[3 * VINDEX(ix,iy) + 2] = pxyz [index+zoff];

                NodeData[2 * VINDEX(ix,iy) + 0] = Tu[ix];
                NodeData[2 * VINDEX(ix,iy) + 1] = Tv[iy];

		++index;
	    }
	}

        /*
	 * Graphics Draw Call with texture
	 */
	 return (1);
    }

    index = 0;

    if (ui.flag_color == COLOR_SOLID) {

	for (iy=0;  iy < sizey;  ++iy) {
	    for (ix=0;  ix < sizex;  ++ix) {

                Vertices[3 * VINDEX(ix,iy) + 0] = pxyz [index];
                Vertices[3 * VINDEX(ix,iy) + 1] = pxyz [index+yoff];
                Vertices[3 * VINDEX(ix,iy) + 2] = pxyz [index+zoff];

		++index;
	    }
	}

        /*
	 * Graphics Draw Call, without texture, solid, no node data
	 */
	 return (1);
    }

    /* Color is flag force magnitude, unnormalized */
    index = 0;
    if (ui.flag_color == COLOR_FORCEMAG)
    {
	for (iy=0;  iy < sizey;  ++iy) {
	    for (ix=0;  ix < sizex;  ++ix) {

               intensity = fxyz[index]      * fxyz[index] +
                           fxyz[index+yoff] * fxyz[index+yoff] +
                           fxyz[index+zoff] * fxyz[index+zoff];

               NodeData[ VINDEX(ix,iy) ] = sqrt ( intensity );

               Vertices[3 * VINDEX(ix,iy) + 0] = pxyz [index];
               Vertices[3 * VINDEX(ix,iy) + 1] = pxyz [index+yoff];
               Vertices[3 * VINDEX(ix,iy) + 2] = pxyz [index+zoff];

               ++index;
	    }
	}
	 return (1);
    }


    /* Default; Color type is COLOR_PSEUDO. */

    index = 0;

    for (iy=0;  iy < sizey;  ++iy) {
	for (ix=0;  ix < sizex;  ++ix) {

            Vertices[3 * VINDEX(ix,iy) + 0] = pxyz [index];
            Vertices[3 * VINDEX(ix,iy) + 1] = pxyz [index+yoff];
            Vertices[3 * VINDEX(ix,iy) + 2] = pxyz [index+zoff];

	    ++index;

	    /* Compute the vertex color based on the difference between the
	    ** previous and current position of each vertex.  For the first
	    ** update, set the vertex color to gray. */

	    if (traversal_counter == 0) {

		r_data = 0.5;
		g_data = 0.5;
		b_data = 0.5;

	    } else {


		if (ui.flag_color == COLOR_VELOCITY)
		{
		    intensity = vxyz[index] * 1.5;
		    if (intensity < 0.0) intensity = -intensity;
		    intensity += 0.2;
		    r_data = CLAMP (intensity, 0.0, 1.0);

		    intensity = vxyz[index+yoff] * 1.5;
		    if (intensity < 0.0) intensity = -intensity;
		    intensity += 0.2;
		    g_data = CLAMP (intensity, 0.0, 1.0);

		    intensity = vxyz[index+zoff] * 1.5;
		    if (intensity < 0.0) intensity = -intensity;
		    intensity += 0.2;
		    b_data = CLAMP (intensity, 0.0, 1.0);
		}
		/* Color is flag forces */
		else if (ui.flag_color == COLOR_FORCE)
		{
		    intensity = fxyz[index] * 10.0;
		    if (intensity < 0.0) intensity = -intensity;
		    intensity += 0.2;
		    r_data = CLAMP (intensity, 0.0, 1.0);

		    intensity = fxyz[index+yoff] * 10.0;
		    if (intensity < 0.0) intensity = -intensity;
		    intensity += 0.2;
		    g_data = CLAMP (intensity, 0.0, 1.0);

		    intensity = fxyz[index+zoff] * 10.0;
		    if (intensity < 0.0) intensity = -intensity;
		    intensity += 0.2;
		    b_data = CLAMP (intensity, 0.0, 1.0);
		}
	    }
            NodeData[3 * VINDEX(ix,iy) + 0] = r_data;
            NodeData[3 * VINDEX(ix,iy) + 1] = g_data;
            NodeData[3 * VINDEX(ix,iy) + 2] = b_data;
	}
    }

    /*
     * Graphics Draw Call, without texture, RGB vector node data
     */
	 return (1);
}

/**************************************************************************/

void calc_wind()
{
    windx = sin((ui.flag_wind[1]+90)* M_PI /180)*ui.flag_wind[0];
    windz = -cos((ui.flag_wind[1]+90)* M_PI /180)*ui.flag_wind[0];
}

