/*
 * INTERNATIONAL AVS CENTRE - WARRANTY DISCLAIMER
 * Please read the file DISCLAIMER for conditions associated with this file.
 * avs@iavsc.org, www.iavsc.org
 */

/* ----------------------------------------------------------------------
 * Note:
 *   The gen.h include file is generated by Express when the module is 
 *   compiled. It avoids including "user.h" or "express.h" directly, so 
 *   that the module may be moved across the processes by changing the V
 *   properties in the library.
 * ----------------------------------------------------------------------
 */

#include "iac_proj/rd_group/gen.h"
#include <avs/om_type.h>

// ------------------------------------------------------------------------- //
char*
ReadGroup_ReadGroup::ReadString (FILE *fp)
{
  // get the string's length
  int stringLen;
  if (fread (&stringLen, sizeof stringLen, 1, fp) != 1)
    return 0;

  // allocate the buffer in which to store the string and read the string
  char *string = (char*) malloc (stringLen);
  if (fread (string, sizeof *string, stringLen, fp) != stringLen) {
    free (string);
    return 0;
  }

  // successfull completion
  return string;
}

// ------------------------------------------------------------------------- //
int
ReadGroup_ReadGroup::DeserializeGroup (OMobj_id groupId,
				   FILE *fp)
{
  // get the object at the end of chain of references
  (void) OMget_obj_val (groupId, &groupId);

  // get the number of subobjects stored in the file
  int numSubObjs;
  if (fread (&numSubObjs, sizeof numSubObjs, 1, fp) != 1)
    return OM_STAT_FAILURE;

  // iterate through all the subobjects stored in the file
  for (int subObjIndex = 0; subObjIndex < numSubObjs; subObjIndex++) {
    // deserialize the subobject
    if (Deserialize (groupId, fp) != OM_STAT_SUCCESS)
      return OM_STAT_FAILURE;
  }

  // successfull completion
  return OM_STAT_SUCCESS;
}

// ------------------------------------------------------------------------- //
int
ReadGroup_ReadGroup::Deserialize (OMobj_id groupId,
			      FILE *fp)
{
  // deserialize the objects name
  const char *name = ReadString (fp);
  if (!name)
    return OM_STAT_FAILURE;
  
  // deserialize the objects type
  int storedType;
  if (fread (&storedType, sizeof storedType, 1, fp) != 1)
    return OM_STAT_FAILURE;
  
  // deserialize the object's dimensionality
  int storedNumDim;
  int storedDims[OM_ARRAY_MAXDIM];
  (void) memset (storedDims, 0, sizeof storedDims);
  if (fread (&storedNumDim, sizeof storedNumDim, 1, fp) != 1 ||
      fread (storedDims, sizeof *storedDims, storedNumDim, fp) != storedNumDim)
    return OM_STAT_FAILURE;
  
  // locate the subobject, gets its data type, dimensionality and dimensions
  OMobj_id objId =
    OMfind_subobj (groupId, OMstr_to_name ((char*) name), OM_OBJ_RW);
  
  int dataType;
  (void) OMget_data_type (objId, &dataType);
  
  int numDim;
  int dims[OM_ARRAY_MAXDIM] = { 0, 0, 0, 0 };
  (void) OMget_array_dims (objId, &numDim, dims);
  
  // check that the object exists, is of the correct type and has
  // equal dimensionality as well as dimensions
  if (OMis_null_obj (objId))
    return OM_STAT_FAILURE;
  
  if ((storedType == OM_TYPE_GROUP &&
       !OMobj_is_type_of (objId, OM_type_group)) ||
      (storedType != OM_TYPE_GROUP && dataType != storedType &&
       !OMobj_is_type_of (objId, OM_type_prim)))
    return OM_STAT_FAILURE;
  
  int index = 0;
  if (numDim == storedNumDim)
    if (numDim == 1) {
      if (dims[0] != storedDims[0])
	(void) OMset_array_size (objId, storedDims[0]);
      index = numDim;
    } else
      for (; index < numDim; index++)
	if (dims[index] != storedDims[index])
	  break;
  if (index < numDim)
    return OM_STAT_FAILURE;
  
  // the stored and existing object is compatible, now load the
  // value(s) and store them in the object

  // is the value null?
  if (storedType != OM_TYPE_GROUP) {
    bool isNull;
    if (fread (&isNull, sizeof isNull, 1, fp) != 1)
      return OM_STAT_FAILURE;
    if (isNull) {
      (void) OMset_obj_val (objId, OMnull_obj, 0);
      return OM_STAT_SUCCESS;
    }
  }
  
  // calculate the product of the dimensions
  int arraySize = 1;
  if (numDim)
    for (int index = 0; index < numDim; index++)
      arraySize *= storedDims[index];
  
  // group's are recursive and have no value
  if (storedType == OM_TYPE_GROUP) {
    if (numDim)
      for (index = 0; index < arraySize; index++) {
	OMobj_id subObjId;
	(void) OMget_array_val (objId, index, &subObjId, OM_OBJ_RD);
	if (DeserializeGroup (subObjId, fp) != OM_STAT_SUCCESS)
	  return OM_STAT_FAILURE;
      }
    else
      if (DeserializeGroup (objId, fp) != OM_STAT_SUCCESS)
	return OM_STAT_FAILURE;
    return OM_STAT_SUCCESS;
  }
  
  // strings are handled separately from the primitives
  if (storedType == OM_TYPE_STRING) {
    if (numDim)
      for (index = 0; index < arraySize; index++) {
	char *string = ReadString (fp);
	if (!string)
	  return OM_STAT_FAILURE;
	(void) OMset_str_array_val (objId, index, string);
	free (string);
      }
    else {
      char *string = ReadString (fp);
      if (!string)
	return OM_STAT_FAILURE;
      (void) OMset_str_val (objId, string);
    }
    return OM_STAT_SUCCESS;
  }
    
  // handle primitive arrays genericly
  if (numDim) {
    static const size_t typeSize[] = {
      sizeof (unsigned char), sizeof (char), sizeof (short), sizeof (int),
      sizeof (float), sizeof (double)
    };

    void *array =
      OMret_typed_array_ptr (objId, OM_GET_ARRAY_WR, storedType, 0);
    if (fread (array, typeSize[storedType], arraySize, fp) != arraySize) {
      ARRfree (array);
      return OM_STAT_FAILURE;
    }
    ARRfree (array);
    return OM_STAT_SUCCESS;
  }

  //
  switch (storedType) {
  case OM_TYPE_BYTE: {
    unsigned char value;
    if (fread (&value, sizeof value, 1, fp) != 1)
      return OM_STAT_FAILURE;
    (void) OMset_int_val (objId, value);
    break;
  }

  case OM_TYPE_CHAR: {
    char value;
    if (fread (&value, sizeof value, 1, fp) != 1)
      return OM_STAT_FAILURE;
    (void) OMset_int_val (objId, value);
    break;
  }

  case OM_TYPE_SHORT: {
    short value;
    if (fread (&value, sizeof value, 1, fp) != 1)
      return OM_STAT_FAILURE;
    (void) OMset_int_val (objId, value);
    break;
  }

  case OM_TYPE_INT: {
    int value;
    if (fread (&value, sizeof value, 1, fp) != 1)
      return OM_STAT_FAILURE;
    (void) OMset_int_val (objId, value);
    break;
  }

  case OM_TYPE_FLOAT: {
    float value;
    if (fread (&value, sizeof value, 1, fp) != 1)
      return OM_STAT_FAILURE;
    (void) OMset_real_val (objId, value);
    break;
  }

  case OM_TYPE_DOUBLE: {
    double value;
    if (fread (&value, sizeof value, 1, fp) != 1)
      return OM_STAT_FAILURE;
    (void) OMset_real_val (objId, value);
    break;
  }

  default:
    return OM_STAT_FAILURE;
  }

  return OM_STAT_SUCCESS;
}

// ------------------------------------------------------------------------- //
int
ReadGroup_ReadGroup::Read (OMevent_mask event_mask,
		       int seq_num)
{
  // open the file
  char *filename = this->filename;
  FILE *fp;
  if ((fp = fopen (filename, "rb")) == NULL) {
    ERRverror ("ReadGroup::Read", ERR_ERROR,
	       "Unable to open '%s' for writing", filename);
    free (filename);
    return OM_STAT_FAILURE;
  }
  free (filename);

  // serialize the 'group'
  int retval = DeserializeGroup (group, fp);
  if (retval != OM_STAT_SUCCESS)
    ERRverror ("ReadGroup_ReadGroup::Read", ERR_ERROR,
	       "Error de-serializing group");

  // cleanup
  (void) fclose (fp);

  // return the status code
  return retval;
}
