/*
 * INTERNATIONAL AVS CENTRE - WARRANTY DISCLAIMER
 * Please read the file DISCLAIMER for conditions associated with this file.
 * avs@iavsc.org, www.iavsc.org
 */

/* ----------------------------------------------------------------------
 * SphereSurface Module
 * ----------------------------------------------------------------------
 * Description:
 *   Generates a 2D sphere surface coordinates grid and mesh in 3D space,
 *   using a 2D structured grid, as opposed to the AVS 3D field sphere
 *   mesh->bounds method.
 *   This approach also removes the 'edge' found in the standard sphere,
 *   and thus appears to be a continuous surface (other modules use this
 *   additional connectivity information to produce seemless output).
 *
 * Author:
 *   Paul G. Lever - 16th March 1998
 *
 * Revision: 
 *
 * ----------------------------------------------------------------------
 * Note:
 *   The gen.h include file is generated by Express when the module is 
 *   compiled. It avoids including "user.h" or "express.h" directly, so 
 *   that the module may be moved across the processes by changing the V
 *   properties in the library.
 * ----------------------------------------------------------------------
 */

#define PI 3.14159
#define DIV 180.0*PI

#include <math.h>
#include "iac_proj/spsurf/gen.h"

int
sphere_surface_gen(OMobj_id SphereSurfaceGen_id, OMevent_mask event_mask, int seq_num)
{
  /***********************/
  /*  Declare variables  */
  /***********************/
  
  OMobj_id mesh_id;
  int mesh_ndim, *mesh_dims, mesh_nspace, mesh_nnodes;
  float *mesh_coord;
  
  int  mesh_nsets, mesh_set_count, mesh_ncells;
  int  mesh_cell_nnodes, *mesh_node_connect;
  OMobj_id mesh_cell_set;
  
  int  longdim;
  int  latdim;
  double dend_lat;
  double dstart_lat;
  double dend_long;
  double dstart_long;
  double radius;
  int closedlong;
  int closedlat;
  
  double end_lat, start_lat, end_long, start_long;
  
  double interval, lambda, mu;
  double *sinl, *sinu, *cosl, *cosu;
  int longi, lati;

  int index=0;
  
  double i;
  
  /***********************/
  /*  Get input values   */
  /***********************/
  
  /* Get longdim's value */ 
  
  if( OMget_name_int_val( SphereSurfaceGen_id,
			  OMstr_to_name("longdim"),
			  &longdim ) != 1) 
    longdim = 0;

  /* Get end_lat's value */
  
  if( OMget_name_real_val( SphereSurfaceGen_id,
			   OMstr_to_name("end_lat"),
			   &dend_lat ) != 1)
    dend_lat = 0.0;

  /* Get start_lat's value */
  
  if( OMget_name_real_val( SphereSurfaceGen_id,
			   OMstr_to_name("start_lat"),
			   &dstart_lat ) != 1)
    dstart_lat = 0.0;

  /* Get end_long's value */
  
  if( OMget_name_real_val( SphereSurfaceGen_id,
			   OMstr_to_name("end_long"),
			   &dend_long ) != 1)
    dend_long = 0.0;

  /* Get start_long's value */
  
  if( OMget_name_real_val( SphereSurfaceGen_id,
			   OMstr_to_name("start_long"),
			   &dstart_long ) != 1)
    dstart_long = 0.0;

  /* Get radius's value */
  
  if( OMget_name_real_val( SphereSurfaceGen_id,
			   OMstr_to_name("radius"),
			   &radius ) != 1)
    radius = 1.0;

  /* Get latdim's value */ 
  
  if( OMget_name_int_val( SphereSurfaceGen_id,
			  OMstr_to_name("latdim"),
			  &latdim ) != 1) 
    latdim = 0;

  /* Get closedlong's value */ 
  
  if( OMget_name_int_val( SphereSurfaceGen_id,
			  OMstr_to_name("closedlong"),
			  &closedlong ) != 1)
    closedlong = 0;

  /* Get closedlat's value */ 
  
  if( OMget_name_int_val( SphereSurfaceGen_id,
			  OMstr_to_name("closedlat"),
			  &closedlat ) != 1)
    closedlat = 0;

  
  
  /***********************/
  /* Function's Body     */
  /***********************/
  
  /* convert degrees to radians	 */
  
  start_long = dstart_long/DIV;
  end_long = dend_long/DIV;
  
  start_lat = dstart_lat/DIV;
  end_lat = dend_lat/DIV;
  
  /* Generate LUTs	 */
  
  /* generate the lambda sin/cos LUTs	 */

  interval = (end_long - start_long)/(double)longdim;
  
  sinl = (double *)ARRalloc(NULL, DTYPE_DOUBLE, longdim, NULL );
  sinu = (double *)ARRalloc(NULL, DTYPE_DOUBLE, latdim, NULL );
  cosl = (double *)ARRalloc(NULL, DTYPE_DOUBLE, longdim, NULL );
  cosu = (double *)ARRalloc(NULL, DTYPE_DOUBLE, latdim, NULL );
  
  for( longi=0; longi<longdim; longi++ ) {
    lambda = start_long + interval*(double)longi;

    sinl[longi] = (double) sin(lambda);
    cosl[longi] = (double) cos(lambda);
  }

  /* generate the mu sin/cos LUTs	 */

  interval = (end_lat - start_lat)/(double)(latdim-1);

  for( lati=0; lati<latdim; lati++ ) {
    mu = start_lat + interval*(double)lati;

    sinu[lati] = (double) sin(mu);
    cosu[lati] = (double) cos(mu);
  }

  /***********************/
  /*  Set output values  */
  /***********************/
  
  /* Set mesh structured mesh */

  /* Get mesh id */
  
  mesh_id = OMfind_subobj( SphereSurfaceGen_id,
			   OMstr_to_name("mesh"),
			   OM_OBJ_RW );

  /* Set mesh dimensionality, mesh_ndim can be 1,2 or 3 */
  
  mesh_ndim = 2;
  
  FLDset_ndim( mesh_id, mesh_ndim );

  /* Set mesh dims array */
  
  mesh_dims = (int *)ARRalloc(NULL, DTYPE_INT, mesh_ndim, NULL );

  /*** fill in dims array with your values ***/
  
  mesh_dims[0] = longdim;
  mesh_dims[1] = latdim;
  
  FLDset_dims( mesh_id, mesh_dims );
  
  if( mesh_dims )
    ARRfree( (char *)mesh_dims );

   /* Set mesh nspace, mesh_nspace can be 1,2 or 3 */
  
  mesh_nspace = 3;
  
  FLDset_nspace (mesh_id, mesh_nspace );

  /* Set mesh coordinates */
  /* first allocate mesh_coord array */
  
  FLDget_nnodes( mesh_id, &mesh_nnodes );
   
  mesh_coord = (float *)ARRalloc( NULL, DTYPE_FLOAT, 
				  mesh_nspace*mesh_nnodes, NULL );

  /* generate the grid coordinates array	 */
  
  for( lati=0; lati<latdim; lati++ ) {
    for( longi=0; longi<longdim; longi++ ) {
      mesh_coord[index++] = (float)(radius * cosu[lati] * sinl[longi]);
      mesh_coord[index++] = (float)(radius * sinu[lati]);
      mesh_coord[index++] = (float)(radius * cosu[lati] * cosl[longi]);
    }
  }
  
  if( sinl )
    ARRfree( (char *)sinl );
  if( sinu )
    ARRfree( (char *)sinu );
  if( cosl )
    ARRfree( (char *)cosl );
  if( cosu )
    ARRfree( (char *)cosu );

  /*** fill in mesh_coord array with X[,Y,Z] values at each node ***/
  
  FLDset_coord( mesh_id, mesh_coord, mesh_nspace*mesh_nnodes,
		OM_SET_ARRAY_FREE );
  
  /* Generate the cell set	 */
  
  FLDset_ncell_sets( mesh_id, 0 );
  
  mesh_nsets = 1;
  
  FLDadd_cell_set( mesh_id, "Quad" );
  
  /* Get cell set id */

  FLDget_cell_set( mesh_id, 0, &mesh_cell_set );

  /* Set number of cells */
  
  mesh_ncells = (longdim+closedlong-1) * (latdim+closedlat-1);
  
  FLDset_ncells( mesh_cell_set, mesh_ncells );

  /* Set node connectivity list */
  /* First allocate mesh_node_connect */

  FLDget_cell_set_nnodes( mesh_cell_set,  &mesh_cell_nnodes);
  mesh_node_connect = (int *)ARRalloc( NULL, DTYPE_INT,
				       mesh_ncells*mesh_cell_nnodes, NULL);
  
  /*** fill in  mesh_node_connect array with node indecies for each cell ***/
  
  index = 0;
  for( longi=0; longi<longdim+closedlong-1; longi++ ) {
    for( lati=0; lati<latdim+closedlat-1; lati++ ) {
      
      mesh_node_connect[index++] = longi+(lati*longdim);
      mesh_node_connect[index++] = longi+(((lati+1)%latdim)*longdim);
      mesh_node_connect[index++] = ((longi+1)%longdim)+(((lati+1)%latdim)*longdim);
      mesh_node_connect[index++] = ((longi+1)%longdim)+(lati*longdim);
    }
  }
  
  FLDset_node_connect( mesh_cell_set, mesh_node_connect,
		       mesh_ncells*mesh_cell_nnodes, OM_SET_ARRAY_FREE);


  return(1);
}

