/*
 * INTERNATIONAL AVS CENTRE - WARRANTY DISCLAIMER
 * Please read the file DISCLAIMER for conditions associated with this file.
 * avs@iavsc.org, www.iavsc.org
 */

/* ----------------------------------------------------------------------
 * Note:
 *   The gen.h include file is generated by Express when the module is 
 *   compiled. It avoids including "user.h" or "express.h" directly, so 
 *   that the module may be moved across the processes by changing the V
 *   properties in the library.
 * ----------------------------------------------------------------------
 */

#include "iac_proj/wr_group/gen.h"
#include <avs/om_type.h>

// ------------------------------------------------------------------------- //
int
WriteGroup_WriteGroup::WriteString (const char *string,
			       FILE *fp)
{
  int stringLen = strlen (string) + 1;
  if (fwrite (&stringLen, sizeof stringLen, 1, fp) != 1 ||
      fwrite (string, sizeof *string, stringLen, fp) != stringLen)
    return OM_STAT_FAILURE;
  return OM_STAT_SUCCESS;
}

// ------------------------------------------------------------------------- //
int
WriteGroup_WriteGroup::SerializeGroup (OMobj_id groupId,
				  OMobj_id templateId,
				  FILE *fp)
{
  // obtain the object at the end of the chain
  OMobj_id objId;
  (void) OMget_obj_val (groupId, &objId);

  // get and serialize the objects's subobject count
  int numSubObjs;
  (void) OMget_num_subobjs (templateId, &numSubObjs);
  if (fwrite (&numSubObjs, sizeof numSubObjs, 1, fp) != 1)
    return OM_STAT_FAILURE;

  // iterate through all the subobjects of the 'template'
  for (int subObjIndex = 0; subObjIndex < numSubObjs; subObjIndex++) {
    // obtain the name of the subobject
    OMobj_id subTemplId;
    (void) OMget_array_subobj (templateId, subObjIndex, &subTemplId,
			       OM_OBJ_RD);
    OMobj_name name;
    (void) OMget_obj_name (subTemplId, &name);

    // now find the equivalent subobject at the end of the chain
    OMobj_id subObjId = OMfind_subobj (objId, name, OM_OBJ_RD);

    // serialize the subobject
    if (Serialize (subObjId, subTemplId, fp) != OM_STAT_SUCCESS)
      return OM_STAT_FAILURE;
  }

  // successfull completion
  return OM_STAT_SUCCESS;
}

// ------------------------------------------------------------------------- //
int
WriteGroup_WriteGroup::Serialize (OMobj_id objId,
			     OMobj_id templateId,
			     FILE *fp)
{
  // get and serialize the object's name
  const char *name = OMret_obj_name (objId);
  if (WriteString (name, fp) != OM_STAT_SUCCESS)
    return OM_STAT_FAILURE;

  // get and serialize the object's type
  int dataType;
  if (OMobj_is_type_of (objId, OM_type_group))
    dataType = OM_TYPE_GROUP;
  else
    (void) OMget_data_type (objId, &dataType);
  if (fwrite (&dataType, sizeof dataType, 1, fp) != 1)
    return OM_STAT_FAILURE;

  // get and serialize the object's dimensionality and dimensions
  int numDim;
  int dims[OM_ARRAY_MAXDIM];
  (void) memset (dims, 0, sizeof dims);
  (void) OMget_array_dims (objId, &numDim, dims);
  if (fwrite (&numDim, sizeof numDim, 1, fp) != 1 ||
      fwrite (dims, sizeof *dims, numDim, fp) != numDim)
    return OM_STAT_FAILURE;

  // is the value the null object
  if (dataType != OM_TYPE_GROUP) {
    bool isNull = !OMvalid_obj (objId, OMnull_obj, 0);
    if (fwrite (&isNull, sizeof isNull, 1, fp) != 1)
      return OM_STAT_FAILURE;
    if (isNull)
      return OM_STAT_SUCCESS;
  }

  // calculate the product of the dimensions
  int arraySize = 1;
  if (numDim)
    for (int index = 0; index < numDim; index++)
      arraySize *= dims[index];

  //
  switch (dataType) {
  case OM_TYPE_GROUP:
    if (numDim)
      for (int index = 0; index < arraySize; index++) {
	OMobj_id subObjId;
	(void) OMget_array_val (objId, index, &subObjId, OM_OBJ_RD);
	if (SerializeGroup (subObjId, templateId, fp) != OM_STAT_SUCCESS)
	  return OM_STAT_FAILURE;
      }
    else
      if (SerializeGroup (objId, templateId, fp) != OM_STAT_SUCCESS)
	return OM_STAT_FAILURE;
    break;

  case OM_TYPE_BYTE:
    if (numDim) {
      unsigned char *values =
	(unsigned char*) OMret_array_ptr (objId, OM_GET_ARRAY_RD, 0, 0);
      if (fwrite (values, sizeof *values, arraySize, fp) != arraySize) {
	ARRfree (values);
        return OM_STAT_FAILURE;
      }
      ARRfree (values);
    } else {
      int intValue;
      (void) OMget_int_val (objId, &intValue);
      unsigned char value = (unsigned char) intValue;
      if (fwrite (&value, sizeof value, 1, fp) != 1)
        return OM_STAT_FAILURE;
    }
    break;

  case OM_TYPE_CHAR:
    if (numDim) {
      char *values = (char*) OMret_array_ptr (objId, OM_GET_ARRAY_RD, 0, 0);
      if (fwrite (values, sizeof *values, arraySize, fp) != arraySize) {
	ARRfree (values);
        return OM_STAT_FAILURE;
      }
      ARRfree (values);
    } else {
      int intValue;
      (void) OMget_int_val (objId, &intValue);
      char value = (char) intValue;
      if (fwrite (&value, sizeof value, 1, fp) != 1)
        return OM_STAT_FAILURE;
    }
    break;

  case OM_TYPE_SHORT:
    if (numDim) {
      short *values = (short*) OMret_array_ptr (objId, OM_GET_ARRAY_RD, 0, 0);
      if (fwrite (values, sizeof *values, arraySize, fp) != arraySize) {
	ARRfree (values);
        return OM_STAT_FAILURE;
      }
      ARRfree (values);
    } else {
      int intValue;
      (void) OMget_int_val (objId, &intValue);
      short value = (short) intValue;
      if (fwrite (&value, sizeof value, 1, fp) != 1)
        return OM_STAT_FAILURE;
    }
    break;

  case OM_TYPE_INT:
    if (numDim) {
      int *values = (int*) OMret_array_ptr (objId, OM_GET_ARRAY_RD, 0, 0);
      if (fwrite (values, sizeof *values, arraySize, fp) != arraySize) {
	ARRfree (values);
        return OM_STAT_FAILURE;
      }
      ARRfree (values);
    } else {
      int value;
      (void) OMget_int_val (objId, &value);
      if (fwrite (&value, sizeof value, 1, fp) != 1)
        return OM_STAT_FAILURE;
    }
    break;

  case OM_TYPE_FLOAT:
    if (numDim) {
      float *values = (float*) OMret_array_ptr (objId, OM_GET_ARRAY_RD, 0, 0);
      if (fwrite (values, sizeof *values, arraySize, fp) != arraySize) {
	ARRfree (values);
        return OM_STAT_FAILURE;
      }
      ARRfree (values);
    } else {
      double realValue;
      (void) OMget_real_val (objId, &realValue);
      float value = (float) realValue;
      if (fwrite (&value, sizeof value, 1, fp) != 1)
        return OM_STAT_FAILURE;
    }
    break;

  case OM_TYPE_DOUBLE:
    if (numDim) {
      double *values =
	(double*) OMret_array_ptr (objId, OM_GET_ARRAY_RD, 0, 0);
      if (fwrite (values, sizeof *values, arraySize, fp) != arraySize) {
	ARRfree (values);
        return OM_STAT_FAILURE;
      }
      ARRfree (values);
    } else {
      double value;
      (void) OMget_real_val (objId, &value);
      if (fwrite (&value, sizeof value, 1, fp) != 1)
        return OM_STAT_FAILURE;
    }
    break;

  case OM_TYPE_STRING:
    if (numDim) {
      for (int index = 0; index < arraySize; index++) {
	char *string = 0;
	(void) OMget_str_array_val (objId, index, &string, 0);
	if (WriteString (string, fp) != OM_STAT_SUCCESS) {
	  free (string);
	  return OM_STAT_FAILURE;
	}
	free (string);
      }
    } else {
      char *string = 0;
      (void) OMget_str_val (objId, &string, 0);
      if (WriteString (string, fp) != OM_STAT_SUCCESS) {
	free (string);
	return OM_STAT_FAILURE;
      }
      free (string);
    }
    break;

  default:
    return OM_STAT_FAILURE;
  }

  // successfull completion
  return OM_STAT_SUCCESS;
}

// ------------------------------------------------------------------------- //
int
WriteGroup_WriteGroup::Write (OMevent_mask event_mask,
			 int seq_num)
{
  // open the file
  char *filename = this->filename;
  FILE *fp;
  if ((fp = fopen (filename, "wb")) == NULL) {
    ERRverror ("WriteGroup::Write", ERR_ERROR,
	       "Unable to open '%s' for writing", filename);
    free (filename);
    return OM_STAT_FAILURE;
  }
  free (filename);

  // serialize the 'group'
  OMobj_id refId;
  (void) OMget_obj_ref (group, &refId, 0);
  int retval = SerializeGroup (refId, refId, fp);
  if (retval != OM_STAT_SUCCESS)
    ERRverror ("WriteGroup_WriteGroup::Write", ERR_ERROR,
	       "Error serializing group");

  // cleanup
  (void) fclose (fp);

  // return the status code
  return retval;
}

