/*
    GNU Gama -- adjudstment of geodetic networks
    Copyright (C) 2010  Ales Cepek <cepek@gnu.org>, 2010 Jiri Novak
    <jiri.novak@petriny.net>

    This file is part of the GNU Gama C++ library.

    This library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  $
*/

#include "gkf2sql.h"
#include <sstream>

using namespace GNU_gama::local;

void Gkf2sql::run(std::istream& istr, std::ostream& ostr)
{
  try 
    {
      char c;
      int  n, finish = 0;
      std::string line;
      do 
	{
	  line = "";
	  n     = 0;
	  while (istr.get(c))
	    {
	      line += c;
	      n++;
	      if (c == '\n') break;
	    }
	  if (!istr) finish = 1;
          
	  gkfparser->xml_parse(line.c_str(), n, finish);
	}
      while (!finish);
  
      write(ostr);
      ostr.flush();
    }
  catch (...) 
    {
      throw;
    }  
}


void Gkf2sql::write(std::ostream& ostr)
{
  ostr.precision(17);

  ostr << "/* generated by gama-local-gkf2sql, configuration: " + config + "\n"
       << " */\n"
       << "begin;\n\n";     // begin transaction
 
  ostr << "insert into gnu_gama_local_configurations (conf_id, conf_name) values ("
       << "(select new_id from (select coalesce(max(conf_id), 0)+1 as new_id from gnu_gama_local_configurations)x),"
       << " '" + config +"');\n";


  /* <parameters> atributes */
  {
    if (gkfparser->m0_apr != 10.0)
    ostr << "update gnu_gama_local_configurations  set sigma_apr = "
	 << gkfparser->m0_apr << " where conf_id = " << cnfg() << ";\n";
    if (gkfparser->konf_pr != 0.95)
    ostr << "update gnu_gama_local_configurations set conf_pr = "
	 << gkfparser->konf_pr << " where conf_id = " << cnfg() << ";\n";
    if (gkfparser->tol_abs != 1000.0)
    ostr << "update gnu_gama_local_configurations set tol_abs = "
	 << gkfparser->tol_abs << " where conf_id = " << cnfg() << ";\n";
    if (gkfparser->update_constr)
    ostr << "update gnu_gama_local_configurations set update_cc = "
	 << (gkfparser->update_constr ? "'yes'" : "'no'") 
         << " where conf_id = " << cnfg() << ";\n";
    if (gkfparser->typ_m0_apriorni)
    ostr << "update gnu_gama_local_configurations set sigma_act = "
	 << (gkfparser->typ_m0_apriorni ? "'apriori'" : "'aposteriori'") 
         << " where conf_id = " << cnfg() << ";\n";
  }


  /* <network> atributes */
  {
    // EN= 1, NW= 2, SE= 4, WS=  8,        plane right-handed systems
    // NE=16, SW=32, ES=64, WN=128,        plane left-handed  systems
    
    std::string lcs="'ne'";
    switch (points.local_coordinate_system)
      {
      case   1: lcs = "'en'"; break;
      case   2: lcs = "'nw'"; break;
      case   4: lcs = "'se'"; break;
      case   8: lcs = "'ws'"; break;
      case  16: lcs = "'ne'"; break;
      case  32: lcs = "'sw'"; break;
      case  64: lcs = "'es'"; break;
      case 128: lcs = "'wn'"; break;
      default:  lcs = "'ne'";
      }
    if (lcs != "'ne'")
      ostr << "update gnu_gama_local_configurations set axes_xy = "
	   << lcs << " where conf_id = " << cnfg() << ";\n";
    
    // if (points.right_handed_angles())
      ostr << "update gnu_gama_local_configurations set angles = "
	   << (points.right_handed_angles() ? "'right-handed'" : "'left-handed'") 
	   << " where conf_id = " << cnfg() << ";\n";

    if (gkfparser->epoch)
      ostr << "update gnu_gama_local_configurations set epoch = "
	   << gkfparser->epoch << " where conf_id = " << cnfg() << ";\n";
  }
  

  /* <points-observations> atributes */
  // {
  //   double da = gkfparser->implicit_stdev_distance_a();
  //   double db = gkfparser->implicit_stdev_distance_b();
  //   double dc = gkfparser->implicit_stdev_distance_c();
  //   if (da + db != 0)
  //     {
  // 	ostr << "insert into gnu_gama_local_atributes (conf_id, atribute, tag, value) values ("
  // 	     << cnfg() << ", 'distance-stdev', 'points-observations', '" << da; 
  // 	if (db)
  // 	  ostr << " " << db << " " << dc;
  // 	ostr << "');\n";
  //     }
  // 
  //   if (double dir=gkfparser->implicit_stdev_direction())
  //     {
  // 	ostr << "insert into gnu_gama_local_atributes (conf_id, atribute, tag, value) values ("
  // 	     << cnfg() << ", 'direction-stdev', 'points-observations', '" << dir << "');\n";
  //     }
  // 
  //   if (double angle=gkfparser->implicit_stdev_angle())
  //     {
  // 	ostr << "insert into gnu_gama_local_atributes (conf_id, atribute, tag, value) values ("
  // 	     << cnfg() << ", 'angle-stdev', 'points-observations', '" << angle << "');\n";
  //     }
  // 
  //     if (double zangle=gkfparser->implicit_stdev_zangle())
  //     {
  // 	ostr << "insert into gnu_gama_local_atributes (conf_id, atribute, tag, value) values ("
  // 	     << cnfg() << ", 'zenith-angle-stdev', 'points-observations', '" << zangle << "');\n";
  //     }
  // }


  /* <description> */
  if (!gkfparser->description.empty())
    {
      const int N = 1000;  // text varchar('N') in gnu_gama_local_descriptions table;
      int indx = 0;
      while (indx*N < gkfparser->description.length())
	{
	  const std::string& s = gkfparser->description.substr(indx*N, N);
	  std::string description;
          for (std::string::const_iterator i=s.begin(); i!=s.end(); ++i)
	    if (*i == '\'')
	      description += "''";
	    else
	      description += *i;
	  ostr << "insert into gnu_gama_local_descriptions (conf_id, indx, text) values ("
	       << cnfg() << ", " << (indx+1) << ", '" << description << "');\n";
	  indx++;
	}
    }


  /* <points-observations><point /> */
  {
    using namespace GNU_gama::local;
    ostr << "\n";
    for (PointData::const_iterator i=points.begin(); i!=points.end(); ++i)
      {
	const PointID&    id = i->first;
	const LocalPoint& pt = i->second;

	std::string atr = "conf_id, id";
	std::ostringstream val;
	val.setf(std::ios_base::fixed, std::ios_base::floatfield);
	val.precision(17);
	val << cnfg() << ", '" << id << "'";

	if (pt.test_xy())
	  {
	    atr += ", x, y";
	    val << ", " << pt.x() << ", " << pt.y();
	  }

	if (pt.test_z())
	  {
	    atr += ", z";
	    val << ", " << pt.z();
	  }

	if (pt.active_xy())
	  {
	    atr += ", txy";
	    if      (pt.fixed_xy()      )  val << ", 'fixed'";
            else if (pt.constrained_xy())  val << ", 'constrained'";
	    else if (pt.free_xy()       )  val << ", 'adjusted'";
	  }

	if (pt.active_z())
	  {
	    atr += ", tz";
	    if      (pt.fixed_z()      )  val << ", 'fixed'";
            else if (pt.constrained_z())  val << ", 'constrained'";
	    else if (pt.free_z()       )  val << ", 'adjusted'";
	  }

	ostr << "insert into gnu_gama_local_points (" << atr << ") "
	     << "values (" << val.str() << ");\n";
      }  
  }
  

  /* clusters */
  {
    using namespace GNU_gama::local;
    
    int cluster = 1;
    for (GNU_gama::local::ObservationData::ClusterList::const_iterator
	   i=observations.clusters.begin(); i!=observations.clusters.end(); ++i)
      {
	const Cluster* c = *i;
	if (const StandPoint* sp = dynamic_cast<const StandPoint*>(c))
	  {
	    /* xml <obs> atributes from, orientation and from_dh,
	     * defined in gama-local.dtd, are ignored in database
	     * schema (from_dh is not even implemented in class
	     * StandPoint)
	     */
	    write_cluster(ostr, c, cluster, "obs");

	    Index index = 1;
	    for (ObservationList::const_iterator 
		   b = c->observation_list.begin(), 
		   e = c->observation_list.end();  b != e;  ++b)
	      {	
		// tag, from_id, to_id, to_id2, val, stdev, from_dh, to_dh, to_dh2, dist
		if      (const Distance*   m = dynamic_cast<const Distance*  >(*b)) 
                {
		  ostr << "insert into gnu_gama_local_obs "
		       << "(conf_id, ccluster, indx, tag, from_id, to_id, "
		       << "val, from_dh, to_dh, rejected) values (" 
                       << cnfg() << ", " << cluster << ", "
		       << index++ << ", 'distance', '"
		       << m->from() << "', '" << m->to() << "', "
		       << m->value();
		  if (m->from_dh()) ostr << ", " << m->from_dh(); else ostr << ", null";
		  if (m->to_dh()  ) ostr << ", " << m->to_dh();   else ostr << ", null";
		  ostr << ", " << rejected(m) << ");\n";
                }
		else if (const Direction*  m = dynamic_cast<const Direction* >(*b))
		{
		  ostr << "insert into gnu_gama_local_obs "
		       << "(conf_id, ccluster, indx, tag, from_id, to_id, "
		       << "val, from_dh, to_dh, rejected) values (" 
                       << cnfg() << ", " << cluster << ", "
		       << index++ << ", 'direction', '"
		       << m->from() << "', '" << m->to() << "', "
		       << m->value();
		  if (m->from_dh()) ostr << ", " << m->from_dh(); else ostr << ", null";
		  if (m->to_dh()  ) ostr << ", " << m->to_dh();   else ostr << ", null";
		  ostr  << ", " << rejected(m) << ");\n";
		}
		else if (const S_Distance* m = dynamic_cast<const S_Distance*>(*b))
		{
		  ostr << "insert into gnu_gama_local_obs "
		       << "(conf_id, ccluster, indx, tag, from_id, to_id, "
		       << "val, from_dh, to_dh, rejected) values (" 
                       << cnfg() << ", " << cluster << ", "
		       << index++ << ", 's-distance', '"
		       << m->from() << "', '" << m->to() << "', "
		       << m->value();
		  if (m->from_dh()) ostr << ", " << m->from_dh(); else ostr << ", null";
		  if (m->to_dh()  ) ostr << ", " << m->to_dh();   else ostr << ", null";
		  ostr  << ", " << rejected(m) << ");\n";
		}
		else if (const Z_Angle*    m = dynamic_cast<const Z_Angle*   >(*b))
		{
		  ostr << "insert into gnu_gama_local_obs "
		       << "(conf_id, ccluster, indx, tag, from_id, to_id, "
		       << "val, from_dh, to_dh, rejected) values (" 
                       << cnfg() << ", " << cluster << ", "
		       << index++ << ", 'z-angle', '"
		       << m->from() << "', '" << m->to() << "', "
		       << m->value();
		  if (m->from_dh()) ostr << ", " << m->from_dh(); else ostr << ", null";
		  if (m->to_dh()  ) ostr << ", " << m->to_dh();   else ostr << ", null";
		  ostr  << ", " << rejected(m) << ");\n";
		}
		else if (const Angle*      m = dynamic_cast<const Angle*     >(*b)) 
		  {
		    ostr << "insert into gnu_gama_local_obs "
			 << "(conf_id, ccluster, indx, tag, from_id, to_id, to_id2, "
			 << "val, from_dh, to_dh, to_dh2, rejected) values (" 
			 << cnfg() << ", " << cluster << ", "
			 << index++ << ", 'angle', '"
			 << m->from() << "', '" << m->bs() << "', '" << m->fs() << "', "
			 << m->value();
		    if (m->from_dh()) ostr << ", " << m->from_dh(); else ostr << ", null";
		    if (m->bs_dh()  ) ostr << ", " << m->bs_dh();   else ostr << ", null";
		    if (m->fs_dh()  ) ostr << ", " << m->fs_dh();   else ostr << ", null";
		    ostr  << ", " << rejected(m) << ");\n";
		  }
		else if (const H_Diff*     m = dynamic_cast<const H_Diff*    >(*b)) 
		  {
		    const H_Diff* m = dynamic_cast<const H_Diff*>(*b);
		    ostr << "insert into gnu_gama_local_obs "
			 << "(conf_id, ccluster, indx, tag, from_id, to_id, "
			 << "val, dist, rejected) values (" << cnfg() << ", " << cluster << ", "
			 << index++ << ", 'dh', '"
			 << m->from() << "', '" << m->to() << "', "
			 << m->value() << ", ";
		    if (m->dist()) 
		      ostr << ", " << m->dist();
		    else
		      ostr << "null";
		    ostr << ", " << rejected(m) << ");\n";
		  }
		else
		  {
		  }
	      }
	  }
	else if (const HeightDifferences* sp = dynamic_cast<const HeightDifferences*>(c))
	  {
	    write_cluster(ostr, c, cluster, "height-differences");

	    Index index = 1;
	    for (ObservationList::const_iterator 
		   b = c->observation_list.begin(), 
		   e = c->observation_list.end();  b != e;  ++b)
	      {
		const H_Diff* hd = dynamic_cast<const H_Diff*>(*b);
		ostr << "insert into gnu_gama_local_obs "
		     << "(conf_id, ccluster, indx, tag, from_id, to_id, "
		     << "val, dist, rejected) values (" << cnfg() << ", " << cluster << ", "
		     << index++ << ", 'dh', '"
		     << hd->from() << "', '" << hd->to() << "', "
		     << hd->value() << ", ";
		if (hd->dist()) 
		  ostr << hd->dist();
		else
		  ostr << "null";
		ostr << ", " << rejected(hd) << ");\n";
	      }
	  }
	else if (const Coordinates* sp = dynamic_cast<const Coordinates*>(c))
	  {
	    write_cluster(ostr, c, cluster, "coordinates");
	    Index index = 1, inc;
	    for (ObservationList::const_iterator 
		   b = c->observation_list.begin(), 
		   e = c->observation_list.end();  b != e;  ++b)
	      {
		inc = 0;
		std::string        ats;
		std::ostringstream xyz;
		int                rejected_point = 0;
		xyz.precision(17);
		std::string pointid = (*b)->from().str();
		if (const X* xcoord = dynamic_cast<const X*>(*b))
		  {
		    inc++;
		    ats += ", x";
		    xyz << ", " << xcoord->value();
		    if (rejected(*b)) rejected_point = rejected(*b);
		    ObservationList::const_iterator t = b;
		    ++t;
		    if (t != e)
		      {
			inc++;
			++b;
			ats += ", y";
			xyz << ", " << (*b)->value();
			if (rejected(*b)) rejected_point = rejected(*b);
			++t;
			if (t != e && dynamic_cast<const Z*>(*t) 
                            && (*t)->from().str()==pointid)
			  {
			    inc++;
			    ++b;
			    ats += ", z";
			    xyz << ", " << (*b)->value();
			    if (rejected(*b)) rejected_point = rejected(*b);
			  }
		      }
		  }
		else if (const Z* zcoord = dynamic_cast<const Z*>(*b))
		  {
		    inc  = 1;
		    ats += ", z";
		    xyz << ", " << zcoord->value();
		    rejected_point = rejected(zcoord);
		  }
		ostr << "insert into gnu_gama_local_coordinates "
		     << "(conf_id, ccluster, indx, id" << ats
		     << ", rejected) "
		     << "values (" << cnfg() << ", " << cluster << ", "
		     << index << ", '" << pointid << "'" 
		     << xyz.str() << ", " << rejected_point << ");\n"; 
		index += inc;
	      }
	  }
	else if (const Vectors* sp = dynamic_cast<const Vectors*>(c))
	  {
	    write_cluster(ostr, c, cluster, "vectors");
	    Index index = 1, inc;
	    for (ObservationList::const_iterator 
		   b = c->observation_list.begin(), 
		   e = c->observation_list.end();  b != e;  ++b)
	      {
		const Xdiff* xd = dynamic_cast<const Xdiff*>(*b++);
		const Ydiff* yd = dynamic_cast<const Ydiff*>(*b++);
		const Zdiff* zd = dynamic_cast<const Zdiff*>(*b);
		int rejected_point = 0;
		if (rejected(xd)) rejected_point = rejected(xd);
		if (rejected(yd)) rejected_point = rejected(yd);
		if (rejected(zd)) rejected_point = rejected(zd);
		ostr << "insert into gnu_gama_local_vectors "
		     << "(conf_id, ccluster, indx, from_id, to_id, "
		     << "dx, dy, dz, from_dh, to_dh, rejected) values ("
		     << cnfg() << ", " << cluster << ", " << index << ", '" 
		     << xd->from().str() << "', '" << xd->to().str() << "', " 
		     << xd->value() << ", "
		     << yd->value() << ", "
		     << zd->value() << ", "
		     << " null, null"  // from_dh to_dh (are they really used?)
		     << ", " << rejected_point << ");\n";
		index += 3;
	      }
	  }
	else
	  throw GNU_gama::local::Exception("gkf2sql --- unknown cluster type");

	cluster++;
      }
  }

  ostr << "\ncommit;\n";  // commit transaction
}


void Gkf2sql::write_cluster(std::ostream& ostr, const Cluster* c, 
                            int cluster, std::string tag)
{
  const GNU_gama::local::Observation::CovarianceMatrix& covmat = c->covariance_matrix;
  Index dim  = covmat.rows();
  Index band = covmat.bandWidth(); 
  
  ostr << "\ninsert into gnu_gama_local_clusters "
       << "(conf_id, ccluster, dim, band, tag) "
       << "values (" << cnfg() << ", "
       << cluster << ", " << dim << ", " << band << ", '" << tag
       << "');\n";
  
  for (Index i=1; i<=dim; i++)
    for (Index j=i; j<=i+band && j <= dim; j++)
      {
	ostr << "insert into gnu_gama_local_covmat "
	     << "(conf_id, ccluster, rind, cind, val) "
	     << "values (" << cnfg() << ", " << cluster 
	     << ", " << i << ", " << j << "," << covmat(i,j) << ");\n";
      }
}
