/* init.c: init program for MCC interim root disk.
 * Copyright (C) 1996  A. V. Le Blanc

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 * This special init program is used on the MCC interim root disk.
 * By avoiding the need for a shell, it saves over 100k on the disk.
 * On the other hand, this means the potential choices must be carefully
 * thought through and useful options made available.
 */

#include <features.h>
#include <stdlib.h>
#include <unistd.h>
#undef __USE_MISC
#undef __USE_BSD
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <sys/ioctl.h>
#include <sys/sysmacros.h>
#include <linux/unistd.h>
#include <linux/fs.h>
#include <linux/major.h>
#define atoi(s) strtol(s, 0, 10)
#define RAMDEV 0x100

_syscall2(int, bdflush, int, func, int, data);

extern char *scsitab[];
extern int scsisize;
char *scsi_command = NULL;

/* The structure 'menus' is used by the menu function. */

struct menus {
  char retval;
  char *line;
};

/* The 'mounts' structure is used to keep track of mounted partitions,
 * so that they can be unmounted when the system is rebooted.
 */

typedef struct list_element {
  char *devicename;
  struct list_element *next;
} mounts;

/* The following strings occur frequently. */

char minusc[]	= "-c";
char minuspfr[] = "-pfdR";
char minusrf[]	= "-rf";
char minuss[]	= "-s";
char wrarg[]	= "w";
char ext2[]	= "ext2";
char groot[]	= "/";
char fd0[]	= "/dev/fd0";
char fdeflt[]	= "/dev/hda1";
char sdeflt[]	= "/dev/hda5";
char backup[]	= "backupdirs";
char bfirst[]	= "/bin/firstinstall";
char binsh[]	= "/bin/sh";
char binmv[]	= "/bin/mv";
char bsecond[]	= "/bin/secondinstall";
char sdev[]	= "/dev";
char setc[]	= "/etc";
char kbmap[]	= "/etc/kbd.conf";
char slib[]	= "/lib";
char dev[]	= "dev";
char etc[]	= "etc";
char mnt[]	= "mnt";
char droot[]	= "/root";
char proc[]	= "/proc";
char ram[]	= "ram0";
char ramone[]	= "/dev/ram2";
char rcconf[]	= "/root/linuxrc.conf";
char scp[]	= "/ibin/cp";
char sfdisk[]	= "/ibin/fdisk";
char sfsck2[]	= "/ibin/fsck.ext2";
char smkdir[]	= "/ibin/mkdir";
char smkfse[]	= "/ibin/mkfs.ext2";
char srm[]	= "/ibin/rm";
char tfstab[]	= "/root/fstab";
char tmtab[]	= "/root/etc/mtab";
char tswtab[]	= "/root/fswtab";
char trdev[]	= "/root/rootdev";
char rproc[]	= "/root/proc";
char ucpio[]	= "/root/mnt/ibin/cpio";
char ucpioz[]	= "mnt/ibin/cpio.z";
char gunzip[]	= "/root/mnt/ibin/gunzip";
char legitm[]	= "That is not a legitimate device.";
char dpath[]	= "/bin:/usr/bin:/sbin:/usr/sbin:/ibin";
char vpath[]	= "PATH";

char *root_device = NULL;
char *swap_device = NULL;
int retcode, installing = 0, installed = 0, shelling = 0,
	basedev, thisdev, rootdev;
mounts *mounted = NULL;
FILE *fstabfile, *mtabfile;

/* The main menu is arranged so that items appear approximately in the
 * order in which they are selected during a normal installation.
 */

struct menus mainmenuf[] = {
  { 'l', "Load a scsi or xd disk driver"},
  { 'd', "Create an extra device to be mounted"},
  { 'f', "Run fdisk to manage the partition table"},
  { 's', "Set up a new swap partition"},
  { 'a', "Activate an existing swap partition"},
  { 'm', "Make a new file system on a partition"},
  { 'i', "Install basic binaries"},
  { 'c', "Check (and repair) an existing file system"},
  { 'h', "Run sash on the root disk and reboot"},
  { 'e', "Execute /bin/sh on an existing file system"},
  { 'r', "Run a customise script"},
  { 'q', "Quit"},
  { 0, 0}
};
struct menus *mainmenu = mainmenuf;

/* This function performs a sync() every 30 seconds, replacing /etc/update.
 */

void sig_alarm(int junk)
{
  signal(SIGALRM, &sig_alarm);
  alarm(10);
  bdflush(1, 0);
}

/* This function executes a command and cleans up the zombie. */

void execute(char **argv)
{
  pid_t child;

  if ((child = fork()))
    while (wait(&retcode) != child);
  else execv(*argv, argv);
}

/* This function executes a command and reads what it writes to its
 * standard output.  Currently it is called only to run 'fdisk -s'.
 */

void read_pipe(char **argv, char *buffer, int size)
{
  int piped[2], i;
  pid_t child;

  pipe(piped);
  if ((child = fork()))
    while (wait(&retcode) != child);
  else {
    close(1);
    dup(piped[1]);
    close(piped[0]);
    close(piped[1]);
    execv(*argv, argv);
  }

/* Non-zero return code means that an error has occurred. */

  if (retcode) i = 0;
  else i = read(piped[0], buffer, size);
  if (i > size) i = size;
  else if (i < 0) i = 0;
  buffer[i] = '\n';
  i = -1;
  while (buffer[++i] != '\n');
  buffer[i] = '\0';
  close(piped[0]);
  close(piped[1]);
}

/* Read a string and convert it to lower case. */

void read_string(char *buffer, int len)
{
  char *ptr;

  fgets(buffer, len, stdin);
  buffer[len - 1] = '\n';
  ptr = buffer - 1;
  while (*++ptr != '\n');
  *ptr = '\0';
  ptr = buffer - 1;
  while (*++ptr)
    if (*ptr >= 'A' && *ptr <= 'Z')
      *ptr += 'a' - 'A';
}

/* Read 'y' or 'n', repeating a query. */

int yes(char *query)
{
  char buffer[20];

  do {
    fputs(query, stdout);
    fputs(" (Y or N): ", stdout);
    read_string(buffer, 20);
    if (buffer[0] == 'y') return 1;
    if (buffer[0] == 'n') return 0;
  } while(1);
}

/* Verbose umount */

void vumount(char *name)
{
  printf(umount(name) ? "WARNING: Unable to unmount %s\n" :
	"Unmounted %s\n", name);
}

void chrootexec(char **argv)
{
  pid_t child1;

  if (!(child1 = fork())) {
    chdir(droot);
    chroot(droot);
    execv(*argv, argv);
  }
  while (wait(&retcode) != child1);
}

/* Reboot the system, after unmounting any mounted partitions. */

void ireboot(int dontask)
{
  char buffer[20];

  chdir(groot);
  if (dontask || yes("Reboot the system now")) {
    if (installed && (basedev == RAMDEV)) {
      sync();
      kill(-1, 15);
      sleep(5);
      kill(-1, 9);
      sync();
    }
    vumount(rproc);
    sync();
    while (mounted) {
      vumount(mounted->devicename);
      mounted = mounted->next;
    }

/* Report the hexadecimal device code after installing. */

    if (installed) {
      if (basedev == RAMDEV)
	fputs("\nIf you have installed LILO on your hard disk, then "
	  "you should be\nable to reboot without using a floppy disk.  "
	  "Simply remove any floppy\nfrom the drive, and press <RETURN>.",
	  stdout);
      if ((rootdev >> 8 == IDE0_MAJOR) || (rootdev >> 8 == IDE1_MAJOR))
	printf("  If you have not installed LILO,\nthen place the "
	  "boot disk in the drive, and\n\nType 'ro root=%03x' at the "
	  "LILO prompt.\n", rootdev);
    }
    else
      puts("\nPress <RETURN> at the LILO prompt.");
    sync();
    printf("\nIf you have just installed LILO on your hard disk, then "
      "remove any disk\nfrom the floppy drive.  Otherwise, place the "
      "boot floppy in the drive.\nThen press <RETURN>.  If the system "
      "doesn't reboot, press the RESET button.\n");
    read_string(buffer, 20);
#ifdef DEBUG
    exit(0);
#endif
    while (1)
      if (reboot(0xfee1dead, 672274793, 0x1234567) == -1)
	printf("reboot fails with error %d.\n", errno);
      else exit(0);
  }
}

/* Read the name of a block device, and check that it is legitimate. */

char *get_blockdev(char *mesg, char *deflt)
{
  char buffer[20], *ptr, *ptr1;
  struct stat sbuf;

  while (1) {
    ptr = deflt;
    if (mesg) {
      printf("%s (block device; default = %s): ", mesg, deflt);
      read_string(buffer, 20);
      if (buffer[0]) ptr = buffer;
    }
    if ((stat(ptr, &sbuf) >= 0) && S_ISBLK(sbuf.st_mode)) {
      thisdev = sbuf.st_rdev;
      if (ptr == sdeflt || ptr == fdeflt) return ptr;
      ptr1 = (char *) malloc(strlen(ptr) + 1);
      strcpy(ptr1, ptr);
      return ptr1;
    }
    if (buffer[0] == 'q') return NULL;
    puts("This is not a block device; try again or type 'q' to quit.");
  }
}

/* Create a new block device for mounting a disk */

void make_blockdev(char *name)
{
  char line[20];
  dev_t device = 0;
  int max = 63, i;

  if (!name) {
    fputs("Legal devices are hd[abcdefgh]n, xd[ab]n, and sd[abcdefgh]n,\n"
	"where n may be omitted for the disk itself, or may be 1-63 for\n"
	"hd or xd disks, or 1-15 for SCSI disks.  Make a device file for\n"
	"what device? ", stdout);
    read_string(line, 20);
    name = line;
  }
  if (!*name) return;
  i = name[2] - 'a';
  if ((name[1] != 'd') || (i < 0) || (i > 7)) max = 0;
  else if (*name = 's') {
    device = makedev(SCSI_DISK_MAJOR, i << 4);
    max = 15;
  }
  else if ((*name = 'x') && (i < 2))
    device = makedev(XT_DISK_MAJOR, i << 7);
  else if ((*name = 'h') && (i < 4))
    device = makedev(i < 2 ? IDE0_MAJOR : IDE1_MAJOR, (i & 1) << 7);
  else if ((*name = 'h') && (i < 8))
    device = makedev(i < 6 ? IDE2_MAJOR : IDE3_MAJOR, (i & 1) << 7);
  else max = 0;
  if (max) i = atoi(name + 3);
  if (!max || (i > max)) {
    fprintf(stderr, "%s is not a legal device name for a disk.\n", name);
    return;
  }
  device |= i;

  sprintf(line + 5, "/dev/%.3s%d", name, i);
  if (!i) line[13] = 0;
  errno = 0;
  if (mknod(line + 5, S_IFBLK | 0660, device) < 0)
    fprintf(stderr, "Mknod of /dev/%s b %x %x failed with error %d: %s",
	line + 5, device >> 8, device & 0xff, errno, strerror(errno));
  else {
    chmod(line + 5, 0660);
    chown(line + 5, 0, 6);
  }
}

/* Print a menu and return an arbitrary character. */

int menu(struct menus *m)
{
  struct menus *temp = m;
  char buffer[10];

  putchar('\n');
  while (temp->retval) {
    putchar(temp->retval);
    fputs("   ", stdout);
    fputs(temp->line, stdout);
    fputs(".\n", stdout);
    temp++;
  }
  fputs("\n? ", stdout);
  read_string(buffer, 10);
  return(*buffer);
}

/* Paranoia message for mkfs and mkswap. */

int stop_now(void)
{
  puts("WARNING: This will destroy any files on this partition.");
  if (yes("Do you want to continue")) return 0;
  return 1;
}

/* Run mkdir, creating any missing intermediate directories. */

void imkdir(char *path)
{
  char *iargv[] = {smkdir, "-pm", "555", path, NULL};
  struct stat sbuf;

  if ((lstat(path, &sbuf) >= 0) && !S_ISDIR(sbuf.st_mode))
    unlink(path);
  execute(iargv);
}

/* Run fdisk. */

void fdisk(char *dsk)
{
  char deflt[] = "/dev/hda";
  char *argv[] = {sfdisk, NULL, NULL};

  if (dsk) argv[1] = dsk;
  else argv[1] = get_blockdev("Run fdisk on", deflt);
  if (argv[1] && !argv[1][8] &&
      (argv[1][5] == 'h' || argv[1][5] == 's' || argv[1][5] == 'x')) {
    execute(argv);
/*    if (!retcode) ireboot(0); */
  }
  else puts(legitm);
  if (!dsk && argv[1] != deflt)
    free(argv[1]);
}

/* Check size before running mkswap or mkfs. */

int giveup(char *buffer, int max, char *usage)
{
  int i = atoi(buffer);
  char bmesg[] = "blocks of this partition";

  if (i <= max) return 0;
  if (max) {
    printf("WARNING: %d %s cannot be used for %s.\n", i - max, bmesg, usage);
    if (!yes("Continue anyway")) return 1;
  }
  sprintf(buffer, "%d", max);
  return(0);
}

/* Test for valid hard disk devices. */

int valid_hd(char *dev)
{
  return dev[8] && (dev[5] == 'h' || dev[5] == 's' || dev[5] == 'x');
}

/* Run mkswap. */

void mkswap(char *device)
{
  char buffer[32];
  char *fargv[] = {sfdisk, minuss, NULL, NULL};
  char *margv[] = {"/ibin/mkswap", minusc, NULL, buffer, NULL};

  if (!device && stop_now()) return;
  if (device) fargv[2] = get_blockdev(NULL, device);
  else fargv[2] = get_blockdev("Run mkswap on", sdeflt);
  if (fargv[2]) {
    if (valid_hd(fargv[2])) {
      read_pipe(fargv, buffer, 32);
      if (buffer[0]) {
	if (!device && giveup(buffer, 130752, "swapping")) goto cleanup;
	margv[2] = fargv[2];
	execute(margv);
	if (!swapon(margv[2], 0) && !swap_device) {
	  swap_device = fargv[2];
	  fargv[2] = sdeflt;
	}
	goto cleanup;
      }
    }
    puts(legitm);
    cleanup: if (fargv[2] != sdeflt)
      free(fargv[2]);
  }
}

/* Activate a swap partition. */

void tswapon(char *dev)
{
  char *argv = dev;

  if (argv) argv = get_blockdev(NULL, argv);
  else argv = get_blockdev("Activate swapping on partition", sdeflt);
  if (argv) {
    if (valid_hd(argv)) {
      if (!swapon(argv, 0) && !swap_device) {
	swap_device = argv;
	argv = sdeflt;
      }
    }
    else puts(legitm);
    if (argv != sdeflt)
      free(argv);
  }
}

/* Run fsck. */

void fsck(void)
{
  char buffer[10] = "-vft";
  char *fargv[] = {sfsck2, buffer, NULL, NULL};
  int i = 4;

  if ((fargv[2] = get_blockdev("Run fsck on", fdeflt)))
    if (valid_hd(fargv[2])) {
      if (yes("Do you want automatic fixing"))
	buffer[i++] = 'a';
      else /* if (yes("Do you want to fix problems on request")) */
	buffer[i++] = 'r';
      buffer[i] = '\0';
      execute(fargv);
    }
    else puts(legitm);
  if (fargv[2] && fargv[2] != fdeflt)
    free(fargv[2]);
}

/* Run mkfs. */

void mkfs(char *dev)
{
  char   varg[] = "-v",
		  buffer[32];
  char *fargv[] = {sfdisk, minuss, NULL, NULL};
  char *margv[] = {smkfse, varg, NULL, buffer, NULL};

  if (!dev && stop_now()) return;
  if (dev) fargv[2] = dev;
  else if (!(fargv[2] = get_blockdev("Run mkfs on", fdeflt)))
    return;
  if (valid_hd(fargv[2])) {
    read_pipe(fargv, buffer, 32);
    if (!buffer[0]) goto cleanup;
    margv[2] = fargv[2];
    execute(margv);
    cleanup: if (!dev && fargv[2] != fdeflt)
	free(fargv[2]);
    return;
  }
  puts(legitm);
}

/* Delete recursively. */

void irm(char *name)
{
  char *rargv[] = {srm, minusrf, name, NULL};

  execute(rargv);
}

/* Get the name of a swap device, and activate it. */

void get_swap(void)
{
  char *device = NULL;

  device = get_blockdev("Swap on", sdeflt);
  if (device)
    if (valid_hd(device)) {
      if (yes("Do you need to initialise this swap area with mkswap"))
	mkswap(device);
      if (!swapon(device, 0) && !swap_device)
	swap_device = device;
    }
    else puts(legitm);
  if (device && device != sdeflt && device != swap_device)
    free(device);
}

/* Write the scsi loaded module and arguments to a file */

void copy_cmd(char *s)
{
  int fd;

  fd = open(rcconf, O_WRONLY | O_CREAT | O_TRUNC);
  if ((fd < 0) || (write(fd, s, strlen(s)) < 0))
    puts("Can't write config file.");
  if (fd >= 0) close(fd);
  chmod (rcconf, 0400);
}

/* Load a scsi driver */

void scsi_module(char *args)
{
  int i, fd;
  pid_t child;
  char line[100], dumb[2];
  char *zargv[] = {"/ibin/gunzip", NULL};
  char *linuxrc[] = {"/root/linuxrc", NULL};

  if (!args) {
    puts("These are the available modules; select one:\n");
    for (i = 0; i < scsisize; ) {
      printf("  %2d  %-13.13s", i + 1, scsitab[i]);
      if (!(++i & 3)) putchar('\n');
    }
    if (scsisize & 3) putchar('\n');
    do {
      fputs("   0  Do not load one of these drivers.\n\nYour choice: ", stdout);
      read_string(line, 100);
      i = atoi(line);
      if (!i) return;
    } while (i >scsisize);
    strncpy(line, scsitab[i - 1], 20);
    line[20] = 0;
    i = strlen(line);
    line[i++] = ' ';

    puts("Some drivers may require arguments to be passed to them.\n"
	"If this driver requires arguments, give them on this line,\n"
	"in the form 'xxx=nnn yyy=0xnn'.  You can probably just press\n"
	"<RETURN>.");
    fgets(line + i, 98 - i, stdin);
    line[99] = 0;
  }
  puts("Now place the scsi-cd driver disk into the drive and press "
	"<RETURN>.");
  read_string(dumb, 2);
  retcode = mount(fd0, droot, ext2, MS_MGC_VAL | MS_RDONLY, NULL);
  if (retcode) {
    puts("Mount of floppy failed.");
    umount(fd0);
    return;
  }

  if ((child = fork()))
    while (wait(&retcode) != child);
  else {
    fd = open("/root/scsi.gz", O_RDONLY);
    if (fd < 0) {
      puts("Read floppy failed");
      exit(1);
    }
    dup2(fd, 0);
    close(fd);
    fd = open(ramone, O_WRONLY);
    if (fd < 0) {
      puts("Ram write failed");
      close(0);
      exit(1);
    }
    dup2(fd, 1);
    close(fd);
    execute(zargv);
    exit(1);
  }
  umount(fd0);
#if 0
  Why does gzip abort with 'interrupted system call'?
  if (retcode) {
    puts("Can't unzip data file.");
    return;
  }
#endif
  errno = 0;
  if (mount(ramone, droot, ext2, MS_MGC_VAL, NULL)) {
    puts("Can't mount ramdisk.");
    goto cleanup;
  }
  copy_cmd(args ? args : line);
  chdir(droot);
  execute(linuxrc);
  if (retcode) puts("Cannot load module");
  else {
    scsi_command = strdup(line);
    mainmenu++;
  }
  chdir(groot);

cleanup:
  umount(ramone);
}

/* Mount a partition on the specified directory. */

void partition(char *dir, char *dev)
{
  char *device = dev, *name = NULL;
  mounts *newmount;

  if (device) device = get_blockdev(NULL, device);
  else {
    fputs("\nWhich device contains the file system you wish to be mounted"
	"\non '", stdout);
    fputs(dir, stdout);
    puts("'?  Give a name in the form '/dev/hda1', or type"
	"\n'q' to quit.");
    device = get_blockdev("This file system is on", fdeflt);
  }
  if (device) {
    if (valid_hd(device)) {
      if (dir[1]) {
	name = (char *) malloc(strlen(dir) + 6);
	strcpy(name, droot);
	strcat(name, dir);
	irm(name);
	imkdir(name);
	chmod(name, 0111);
	chown(name, 0, 0);
      }
      else name = droot;
      retcode = mount(device, name, ext2, MS_MGC_VAL, NULL);
      if (retcode)
	puts("Mount failed.");
      else {
	chmod(name, 0555);
	chown(name, 0, 0);
	if (dir[1])
	  fprintf(mtabfile, "%s %s ext2 defaults 0 0\n", device,
		installing ? dir : name);
	else {
	  rootdev = thisdev;
	  irm(tmtab);
	  imkdir("/root/etc");
	  if (scsi_command) copy_cmd(scsi_command);
	  else irm(rcconf);
	  mtabfile = fopen(tmtab, wrarg);
	  if (installing) {
	    irm(rproc);
	    imkdir(rproc);
	    chmod(rproc, 0111);
	    chown(rproc, 0, 0);
	  }
	  else fprintf(mtabfile, "%s / ext2 ro 0 0\n",
		basedev == RAMDEV ? "/dev/ram0" : fd0);
	  fprintf(mtabfile, "%s %s ext2 rw 0 0\n/proc /proc proc rw 0 0\n",
	    device, installing ? groot : droot);
	  if (installing) mount(proc, rproc, "proc",
		MS_MGC_VAL, NULL);
	  if (!swap_device &&
		yes("Do you wish to have a swap device"))
	    get_swap();
	  if (installing) {
	    irm(tfstab);
	    irm(tswtab);
	    irm(trdev);
	    fstabfile = fopen(trdev, wrarg);
	    fprintf(fstabfile, "%s\n", device);
	    fclose(fstabfile);
	    if (swap_device) {
	      fstabfile = fopen(tswtab, wrarg);
	      fprintf(fstabfile, "%s\tnone\tswap\tdefaults\t0 0\n",
		swap_device);
	      fclose(fstabfile);
	    }
	    fstabfile = fopen(tfstab, wrarg);
	  }
	}
	if (installing)
	  fprintf(fstabfile, "%s\t%s\text2\tdefaults\t0 %d\n", device, dir,
		dir[1] ? 2 : 1);
	if (!dir[1]) {
	  if (installing)
	    fputs("/proc\t\t/proc\tproc\tdefaults\t0 0\n", fstabfile);
	  root_device = device;
	}
	newmount = (mounts *) malloc(sizeof(mounts));
	newmount->devicename = device;
	newmount->next = mounted;
	mounted = newmount;
	device = fdeflt;
      }
    }
    else puts(legitm);
    if (device != fdeflt) free(device);
    if (name && name != droot) free(name);
  }
}

/* Mount one or more partitions for installation or to execute a shell. */

void imount(void)
{
  char buffer[128];

  chdir(groot);
  partition(groot, NULL);
  if (!root_device) {
    printf("\nAborting the %s.\n", installing ? "installation" :
      "attempt to execute '/bin/sh'");
    installing = 0;
    return;
  }
  while (1) {
    fputs("\nIs there another partition you wish to mount?  If so, give the"
	"\npathname it will have in the installed system, such as '/usr/src'"
	"\nwith a slash at the front and no slash at the end.  (Upper case is"
	"\nnot allowed.)  If not, type 'n': ", stdout);
    read_string(buffer, 128);
    if (buffer[0] == 'n') break;
    if (buffer[0] == '/' && buffer[1] && buffer[strlen(buffer) - 1] != '/' &&
	strcmp(buffer, "/mnt") && strcmp(buffer, proc) &&
	strcmp(buffer, "/bin") && strcmp(buffer, sdev) &&
	strcmp(buffer, setc) &&
	strcmp(buffer, slib) && strcmp(buffer, "/sbin"))
      partition(buffer, NULL);
    else puts(legitm);
  }

  if (installing) fclose(fstabfile);
  else shelling = 1;
  fclose(mtabfile);
}

void install(int mounted)
{
  int piped[2];
  pid_t child1, child2, w;
  char *aargv[] = {srm, minusrf, bfirst, bsecond,
	binsh, "bin/files", "lib/" LC1 "*", "lib/" LE2, "lib/" DL1 "*",
	"lib/" LM1 "*", "lib/modules/" KVER, "etc/ld.so.cache",
	"lib/" NC2, "lib/" LD1 "*", "lib/" LR2, mnt, NULL};
  char *bargv[] = {scp, minuspfr, etc, dev, backup, NULL};
  char *cargv[] = {scp, minuspfr, setc, sdev, slib, droot, NULL};
  char *dargv[] = {scp, minuspfr, "/ibin", "/first.tgz", mnt, NULL};
  char *gargv[] = {gunzip, ucpioz, NULL};
  char *hargv[] = {gunzip, NULL};
  char *iargv[] = {ucpio, "-imudV", "-H", "ustar", "[del]*", "bin/[fs]*", NULL};
  char *margv[] = {bfirst, basedev == RAMDEV ? ram : "fd0", NULL};
  char *sargv[] = {"/etc/init", ram, NULL};

  installing = 1;
  if (!mounted) imount();
  if (!root_device) return;
  chdir(droot);
  execute(aargv);
  imkdir(backup);
  imkdir("backupdirs/etc");
  imkdir("bin");
  imkdir(dev);
  execute(bargv);
  irm(etc);
  irm(dev);
  imkdir(etc);
  imkdir(dev);
  imkdir(mnt);
  imkdir("usr");
  imkdir("usr/bin");
  execute(cargv);
  execute(dargv);
  execute(gargv);

  pipe(piped);
  if (!(child1 = fork())) {
    int fd;
    fd = open("mnt/first.tgz", O_RDONLY);
    close(0);
    dup(fd);
    close(fd);
    close(1);
    dup(piped[1]);
    close(piped[1]);
    close(piped[0]);
    execv(*hargv, hargv);
  }
  close(piped[1]);
  if (!(child2 = fork())) {
    close(0);
    dup(piped[0]);
    close(piped[0]);
    execv(*iargv, iargv);
  }
  close(piped[0]);
  do {
    w = wait(&retcode);
    if (w == child1) child1 = 0;
    if (w == child2) child2 = 0;
  } while (child2 || child1);

  chrootexec(margv);
  if(basedev == RAMDEV)
    chrootexec(sargv);
  installed = 1;
  if (!mounted) ireboot(0);
}

int customise(void)
{
  char line[100], *words[10];
  FILE *fd;
  int i, j;

  chdir(groot);
  printf("What is the file's pathname? ");
  read_string(line, 100);
  if (line[0]) fd = fopen(line, "r");
  else return 1;
  if (!fd) {
    printf("Unable to read file %s.\n", line);
    return 1;
  }
  while (fgets(line, 100, fd)) {
    j = 10;
    i = 0;
    do {
      while (isspace(line[i])) i++;
      if (isgraph(line[i])) {
	words[10 - j--] = line + i;
	while (isgraph(line[++i]));
	line[i++] = '\0';
      }
      else words[10 - j--] = NULL;
    } while (j);

    if (words[0] && *words[0] != '#')
      if (!strcmp(words[0], "fdisk"))
	if (words[2]) {
	  int piped[2];
	  pid_t child;

	  pipe(piped);
	  if ((child = fork())) {
	    close(piped[0]);
	    do {
	      if (fgets(line, 100, fd)) {
		if (line[0] != '#')
		  write(piped[1], line, strlen(line));
	      }
	      else line[0] = 'w';
	    } while (line[0] != 'w' && line[0] != 'q');
	    close(piped[1]);
	    while (wait(&retcode) != child);
	  }
	  else {
	    char *argv[] = {sfdisk, words[1], NULL};

	    close(0);
	    dup(piped[0]);
	    close(piped[0]);
	    close(piped[1]);
	    execv(*argv, argv);
	  }
	}
	else fdisk(words[1]);
      else if (!strcmp(words[0], "mkswap"))
	mkswap(words[1]);
      else if (!strcmp(words[0], "swapon"))
	tswapon(words[1]);
      else if (!strcmp(words[0], "mkfs"))
	mkfs(words[1]);
      else if (!strcmp(words[0], "mount")) {
	installing = 1;
	partition(words[2], words[1]);
      }
      else if (!strcmp(words[0], "install")) {
	fclose(fstabfile);
	fclose(mtabfile);
	install(1);
      }
      else if (!strcmp(words[0], "reboot"))
	ireboot(1);
      else if (!strcmp(words[0], "mknod"))
	make_blockdev(words[1]);
      else if (!strcmp(words[0], "load")) {
	char buf[100];

	i = 1; j = 0;
	while ((i < 10) && words[i]) {
	  strcpy(buf + j, words[i++]);
	  j += strlen(buf + j);
	  buf[j++] = ' ';
	}
	buf[j] = '\n';
	buf[j + 1] = 0;
	scsi_module(buf);
      }
      else printf("Bad script command: %s\n", words[0]);
  }
  fclose(fd);
  if (!installed) {
    fclose(fstabfile);
    fclose(mtabfile);
  }
  return !installed;
}

int mmain(void)
{
  char *sash[] = {"/ibin/sash", NULL};

  switch (menu(mainmenu)) {
  case 'a': tswapon(NULL); break;
  case 'c': fsck(); break;
  case 'd': make_blockdev(NULL); break;
  case 'e': if (!mounted) imount();
	    return !root_device;
  case 'f': fdisk(NULL); break;
  case 'h': setenv(vpath, dpath, 1); signal(SIGINT, SIG_DFL);
	    execute(sash); ireboot(1); break;
  case 'i': install(0); return !root_device;
  case 'l': scsi_module(NULL); break;
  case 'm': mkfs(NULL); break;
  case 'q': return 0;
  case 'r': return customise();
  case 's': mkswap(NULL); break;
  default: puts("Menu error");
  }
  return 1;
}

void main(void)
{
  char *sargv[] = {binsh, NULL};
  char *kargv[] = {"/ibin/loadkeys", kbmap, NULL};
  struct stat sbuf;

  umask(066);
  stat(groot, &sbuf);
  basedev = sbuf.st_dev;
#ifndef DEBUG
  if (basedev != RAMDEV) mainmenu += 2;
  reboot(0xfee1dead, 672274793, 0);
  signal(SIGKILL, SIG_IGN);
  signal(SIGTERM, SIG_IGN);
  signal(SIGINT, SIG_IGN);
  signal(SIGHUP, SIG_IGN);
#endif
  chdir(groot);
  if (!fork()) {
    setsid();
    bdflush(0, 0);	/* Should never return */
    exit(1);
  }
  sig_alarm(0);
  if ((stat(kbmap, &sbuf) >= 0) && S_ISREG(sbuf.st_mode))
    execute(kargv);
  setenv(vpath, dpath, 1);

  while (mmain());

  while(1) {
    if (installed || shelling) {
      puts("\nPlease type CTRL-D to reboot, not 'reboot' or 'shutdown'.\n"
	"Also, you must umount any partitions you mount before you exit.");
      signal(SIGINT, SIG_DFL);
      chrootexec(sargv);
    }
    ireboot(0);
  }
}
