/* boot.c  -  Boot image composition */

/* Written 1992,1993 by Werner Almesberger */


#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <a.out.h>
#include <sys/stat.h>

#include <linux/config.h>

#include "config.h"
#include "common.h"
#include "geometry.h"
#include "cfg.h"
#include "map.h"
#include "partition.h"
#include "boot.h"


static GEOMETRY geo;
static struct stat st;


static void check_size(char *name,int sectors)
{
    if (sectors > SETUPSECS+MAX_KERNEL_SECS)
	die("Kernel %s is too big",name);
}


void boot_image(char *spec,IMAGE_DESCR *descr)
{
    BOOT_SECTOR buff;
    int fd,sectors;

    if (verbose > 0) printf("Boot image: %s\n",spec);
    if (dump) cfg_dump("image",0,"%s",spec);
    fd = geo_open(&geo,spec,O_RDONLY);
    map_begin_section();
    if (fstat(fd,&st) < 0) die("fstat %s: %s",spec,strerror(errno));
    if (read(fd,(char *) &buff,SECTOR_SIZE) != SECTOR_SIZE)
	die("read %s: %s",spec,strerror(errno));
    if (buff.par_l.root_dev) descr->root_dev = buff.par_l.root_dev;
    map_add(&geo,0,(st.st_size+SECTOR_SIZE-1)/SECTOR_SIZE);
    check_size(spec,sectors = map_end_section(&descr->start));
    descr->bss_words = (SECTOR_SIZE-(st.st_size & (SECTOR_SIZE-1))) >> 1;
    if (verbose > 2)
	printf("BSS: initializing %d bytes at end.\n",descr->bss_words << 1);
    geo_close(&geo);
    if (verbose > 1)
	printf("Mapped %d sector%s.\n",sectors,sectors == 1 ? "" : "s");
}


void boot_device(char *spec,char *range,IMAGE_DESCR *descr)
{
    char *here;
    int start,secs;
    int sectors;

    if (verbose > 0) printf("Boot device: %s, range %s\n",spec,range);
    if (dump) {
	cfg_dump("image",0,"%s",spec);
	cfg_dump("range",1,"%s",range);
    }
    (void) geo_open(&geo,spec,O_NOACCESS);
    map_begin_section();
    if (here = strchr(range,'-')) {
	*here++ = 0;
	start = to_number(range);
	if ((secs = to_number(here)-start+1) < 0) die("Invalid range");
    }
    else {
	if (here = strchr(range,'+')) {
	    *here++ = 0;
	    start = to_number(range);
	    secs = to_number(here);
	}
	else {
	    start = to_number(range);
	    secs = 1;
	}
    }
    map_add(&geo,start,secs);
    check_size(spec,sectors = map_end_section(&descr->start));
    geo_close(&geo);
    if (verbose > 1)
	printf("Mapped %d sector%s.\n",sectors,sectors == 1 ? "" : "s");
}


static void map_unstripped(GEOMETRY *geo,char *name,IMAGE_DESCR *descr)
{
    struct exec exec;
    int sectors;

    if (lseek(geo->fd,0L,0) < 0) die("lseek %s: %s",name,strerror(errno));
    if (read(geo->fd,(char *) &exec,sizeof(struct exec)) != sizeof(struct exec))
	die("read %s: %s",name,strerror(errno));
    if (N_MAGIC(exec) != ZMAGIC || exec.a_entry || exec.a_trsize ||
      exec.a_drsize || !exec.a_syms) die("Not an unstripped kernel: %s",name);
    map_add(geo,2,(exec.a_text+exec.a_data+SECTOR_SIZE-1)/SECTOR_SIZE);
    if ((exec.a_text+exec.a_data) & 15)
	fprintf(stderr,"Warning: Unsupported BSS - no initialization.\n");
    else {
	if (verbose > 2)
	    printf("BSS: 0x%X+%d\n",exec.a_text+exec.a_data+DEF_SYSSEG,
	      exec.a_bss);
	descr->bss_seg = ((exec.a_text+exec.a_data) >> 4)+DEF_SYSSEG;
	descr->bss_segs = exec.a_bss >> 16;
	descr->bss_words = ((exec.a_bss & 0xffff)+1) >> 1;
    }
    geo_close(geo);
    check_size(name,sectors = map_end_section(&descr->start));
    if (verbose > 1)
	printf("Mapped %d sector%s.\n",sectors,sectors == 1 ? "" : "s");
}


void boot_unstripped(char *boot,char *setup,char *kernel,IMAGE_DESCR *descr)
{
    if (!setup) die("Please specify a setup file for %s",kernel);
    if (verbose > 0)
	printf("Unstripped: boot %s, setup %s,\n  kernel %s\n",boot ? boot :
	  "(default)",setup,kernel);
    if (dump) {
	cfg_dump("unstripped",0,"%s",kernel);
	cfg_dump("setup",1,"%s",setup);
	if (boot) cfg_dump("boot",1,"%s",boot);
    }
    map_begin_section();
    if (!boot) map_add_zero();
    else {
	(void) geo_open(&geo,boot,O_RDONLY);
	map_add(&geo,0,1);
	geo_close(&geo);
    }
    (void) geo_open(&geo,setup,O_RDONLY);
    map_add(&geo,0,SETUPSECS);
    geo_close(&geo);
    (void) geo_open(&geo,kernel,O_RDONLY);
    map_unstripped(&geo,kernel,descr);
}


void boot_compound(char *kernel,IMAGE_DESCR *descr)
{
    int fd;
    struct stat st;

    if (verbose > 0) printf("Compound kernel: %s\n",kernel);
    if (dump) cfg_dump("compound",0,"%s",kernel);
    map_begin_section();
    fd = geo_open(&geo,kernel,O_RDONLY);
    if (fstat(fd,&st) < 0) die("fstat %s: %s",kernel,strerror(errno));
    if ((st.st_size & (SECTOR_SIZE-1)) || st.st_size < sizeof(struct exec)+
      (SETUPSECS+2)*SECTOR_SIZE) die("%s: invalid size",kernel);
    map_add(&geo,(st.st_size/SECTOR_SIZE)-1,1);
    map_add(&geo,(st.st_size/SECTOR_SIZE)-SETUPSECS-1,SETUPSECS);
    map_unstripped(&geo,kernel,descr);
}


void boot_other(char *loader,char *boot,char *part,IMAGE_DESCR *descr)
{
    int b_fd,l_fd,p_fd,walk,found,size;
    unsigned short magic;
    BOOT_SECTOR buff;
    struct stat st;

    if (verbose > 0)
	printf("Boot other: %s%s%s, loader %s\n",boot,part ? ", on " : "",part
	  ? part : "",loader);
    if (dump) {
	cfg_dump("other",0,"%s",boot);
	if (loader) cfg_dump("loader",1,"%s",loader);
	if (part) cfg_dump("table",1,"%s",part);
    }
    if (!loader) loader = DFL_CHAIN;
    if (cfg_get_flag(cf_other,"unsafe")) {
	b_fd = geo_open(&geo,boot,3);
	if (part) die("TABLE and UNSAFE are mutually incompatible.");
	if (dump) cfg_dump("unsafe",1,NULL);
    }
    else {
	b_fd = geo_open(&geo,boot,O_RDONLY);
	if (fstat(b_fd,&st) < 0)
	    die("fstat %s: %s",boot,strerror(errno));
	part_verify(st.st_dev);
	if (lseek(b_fd,(long) BOOT_SIG_OFFSET,0) < 0)
	    die("lseek %s: %s",boot,strerror(errno));
	if ((size = read(b_fd,(char *) &magic,2)) != 2)
	    if (size < 0) die("read %s: %s",boot,strerror(errno));
	    else die("Can't get magic number of %s",boot);
	if (magic != BOOT_SIGNATURE)
	    die("First sector of %s doesn't have a valid boot signature",boot);
    }
    if (!part) memset(&buff,0,SECTOR_SIZE);
    else {
	if ((p_fd = open(part,O_RDONLY)) < 0)
	    die("open %s: %s",part,strerror(errno));
	if (read(p_fd,(char *) &buff,SECTOR_SIZE) != SECTOR_SIZE)
	    die("read %s: %s",part,strerror(errno));
    }
    if ((l_fd = open(loader,O_RDONLY)) < 0)
	die("open %s: %s",loader,strerror(errno));
    if ((size = read(l_fd,(char *) &buff,PART_TABLE_OFFSET+1)) < 0)
	die("read %s: %s",loader,strerror(errno));
    check_version(&buff,STAGE_CHAIN);
    if (size > PART_TABLE_OFFSET)
	die("Chain loader %s is too big",loader);
    if (part) {
	found = 0;
	for (walk = 0; walk < PARTITION_ENTRIES; walk++)
	    if (!PART(buff,walk).sys_ind || PART(buff,walk).start_sect != geo.start) {
		if (PART(buff,walk).sys_ind != PART_DOS12 && PART(buff,walk).
		  sys_ind != PART_DOS16) PART(buff,walk).sys_ind = PART_INVALID;
	    }
	    else {
		if (found) die("Duplicate entry in partition table");
		buff.par_c.offset = walk*PARTITION_ENTRY;
		PART(buff,walk).boot_ind = 0x80;
		found = 1;
	    }
	if (!found) die("Partition entry not found.");
	(void) close(p_fd);
    }
    (void) close(l_fd);
    buff.par_c.drive = geo.device;
    map_begin_section();
    map_add_zero();
    map_add_sector(&buff);
    map_add(&geo,0,1);
    (void) map_end_section(&descr->start);
    geo_close(&geo);
    if (verbose > 1) printf("Mapped 2 (1+1) sectors.\n");
}
