/* lilo.c  -  LILO command-line parameter processing */

/* Written 1992,1993 by Werner Almesberger */


#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include "config.h"
#include "common.h"
#include "lilo.h"
#include "temp.h"
#include "geometry.h"
#include "map.h"
#include "bsect.h"
#include "cfg.h"
#include "identify.h"


#define S2(x) #x
#define S(x) S2(x)


static void show_images(char *map_file)
{
    DESCR_SECTORS descrs;
    char *name;
    int fd,image,i;
    short unsigned checksum;

    if ((fd = open(map_file,O_RDONLY)) < 0)
	die("open %s: %s",map_file,strerror(errno));
    if (lseek(fd,SECTOR_SIZE,0) < 0)
	die("lseek %s: %s",map_file,strerror(errno));
    if (read(fd,(char *) &descrs,sizeof(descrs)) != sizeof(descrs))
	die("read %s: %s",map_file,strerror(errno));
    (void) close(fd);
    for (image = 0; image < MAX_IMAGES; image++)
	if (*(name = descrs.d.descr[image].name)) {
	    printf("%-" S(MAX_IMAGE_NAME) "s%s",name,image ? "  " : " *");
	    if (verbose > 0) {
		printf(" <dev=0x%02x,hd=%d,cyl=%d,sct=%d>",
		  descrs.d.descr[image].start.device,
		  descrs.d.descr[image].start.head,
		  descrs.d.descr[image].start.track,
		  descrs.d.descr[image].start.sector);
		if (descrs.d.descr[image].present & OVR_ROOT)
		    printf(",root=0x%04x",descrs.d.descr[image].root_dev);
	    }
	    printf("\n");
	}
    checksum = INIT_CKS;
    for (i = 0; i < sizeof(descrs)/sizeof(unsigned short); i++)
	checksum ^= ((unsigned short *) &descrs)[i];
    if (!checksum) exit(0);
    fflush(stdout);
    fprintf(stderr,"Checksum error.\n");
    exit(1);
}


static void usage(char *name)
{
    char *here;

    if (here = strchr(name,'/')) name = here+1;
    fprintf(stderr,"usage: %s [ -q ] [ -m map_file ] [ -v ]\n",name);
    fprintf(stderr,"%7s%s [ -D ] [ -b boot_device ] [ -c ] [ -l ] "
      "[ -i boot_sector ]\n","",name);
    fprintf(stderr,"%12s[ -f disk_tab ] [ -m map_file ] [ -d delay ] "
      "[ -v ] [ -t ]\n","");
    fprintf(stderr,"%12s[ [ -s | -S ] save_file ] [ -r root_dir ] "
      "[ boot_descr ... ]\n","");
    fprintf(stderr,"%7s%s [ -C config_file ] [ -D ] [ -b boot_device ] [ -c ] "
      "[ -l ]\n","",name);
    fprintf(stderr,"%12s[ -i boot_sector ] [ -f disk_tab ] [ -m map_file ] "
      "[ -d delay ]\n","");
    fprintf(stderr,"%12s[ -v ] [ -t ] [ [ -s | -S ] save_file ] [ -P ]"
      "[ -r root_dir ]\n","");
    fprintf(stderr,"%7s%s [ -C config_file ] [ -m map_file ] [ -P ] [ -R "
      "[ word ... ] ]\n","",name);
    fprintf(stderr,"%7s%s [ -C config_file ] [ -I name [ options ] ]\n\n","",
      name);
    fprintf(stderr,"Descriptor format:\n");
    fprintf(stderr,"    [ label= ] image(s) [ ,root_dev ]\n\n");
    fprintf(stderr,"Images:\n");
    fprintf(stderr,"  Boot image:  file_name\n");
    fprintf(stderr,"  Device:      device #start [ +num | -end ]\n");
    fprintf(stderr,"  Unstripped:  setup+kernel\n");
    fprintf(stderr,"  Other OS:    loader+boot_sect@ [ part_table ]\n");
    exit(1);
}


int main(int argc,char **argv)
{
    char *name,*config_file,*reboot_arg,*identify,*ident_opt,*new_root;
    int image,query,more;
    BOOT_SECTOR dummy;
    IMAGE_DESCR dummy2;

    config_file = reboot_arg = identify = ident_opt = new_root = NULL;
    query = 0;
    name = *argv++;
    argc--;
    cfg_init(cf_options);
    while (argc && **argv == '-') {
	argc--;
	if (argv[0][2]) usage(name);
	switch ((*argv++)[1]) {
	    case 'b':
		if (!argc) usage(name);
		cfg_set(cf_options,"boot",*argv++,NULL);
		argc--;
		break;
	    case 'c':
		cfg_set(cf_options,"compact",NULL,NULL);
		compact = 1;
		break;
	    case 'd':
		if (!argc) usage(name);
		cfg_set(cf_options,"delay",*argv++,NULL);
		argc--;
		break;
	    case 'f':
		if (!argc) usage(name);
		cfg_set(cf_options,"disktab",*argv++,NULL);
		argc--;
		break;
	    case 'l':
		cfg_set(cf_options,"linear",NULL,NULL);
		linear = 1;
		break;
	    case 'm':
		if (!argc) usage(name);
		cfg_set(cf_options,"map",*argv++,NULL);
		argc--;
		break;
	    case 'i':
		if (!argc) usage(name);
		cfg_set(cf_options,"install",*argv++,NULL);
		argc--;
		break;
	    case 'I':
		if (!argc) usage(name);
		identify = *argv++;
		if (--argc) {
		    ident_opt = *argv++;
		    argc--;
		}
		break;
	    case 'X':
		printf("-DIMAGES=%d -DCODE_START_1=%d -DCODE_START_2=%d "
		  "-DDESCR_SIZE=%d -DDSC_OFF=%d -DDSC_OFF2=%d -DDFCMD_OFF=%d "
		  "-DMSG_OFF=%d -DPRES_OFF=%d\n",MAX_IMAGES,
		  sizeof(BOOT_PARAMS_1),sizeof(BOOT_PARAMS_2),
		  sizeof(IMAGE_DESCR),
		  (void *) &dummy.par_1.descr[0]-(void *) &dummy,
		  (void *) &dummy.par_1.descr[1]-(void *) &dummy,
		  (void *) &dummy.par_1.descr[2]-(void *) &dummy,
		  (void *) &dummy.par_1.msg_len-(void *) &dummy,
		  (void *) &dummy2.present-(void *) &dummy2);
		exit(0);
	    case 'C':
		if (!argc) usage(name);
		config_file = *argv++;
		argc--;
		break;
	    case 'D':
		dump = test = 1;
		break;
	    case 'S':
		if (!argc) usage(name);
		cfg_set(cf_options,"force-backup",*argv++,NULL);
		argc--;
		break;
	    case 's':
		if (!argc) usage(name);
		cfg_set(cf_options,"backup",*argv++,NULL);
		argc--;
		break;
	    case 'P':
		cfg_set(cf_options,"fix-table",NULL,NULL);
		break;
	    case 'q':
		query = 1;
		break;
	    case 'r':
		if (!argc) usage(name);
		new_root = *argv++;
		argc--;
		break;
	    case 'R':
		if (!argc) reboot_arg = "";
		else while (argc--) {
			if (!reboot_arg)
			    *(reboot_arg = alloc(strlen(reboot_arg)+1)) = 0;
			else strcat(reboot_arg = ralloc(reboot_arg,
			      strlen(reboot_arg)+strlen(*argv)+2)," ");
			strcat(reboot_arg,*argv++);
		    }
		break;
	    case 't':
		test = 1;
		break;
	    case 'v':
		verbose++;
		break;
	    default:
		usage(name);
	}
    }
    if (!new_root) new_root = getenv("ROOT");
    if (new_root && *new_root) {
	if (chroot(new_root) < 0) die("chroot %s: %s",new_root,strerror(errno));
	if (chdir("/") < 0) die("chdir /: %s",strerror(errno));
    }
    if (atexit(temp_remove)) die("atexit() failed");
    if (verbose > 0 && !dump)
	printf("LILO version 0.9\nWritten 1992,1993 by Werner Almesberger\n\n");
#if 0
    if (((install || test || boot_device || disktab_file || compact) && !argc)
      || (compact && linear && 0)) usage(name);
#endif
    if (config_file) {
	cfg_open(config_file);
	more = cfg_parse(cf_options);
	if (identify) identify_image(identify,ident_opt);
    }
    compact = cfg_get_flag(cf_options,"compact");
    linear = cfg_get_flag(cf_options,"linear");
    if (cfg_get_strg(cf_options,"verbose"))
	verbose += to_number(cfg_get_strg(cf_options,"verbose"));
    if (reboot_arg) {
	map_patch_first(cfg_get_strg(cf_options,"map") ? cfg_get_strg(
	  cf_options,"map") : MAP_FILE,reboot_arg);
	exit(0);
    }
    if (cfg_get_strg(cf_options,"delay") && cfg_get_flag(cf_options,"prompt"))
	die("Can't use DELAY and PROMPT at the same time.");
    if (dump) {
#define DUMP_FLAG(x) \
  if (cfg_get_flag(cf_options,x)) cfg_dump(x,0,NULL);
#define DUMP_STRG(x) \
  if (cfg_get_strg(cf_options,x)) \
    cfg_dump(x,0,"%s",cfg_get_strg(cf_options,x));
	DUMP_STRG("boot");
	DUMP_STRG("delay");
	DUMP_FLAG("prompt");
	DUMP_STRG("timeout");
	DUMP_STRG("install");
	DUMP_STRG("disktab");
	DUMP_STRG("map");
	DUMP_STRG("message");
	if (verbose) printf("verbose = %d\n",verbose);
	DUMP_STRG("backup");
	DUMP_STRG("force-backup");
	DUMP_FLAG("fix-table");
	DUMP_FLAG("compact");
	DUMP_FLAG("linear");
	DUMP_STRG("serial");
	DUMP_STRG("root");
	DUMP_STRG("vga");
	DUMP_STRG("ramdisk");
	DUMP_STRG("password");
	verbose = -1;
    }
    if ((!argc && !config_file) || query)
	show_images(!cfg_get_strg(cf_options,"map") ? MAP_FILE :
	  cfg_get_strg(cf_options,"map"));
    if ((remaining = MAX_IMAGES-argc) < 0) {
	fflush(stdout);
	fprintf(stderr,"Only %d images can be defined\n",MAX_IMAGES);
	exit(1);
    }
    geo_init(cfg_get_strg(cf_options,"disktab"));
    bsect_open(cfg_get_strg(cf_options,"boot"),cfg_get_strg(cf_options,"map") ?
      cfg_get_strg(cf_options,"map") : MAP_FILE,cfg_get_strg(cf_options,
      "install"),cfg_get_strg(cf_options,"delay") ? to_number(cfg_get_strg(
      cf_options,"delay")) : 0,cfg_get_strg(cf_options,"timeout") ?
      to_number(cfg_get_strg(cf_options,"timeout")) : -1);
    if (!config_file)
	for (image = 0; image < argc; image++) bsect_image(*argv++);
    else {
	if (argc) usage(name);
	if (more) {
	    cfg_init(cf_top);
	    if (cfg_parse(cf_top)) cfg_error("Syntax error");
	}
    }
    if (!bsect_number()) die("No images have been defined.");
    if (!test)
	if (cfg_get_strg(cf_options,"force-backup"))
	    bsect_update(cfg_get_strg(cf_options,"force-backup"),1);
	else bsect_update(cfg_get_strg(cf_options,"backup"),0);
    else {
	bsect_cancel();
	fprintf(stderr,"The boot sector and the map file have *NOT* been "
	  "altered.\n");
    }
    return 0;
}
