/* init.c: init program for MCC interim root disk.
 * Copyright (C) 1994  A. V. Le Blanc

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 * This special init program is used on the MCC interim root disk.
 * By avoiding the need for a shell, it saves over 100k on the disk.
 * On the other hand, this means the potential choices must be carefully
 * thought through and useful options made available.
 */

#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/mount.h>

extern int swapon(const char *);
extern void setenv(const char *name, const char *value);
extern void sync(void);

/* The structure 'menus' is used by the menu function. */

struct menus {
  char retval;
  char *line;
};

/* The 'mounts' structure is used to keep track of mounted partitions,
 * so that they can be unmounted when the system is rebooted.
 */

typedef struct list_element {
  char *devicename;
  struct list_element *next;
} mounts;

/* The following strings occur frequently. */

char minusc[]	= "-c";
char minusp[]	= "-p";
char minusrf[]	= "-rf";
char minuss[]	= "-s";
char wrarg[]	= "w";
char ext2[]	= "ext2";
char groot[]	= "/";
char fdeflt[]	= "/dev/hda1";
char sdeflt[]	= "/dev/hda5";
char bbin[]	= "bin";
char bincp[]	= "/bin/cp";
char bmcci[]	= "/usr/sbin/mccinstall";
char binsh[]	= "/bin/sh";
char kbmap[]	= "/etc/kbmap";
char droot[]	= "/root";
char proc[]	= "/proc";
char scp[]	= "/ibin/cp";
char sfdisk[]	= "/ibin/fdisk";
char sfsck2[]	= "/ibin/fsck.ext2";
char smkdir[]	= "/ibin/mkdir";
char smkfsm[]	= "/ibin/mkfs.minix";
char smkfse[]	= "/ibin/mkfs.ext2";
char srm[]	= "/ibin/rm";
char tfstab[]	= "/root/fstab";
char tmtab[]	= "/root/etc/mtab";
char tswtab[]	= "/root/fswtab";
char trdev[]	= "/root/rootdev";
char usbin[]	= "usr/bin";
char ucpio[]	= "/usr/bin/cpio";
char ucpioz[]	= "/usr/bin/cpio.z";
char gunzip[]	= "/usr/bin/gunzip";
char legitm[]	= "That is not a legitimate device.";
char ext2m[]	= "Does this device contain an ext2 file system";

char *root_device = NULL;
char *swap_device = NULL;
int retcode, installing = 0, installed = 0, shelling = 0,
	basedev, thisdev, rootdev;
mounts *mounted = NULL;
FILE *fstabfile, *mtabfile;

/* The main menu is arranged so that items appear approximately in the
 * order in which they are selected during a normal installation.
 */

struct menus mainmenu[] = {
  { 'f', "Run fdisk to manage the partition table"},
  { 's', "Set up a new swap partition"},
  { 'a', "Activate an existing swap partition"},
  { 'm', "Make a new file system on a partition"},
  { 'i', "Install basic binaries"},
  { 'c', "Check (and repair) an existing file system"},
  { 'e', "Execute /bin/sh on an existing file system"},
  { 'r', "Run a customise script"},
  { 'q', "Quit"},
  { 0, 0}
};

/* This function performs a sync() every 30 seconds, replacing /etc/update.
 */

void sig_alarm(int junk)
{
  signal(SIGALRM, &sig_alarm);
  alarm(30);
  sync();
}

/* This function executes a command and cleans up the zombie. */

void execute(char **argv)
{
  pid_t child;

  if ((child = fork()))
    while (wait(&retcode) != child);
  else execv(*argv, argv);
}

/* This function executes a command and reads what it writes to its
 * standard output.  Currently it is called only to run 'fdisk -s'.
 */

void read_pipe(char **argv, char *buffer, int size)
{
  int piped[2], i;
  pid_t child;

  pipe(piped);
  if ((child = fork()))
    while (wait(&retcode) != child);
  else {
    close(1);
    dup(piped[1]);
    close(piped[0]);
    close(piped[1]);
    execv(*argv, argv);
  }

/* Non-zero return code means that an error has occurred. */

  if (retcode) i = 0;
  else i = read(piped[0], buffer, size);
  if (i > size) i = size;
  else if (i < 0) i = 0;
  buffer[i] = '\n';
  i = -1;
  while (buffer[++i] != '\n');
  buffer[i] = '\0';
  close(piped[0]);
  close(piped[1]);
}

/* Read a string and convert it to lower case. */

void read_string(char *buffer, int len)
{
  char *ptr;

  fgets(buffer, len, stdin);
  buffer[len - 1] = '\n';
  ptr = buffer - 1;
  while (*++ptr != '\n');
  *ptr = '\0';
  ptr = buffer - 1;
  while (*++ptr)
    if (*ptr >= 'A' && *ptr <= 'Z')
      *ptr += 'a' - 'A';
}

/* Read 'y' or 'n', repeating a query. */

int yes(char *query)
{
  char buffer[20];

  do {
    fputs(query, stdout);
    fputs(" (Y or N): ", stdout);
    read_string(buffer, 20);
    if (buffer[0] == 'y') return 1;
    if (buffer[0] == 'n') return 0;
  } while(1);
}

/* Reboot the system, after unmounting any mounted partitions. */

void ireboot(int dontask)
{
  char buffer[20];

  chdir(groot);
  if (dontask || yes("Reboot the system now")) {
    sync();
    while (mounted) {
      if (!umount(mounted->devicename))
	printf("Unmounted %s\n", mounted->devicename);
      mounted = mounted->next;
    }

/* Report the hexadecimal device code after installing. */

    if (installed) {
      if (basedev == 0x101)
        fputs("\nIf you have installed LILO on your hard disk, then "
	  "you should be\nable to reboot without using a floppy disk.  "
	  "Simply remove any floppy\nfrom the drive, and press <RETURN>.  "
	  "If you have not installed LILO,\nthen place the boot disk in "
	  "the drive, and\n", stdout);
      printf("\nType 'ro root=%03x' at the LILO prompt.\n", rootdev);
    }
    else
      puts("\nPress <RETURN> at the LILO prompt.");
    sync();
    printf("\nIf you have just installed LILO on your hard disk, then "
      "remove any disk\nfrom the floppy drive.  Otherwise, place the "
      "boot floppy in the drive.\nThen press <RETURN>.  If the system "
      "doesn't reboot, press the RESET button.\n");
    read_string(buffer, 20);
    while (1)
      if (reboot(0xfee1dead, 672274793, 0x1234567) == -1)
	printf("reboot fails with error %d.\n", errno);
      else exit(0);
  }
}

/* Read the name of a block device, and check that it is legitimate. */

char *get_blockdev(char *mesg, char *deflt)
{
  char buffer[20], *ptr, *ptr1;
  struct stat sbuf;

  while (1) {
    ptr = deflt;
    if (mesg) {
      printf("%s (block device; default = %s): ", mesg, deflt);
      read_string(buffer, 20);
      if (buffer[0]) ptr = buffer;
    }
    if ((stat(ptr, &sbuf) >= 0) && S_ISBLK(sbuf.st_mode)) {
      thisdev = sbuf.st_rdev;
      if (ptr == sdeflt || ptr == fdeflt) return ptr;
      ptr1 = (char *) malloc(strlen(ptr) + 1);
      strcpy(ptr1, ptr);
      return ptr1;
    }
    if (buffer[0] == 'q') return NULL;
    puts("This is not a block device; try again or type 'q' to quit.");
  }
}

/* Print a menu and return an arbitrary character. */

int menu(struct menus *m)
{
  struct menus *temp = m;
  char buffer[10];

  putchar('\n');
  while (temp->retval) {
    putchar(temp->retval);
    fputs("   ", stdout);
    fputs(temp->line, stdout);
    fputs(".\n", stdout);
    temp++;
  }
  fputs("\n? ", stdout);
  read_string(buffer, 10);
  return(*buffer);
}

/* Paranoia message for mkfs and mkswap. */

int stop_now(void)
{
  puts("WARNING: This will destroy any files on this partition.");
  if (yes("Do you want to continue")) return 0;
  return 1;
}

/* Run mkdir, creating any missing intermediate directories. */

void imkdir(char *path)
{
  char *iargv[] = {smkdir, minusp, "-m", "555", path, NULL};
  struct stat sbuf;

  if ((lstat(path, &sbuf) >= 0) && !S_ISDIR(sbuf.st_mode))
    unlink(path);
  execute(iargv);
}

/* Run fdisk. */

void fdisk(char *dsk)
{
  char deflt[] = "/dev/hda";
  char *argv[] = {sfdisk, NULL, NULL};

  if (dsk) argv[1] = dsk;
  else argv[1] = get_blockdev("Run fdisk on", deflt);
  if (argv[1] && !argv[1][8] &&
      (argv[1][5] == 'h' || argv[1][5] == 's' || argv[1][5] == 'x')) {
    execute(argv);
/*    if (!retcode) ireboot(0); */
  }
  else puts(legitm);
  if (!dsk && argv[1] != deflt)
    free(argv[1]);
}

/* Check size before running mkswap or mkfs. */

int giveup(char *buffer, int max, char *usage)
{
  int i = atoi(buffer);
  char bmesg[] = "blocks of this partition";

  if (i <= max) return 0;
  if (max) {
    printf("WARNING: %d %s cannot be used for %s.\n", i - max, bmesg, usage);
    if (!yes("Continue anyway")) return 1;
  }
#if 0
  else {
    printf("By default all %d %s are used for the file system.\n"
	"Sometimes this causes problems, due to a bug in mkfs.ext2.\n",
	i, bmesg);
    do {
      if (!yes("Reduce this number")) return 0;
      fputs("Give the reduced size: ", stdout);
      read_string(buffer, 32);
      max = atoi(buffer);
    } while (max > i);
  }
#endif
  sprintf(buffer, "%d", max);
  return(0);
}

/* Test for valid hard disk devices. */

int valid_hd(char *dev)
{
  return dev[8] && (dev[5] == 'h' || dev[5] == 's' || dev[5] == 'x');
}

/* Run mkswap. */

void mkswap(char *device)
{
  char buffer[32];
  char *fargv[] = {sfdisk, minuss, NULL, NULL};
  char *margv[] = {"/ibin/mkswap", minusc, NULL, buffer, NULL};

  if (!device && stop_now()) return;
  if (device) fargv[2] = get_blockdev(NULL, device);
  else fargv[2] = get_blockdev("Run mkswap on", sdeflt);
  if (fargv[2]) {
    if (valid_hd(fargv[2])) {
      read_pipe(fargv, buffer, 32);
      if (buffer[0]) {
	if (!device && giveup(buffer, 16384, "swapping")) goto cleanup;
	margv[2] = fargv[2];
	execute(margv);
	swapon(margv[2]);
	if (!swap_device) {
	  swap_device = fargv[2];
	  fargv[2] = sdeflt;
	}
	goto cleanup;
      }
    }
    puts(legitm);
    cleanup: if (fargv[2] != sdeflt)
      free(fargv[2]);
  }
}

/* Activate a swap partition. */

void tswapon(char *dev)
{
  char *argv = dev;

  if (argv) argv = get_blockdev(NULL, argv);
  else argv = get_blockdev("Activate swapping on partition", sdeflt);
  if (argv) {
    if (valid_hd(argv)) {
      swapon(argv);
      if (!swap_device) {
	swap_device = argv;
	argv = sdeflt;
      }
    }
    else puts(legitm);
    if (argv != sdeflt)
      free(argv);
  }
}

/* Run fsck. */

void fsck(void)
{
  char buffer[10] = "-vftc";
  char *fargv[] = {sfsck2, buffer, NULL, NULL};
  int i = 4;

  if ((fargv[2] = get_blockdev("Run fsck on", fdeflt)))
    if (valid_hd(fargv[2])) {
      if (yes(ext2m)) {
	if (yes("Check for bad blocks")) i++;
      }
      else {
	fargv[0] = "/ibin/fsck.minix";
	buffer[3] = 's';
      }
      if (yes("Do you want automatic fixing"))
	buffer[i++] = 'a';
      else /* if (yes("Do you want to fix problems on request")) */
	buffer[i++] = 'r';
      buffer[i] = '\0';
      execute(fargv);
    }
    else puts(legitm);
  if (fargv[2] && fargv[2] != fdeflt)
    free(fargv[2]);
}

/* Run mkfs. */

void mkfs(char *kind, char *dev)
{
  char	narg30[] = "-n30",
	varg[] = "-v", buffer[32];
  char *fargv[] = {sfdisk, minuss, NULL, NULL};
  char *margv[] = {smkfsm, minusc, narg30, NULL, buffer, NULL};
/*  char *cargv[] = {sfsck2, "-vfta", NULL, NULL}; */

  if (!kind && !dev && stop_now()) return;
  if (dev) fargv[2] = dev;
  else if (!(fargv[2] = get_blockdev("Run mkfs on", fdeflt)))
    return;
  if (valid_hd(fargv[2])) {
    read_pipe(fargv, buffer, 32);
    if (buffer[0]) {
      margv[3] = fargv[2];
      if (kind && !strcmp(kind, ext2) ||
	  !kind && yes("Create an ext2 file system")) {
	margv[0] = smkfse;
	margv[2] = varg;
      }
      else {
	if (!kind && giveup(buffer, 65536, "a Minix file system"))
	  goto cleanup;
      }
      execute(margv);
  /*    if (margv[0] == smkfse) {
	cargv[2] = fargv[2];
	execute(cargv);
      } */
      cleanup: if (!dev && fargv[2] != fdeflt)
	free(fargv[2]);
      return;
    }
  }
  puts(legitm);
}

/* Delete recursively. */

void irm(char *name)
{
  char *rargv[] = {srm, minusrf, name, NULL};

  execute(rargv);
}

/* Get the name of a swap device, and activate it. */

void get_swap(void)
{
  char *device = NULL;

  device = get_blockdev("Swap on", sdeflt);
  if (device)
    if (valid_hd(device)) {
      if (yes("Do you need to initialise this swap area with mkswap"))
	mkswap(device);
      if (!swap_device) swap_device = device;
      swapon(device);
    }
    else puts(legitm);
  if (device && device != sdeflt && device != swap_device)
    free(device);
}

/* Mount a partition on the specified directory. */

void partition(char *dir, char *dev, char *kind)
{
  char *device = dev, *name = NULL;
  char *margv;
  mounts *newmount;

  if (device) device = get_blockdev(NULL, device);
  else {
    fputs("\nWhich device contains the file system you wish to be mounted"
	"\non '", stdout);
    fputs(dir, stdout);
    puts("'?  Give a name in the form '/dev/hda1', or type"
	"\n'q' to quit.");
    device = get_blockdev("This file system is on", fdeflt);
  }
  if (device) {
    if (valid_hd(device)) {
      if (dir[1]) {
	name = (char *) malloc(strlen(dir) + 6);
	strcpy(name, droot);
	strcat(name, dir);
	irm(name);
	imkdir(name);
	chmod(name, 0111);
	chown(name, 0, 0);
      }
      else name = droot;
      if (kind) margv = kind;
      else if (yes(ext2m))
	margv = ext2;
      else margv = "minix";
      retcode = mount(device, name, margv, 0xc0ed0000, NULL);
      if (retcode)
	puts("Mount failed.");
      else {
	chmod(name, 0555);
	chown(name, 0, 0);
	if (!dir[1]) {
	  rootdev = thisdev;
	  irm(tmtab);
	  imkdir("/root/etc");
	  mtabfile = fopen(tmtab, wrarg);
	  if (basedev == 0x101)
	    fputs("/dev/ram / minix ro 0 0\n/dev/fd0 /mnt", mtabfile);
	  else fputs("/dev/fd0 /", mtabfile);
	  fputs(" minix ro 0 0\n/proc /proc proc defaults 0 0\n",
	    mtabfile);
	  if (!swap_device &&
		yes("Do you wish to have a swap device"))
	    get_swap();
	  mount(proc, proc, "proc", 0xc0ed0000, NULL);
	  if (installing) {
	    irm(tfstab);
	    irm(tswtab);
	    irm(trdev);
	    fstabfile = fopen(trdev, wrarg);
	    fprintf(fstabfile, "%s\n", device);
	    fclose(fstabfile);
	    if (swap_device) {
	      fstabfile = fopen(tswtab, wrarg);
	      fprintf(fstabfile, "%s\tnone\tswap\tdefaults\n", swap_device);
	      fclose(fstabfile);
	    }
	    fstabfile = fopen(tfstab, wrarg);
	  }
	}
	if (installing)
	  fprintf(fstabfile, "%s\t%s\t%s\tdefaults\n", device, dir, margv);
	fprintf(mtabfile, "%s %s %s defaults 0 0\n", device, name, margv);
	if (!dir[1]) {
	  if (installing)
	    fputs("/proc\t\t/proc\tproc\tdefaults\n", fstabfile);
	  root_device = device;
	}
	newmount = (mounts *) malloc(sizeof(mounts));
	newmount->devicename = device;
	newmount->next = mounted;
	mounted = newmount;
	device = fdeflt;
      }
    }
    else puts(legitm);
    if (device != fdeflt) free(device);
    if (name && name != droot) free(name);
  }
}

/* Mount one or more partitions for installation or to execute a shell. */

void imount(void)
{
  char buffer[128];

  chdir(groot);
  partition(groot, NULL, NULL);
  if (!root_device) {
    printf("\nAborting the %s.\n", installing ? "installation" :
      "attempt to execute '/bin/sh'");
    installing = 0;
    return;
  }
  while (1) {
    fputs("\nIs there another partition you wish to mount?  If so, give the"
	"\npathname it will have in the installed system, such as '/usr/src'"
	"\nwith a slash at the front and no slash at the end.  (Upper case is"
	"\nnot allowed.)  If not, type 'n': ", stdout);
    read_string(buffer, 128);
    if (buffer[0] == 'n') break;
    if (buffer[0] == '/' && buffer[1] && buffer[strlen(buffer) - 1] != '/' &&
	strcmp(buffer, "/mnt") && strcmp(buffer, proc) &&
	strcmp(buffer, "/bin") && strcmp(buffer, "/dev") &&
	strcmp(buffer, "/etc") && strcmp(buffer, "/boot") &&
	strcmp(buffer, "/lib") && strcmp(buffer, "/sbin"))
      partition(buffer, NULL, NULL);
    else puts(legitm);
  }

  if (installing) fclose(fstabfile);
  else shelling = 1;
  fclose(mtabfile);
}

void install(int mounted)
{
  int piped[2];
  pid_t child1, child2, w;
  char *cargv[] = {scp, minusp, scp, smkdir, srm, bbin, NULL};
  char *dargv[] = {bincp, minusp, "/ibin/cpio.z", "/ibin/gunzip",
    usbin, NULL};
  char *gargv[] = {gunzip, ucpioz, NULL};
  char *hargv[] = {gunzip, NULL};
  char *iargv[] = {ucpio, "-imu", "-H", "ustar", NULL};
  char *margv[] = {bmcci, "Base.tgz", NULL};
  char *rargv[] = {srm, minusrf, bincp, ucpioz, "bin/cpio", "usr/bin/find",
	"bin/gunzip", "bin/mkdir", binsh, "bin/bash", "bin/mv", "bin/rm",
	ucpio, gunzip, bmcci, NULL};
  char *rc[] = {"/root/etc/rc", NULL};

  installing = 1;
  if (!mounted) imount();
  if (!root_device) return;
  chdir(droot);
  imkdir(bbin);
  imkdir("usr");
  imkdir(usbin);
  imkdir("usr/sbin");
  execute(rargv);
  execute(cargv);
  execute(dargv);
  execute(gargv);

  pipe(piped);
  if (!(child1 = fork())) {
    int fd;
    fd = open("/first.tgz", O_RDONLY);
    close(0);
    dup(fd);
    close(fd);
    close(1);
    dup(piped[1]);
    close(piped[1]);
    close(piped[0]);
    execv(*hargv, hargv);
  }
  close(piped[1]);
  if (!(child2 = fork())) {
    close(0);
    dup(piped[0]);
    close(piped[0]);
    execv(*iargv, iargv);
  }
  close(piped[0]);
  do {
    w = wait(&retcode);
    if (w == child1) child1 = 0;
    if (w == child2) child2 = 0;
  } while (child2 || child1);

  setenv("ROOT", droot);
  setenv("prompt", "n");
  setenv("mnt", groot);
  execute(margv);
  if (basedev == 0x101) {
    execute(rc);
    umount("/dev/fd0");
  }
  installed = 1;
  if (!mounted) ireboot(0);
}

int customise(void)
{
  char line[100], *words[4];
  FILE *fd;
  int i, j;

  chdir(groot);
  printf("What is the file's pathname? ");
  read_string(line, 100);
  if (line[0]) fd = fopen(line, "r");
  else return 1;
  if (!fd) {
    printf("Unable to read file %s.\n", line);
    return 1;
  }
  while (fgets(line, 100, fd)) {
    j = 4;
    i = 0;
    do {
      while (isspace(line[i])) i++;
      if (isgraph(line[i])) {
	words[4 - j--] = line + i;
	while (isgraph(line[++i]));
	line[i++] = '\0';
      }
      else words[4 - j--] = NULL;
    } while (j);

    if (words[0] && *words[0] != '#')
      if (!strcmp(words[0], "fdisk"))
	if (words[2]) {
	  int piped[2];
	  pid_t child;

	  pipe(piped);
	  if ((child = fork())) {
	    close(piped[0]);
	    do {
	      if (fgets(line, 100, fd)) {
		if (line[0] != '#')
		  write(piped[1], line, strlen(line));
	      }
	      else line[0] = 'w';
	    } while (line[0] != 'w' && line[0] != 'q');
	    close(piped[1]);
	    while (wait(&retcode) != child);
	  }
	  else {
	    char *argv[] = {sfdisk, words[1], NULL};

	    close(0);
	    dup(piped[0]);
	    close(piped[0]);
	    close(piped[1]);
	    execv(*argv, argv);
	  }
	}
	else fdisk(words[1]);
      else if (!strcmp(words[0], "mkswap"))
	mkswap(words[1]);
      else if (!strcmp(words[0], "swapon"))
	tswapon(words[1]);
      else if (!strcmp(words[0], "mkfs"))
	mkfs(words[1], words[2]);
      else if (!strcmp(words[0], "mount")) {
	installing = 1;
	partition(words[3], words[2], words[1]);
      }
      else if (!strcmp(words[0], "install")) {
	fclose(fstabfile);
	fclose(mtabfile);
	install(1);
      }
      else if (!strcmp(words[0], "reboot"))
	ireboot(1);
      else printf("Bad script command: %s\n", words[0]);
  }
  fclose(fd);
  if (!installed) {
    fclose(fstabfile);
    fclose(mtabfile);
  }
  return !installed;
}

int mmain(void)
{
  switch (menu(mainmenu)) {
  case 'a': tswapon(NULL); break;
  case 'c': fsck(); break;
  case 'e': if (!mounted) imount();
	    return !root_device;
  case 'f': fdisk(NULL); break;
  case 'i': install(0); return !root_device;
  case 'm': mkfs(NULL, NULL); break;
  case 'q': return 0;
  case 'r': return customise();
  case 's': mkswap(NULL); break;
  default: puts("Menu error");
  }
  return 1;
}

void main(void)
{
  char *sargv[] = {binsh, NULL};
  char *kargv[] = {"/ibin/loadkeys", kbmap, NULL};
  struct stat sbuf;

  stat(groot, &sbuf);
  basedev = sbuf.st_dev;
  reboot(0xfee1dead, 672274793, 0);
  signal(SIGTERM, SIG_IGN);
  signal(SIGINT, SIG_IGN);
  chdir(groot);
  sig_alarm(0);
  if ((stat(kbmap, &sbuf) >= 0) && S_ISREG(sbuf.st_mode))
    execute(kargv);
  setenv("PATH", "/bin:/usr/bin:/sbin:/usr/sbin:/ibin");

  while (mmain());

  while(1) {
    if (installed || shelling) {
      puts("\nPlease type CTRL-D to reboot, not 'reboot' or 'shutdown'.");
      execute(sargv);
    }
    ireboot(0);
  }
}
