/*
 * This file is part of gtkD.
 *
 * gtkD is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version, with
 * some exceptions, please read the COPYING file.
 *
 * gtkD is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with gtkD; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110, USA
 */

// generated automatically - do not change
// find conversion definition on APILookup.txt
// implement new conversion functionalities on the wrap.utils pakage


module gtk.Statusbar;

private import glib.ConstructionException;
private import glib.Str;
private import gobject.ObjectG;
private import gobject.Signals;
private import gtk.Box;
private import gtk.Widget;
private import gtkc.gtk;
public  import gtkc.gtktypes;
private import std.algorithm;


/**
 * A #GtkStatusbar is usually placed along the bottom of an application's
 * main #GtkWindow. It may provide a regular commentary of the application's
 * status (as is usually the case in a web browser, for example), or may be
 * used to simply output a message when the status changes, (when an upload
 * is complete in an FTP client, for example).
 * 
 * Status bars in GTK+ maintain a stack of messages. The message at
 * the top of the each bar’s stack is the one that will currently be displayed.
 * 
 * Any messages added to a statusbar’s stack must specify a
 * context id that is used to uniquely identify
 * the source of a message. This context id can be generated by
 * gtk_statusbar_get_context_id(), given a message and the statusbar that
 * it will be added to. Note that messages are stored in a stack, and when
 * choosing which message to display, the stack structure is adhered to,
 * regardless of the context identifier of a message.
 * 
 * One could say that a statusbar maintains one stack of messages for
 * display purposes, but allows multiple message producers to maintain
 * sub-stacks of the messages they produced (via context ids).
 * 
 * Status bars are created using gtk_statusbar_new().
 * 
 * Messages are added to the bar’s stack with gtk_statusbar_push().
 * 
 * The message at the top of the stack can be removed using
 * gtk_statusbar_pop(). A message can be removed from anywhere in the
 * stack if its message id was recorded at the time it was added. This
 * is done using gtk_statusbar_remove().
 * 
 * # CSS node
 * 
 * GtkStatusbar has a single CSS node with name statusbar.
 */
public class Statusbar : Box
{
	/** the main Gtk struct */
	protected GtkStatusbar* gtkStatusbar;

	/** Get the main Gtk struct */
	public GtkStatusbar* getStatusbarStruct(bool transferOwnership = false)
	{
		if (transferOwnership)
			ownedRef = false;
		return gtkStatusbar;
	}

	/** the main Gtk struct as a void* */
	protected override void* getStruct()
	{
		return cast(void*)gtkStatusbar;
	}

	protected override void setStruct(GObject* obj)
	{
		gtkStatusbar = cast(GtkStatusbar*)obj;
		super.setStruct(obj);
	}

	/**
	 * Sets our main struct and passes it to the parent class.
	 */
	public this (GtkStatusbar* gtkStatusbar, bool ownedRef = false)
	{
		this.gtkStatusbar = gtkStatusbar;
		super(cast(GtkBox*)gtkStatusbar, ownedRef);
	}

	/**
	 * Retrieves the box containing the label widget.
	 * Since 2.20
	 * Returns: a GtkBox. [transfer none]
	 */
	public Box getMessageArea()
	{
		auto p = gtk_statusbar_get_message_area(gtkStatusbar);
		if(p is null)
		{
			return null;
		}
		return new Box(cast(GtkBox*) p);
	}

	/**
	 */

	/** */
	public static GType getType()
	{
		return gtk_statusbar_get_type();
	}

	/**
	 * Creates a new #GtkStatusbar ready for messages.
	 *
	 * Returns: the new #GtkStatusbar
	 *
	 * Throws: ConstructionException GTK+ fails to create the object.
	 */
	public this()
	{
		auto p = gtk_statusbar_new();
		
		if(p is null)
		{
			throw new ConstructionException("null returned by new");
		}
		
		this(cast(GtkStatusbar*) p);
	}

	/**
	 * Returns a new context identifier, given a description
	 * of the actual context. Note that the description is
	 * not shown in the UI.
	 *
	 * Params:
	 *     contextDescription = textual description of what context
	 *         the new message is being used in
	 *
	 * Returns: an integer id
	 */
	public uint getContextId(string contextDescription)
	{
		return gtk_statusbar_get_context_id(gtkStatusbar, Str.toStringz(contextDescription));
	}

	/**
	 * Removes the first message in the #GtkStatusbar’s stack
	 * with the given context id.
	 *
	 * Note that this may not change the displayed message, if
	 * the message at the top of the stack has a different
	 * context id.
	 *
	 * Params:
	 *     contextId = a context identifier
	 */
	public void pop(uint contextId)
	{
		gtk_statusbar_pop(gtkStatusbar, contextId);
	}

	/**
	 * Pushes a new message onto a statusbar’s stack.
	 *
	 * Params:
	 *     contextId = the message’s context id, as returned by
	 *         gtk_statusbar_get_context_id()
	 *     text = the message to add to the statusbar
	 *
	 * Returns: a message id that can be used with
	 *     gtk_statusbar_remove().
	 */
	public uint push(uint contextId, string text)
	{
		return gtk_statusbar_push(gtkStatusbar, contextId, Str.toStringz(text));
	}

	/**
	 * Forces the removal of a message from a statusbar’s stack.
	 * The exact @context_id and @message_id must be specified.
	 *
	 * Params:
	 *     contextId = a context identifier
	 *     messageId = a message identifier, as returned by gtk_statusbar_push()
	 */
	public void remove(uint contextId, uint messageId)
	{
		gtk_statusbar_remove(gtkStatusbar, contextId, messageId);
	}

	/**
	 * Forces the removal of all messages from a statusbar's
	 * stack with the exact @context_id.
	 *
	 * Params:
	 *     contextId = a context identifier
	 *
	 * Since: 2.22
	 */
	public void removeAll(uint contextId)
	{
		gtk_statusbar_remove_all(gtkStatusbar, contextId);
	}

	protected class OnTextPoppedDelegateWrapper
	{
		static OnTextPoppedDelegateWrapper[] listeners;
		void delegate(uint, string, Statusbar) dlg;
		gulong handlerId;
		
		this(void delegate(uint, string, Statusbar) dlg)
		{
			this.dlg = dlg;
			this.listeners ~= this;
		}
		
		void remove(OnTextPoppedDelegateWrapper source)
		{
			foreach(index, wrapper; listeners)
			{
				if (wrapper.handlerId == source.handlerId)
				{
					listeners[index] = null;
					listeners = std.algorithm.remove(listeners, index);
					break;
				}
			}
		}
	}

	/**
	 * Is emitted whenever a new message is popped off a statusbar's stack.
	 *
	 * Params:
	 *     contextId = the context id of the relevant message/statusbar
	 *     text = the message that was just popped
	 */
	gulong addOnTextPopped(void delegate(uint, string, Statusbar) dlg, ConnectFlags connectFlags=cast(ConnectFlags)0)
	{
		auto wrapper = new OnTextPoppedDelegateWrapper(dlg);
		wrapper.handlerId = Signals.connectData(
			this,
			"text-popped",
			cast(GCallback)&callBackTextPopped,
			cast(void*)wrapper,
			cast(GClosureNotify)&callBackTextPoppedDestroy,
			connectFlags);
		return wrapper.handlerId;
	}
	
	extern(C) static void callBackTextPopped(GtkStatusbar* statusbarStruct, uint contextId, char* text, OnTextPoppedDelegateWrapper wrapper)
	{
		wrapper.dlg(contextId, Str.toString(text), wrapper.outer);
	}
	
	extern(C) static void callBackTextPoppedDestroy(OnTextPoppedDelegateWrapper wrapper, GClosure* closure)
	{
		wrapper.remove(wrapper);
	}

	protected class OnTextPushedDelegateWrapper
	{
		static OnTextPushedDelegateWrapper[] listeners;
		void delegate(uint, string, Statusbar) dlg;
		gulong handlerId;
		
		this(void delegate(uint, string, Statusbar) dlg)
		{
			this.dlg = dlg;
			this.listeners ~= this;
		}
		
		void remove(OnTextPushedDelegateWrapper source)
		{
			foreach(index, wrapper; listeners)
			{
				if (wrapper.handlerId == source.handlerId)
				{
					listeners[index] = null;
					listeners = std.algorithm.remove(listeners, index);
					break;
				}
			}
		}
	}

	/**
	 * Is emitted whenever a new message gets pushed onto a statusbar's stack.
	 *
	 * Params:
	 *     contextId = the context id of the relevant message/statusbar
	 *     text = the message that was pushed
	 */
	gulong addOnTextPushed(void delegate(uint, string, Statusbar) dlg, ConnectFlags connectFlags=cast(ConnectFlags)0)
	{
		auto wrapper = new OnTextPushedDelegateWrapper(dlg);
		wrapper.handlerId = Signals.connectData(
			this,
			"text-pushed",
			cast(GCallback)&callBackTextPushed,
			cast(void*)wrapper,
			cast(GClosureNotify)&callBackTextPushedDestroy,
			connectFlags);
		return wrapper.handlerId;
	}
	
	extern(C) static void callBackTextPushed(GtkStatusbar* statusbarStruct, uint contextId, char* text, OnTextPushedDelegateWrapper wrapper)
	{
		wrapper.dlg(contextId, Str.toString(text), wrapper.outer);
	}
	
	extern(C) static void callBackTextPushedDestroy(OnTextPushedDelegateWrapper wrapper, GClosure* closure)
	{
		wrapper.remove(wrapper);
	}
}
