'use strict';

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Ruby tokenizer template.
 */
var RUBY_TOKENIZER_TEMPLATE = _fs2.default.readFileSync(__dirname + '/templates/tokenizer.template.rb', 'utf-8');

/**
 * The trait is used by parser generators (LL/LR) for Ruby.
 */
/**
 * The MIT License (MIT)
 * Copyright (c) 2015-present Dmitry Soshnikov <dmitry.soshnikov@gmail.com>
 */

var RubyParserGeneratorTrait = {

  /**
   * Generates parser class name.
   */
  generateParserClassName: function generateParserClassName(className) {
    this.writeData('PARSER_CLASS_NAME', className);
  },


  /**
   * Generates parsing table in Ruby hash format.
   */
  generateParseTable: function generateParseTable() {
    this.writeData('TABLE', this._toRubyHash(this.generateParseTableData()));
  },


  /**
   * Generates tokens table in Ruby hash format.
   */
  generateTokensTable: function generateTokensTable() {
    this.writeData('TOKENS', this._toRubyHash(this._tokens));
  },


  /**
   * Production handlers are implemented as private methods
   * on the `YYParse` class.
   */
  buildSemanticAction: function buildSemanticAction(production) {
    var action = this.getSemanticActionCode(production);

    if (!action) {
      return null;
    }

    action = this._scopeVars(action);

    var args = this._scopeVars(this.getSemanticActionParams(production).join(','));

    // Save the action, they are injected later.
    this._productionHandlers.push({ args: args, action: action });
    return '\'_handler' + this._productionHandlers.length + '\'';
  },


  /**
   * Generates built-in tokenizer instance.
   */
  generateBuiltInTokenizer: function generateBuiltInTokenizer() {
    this.writeData('TOKENIZER', RUBY_TOKENIZER_TEMPLATE);
  },


  /**
   * Generates rules for tokenizer.
   */
  generateLexRules: function generateLexRules() {
    var _this = this;

    var lexRules = this._grammar.getLexGrammar().getRules().map(function (lexRule) {
      var action = _this._scopeVars(lexRule.getRawHandler());
      _this._lexHandlers.push({ args: '', action: action });

      // NOTE: Ruby's beginning of a string `^` symbol matches beginning of
      // every line, so use `\A` instead for it.

      var flags = [];

      if (lexRule.isCaseInsensitive()) {
        flags.push('i');
      }

      return '[/\\A' + lexRule.getOriginalMatcher() + '/' + flags.join('') + ', ' + ('\'_lex_rule' + _this._lexHandlers.length + '\']');
    });

    this.writeData('LEX_RULES', '[' + lexRules.join(',\n') + ']');
  },
  generateLexRulesByStartConditions: function generateLexRulesByStartConditions() {
    var lexGrammar = this._grammar.getLexGrammar();
    var lexRulesByConditions = lexGrammar.getRulesByStartConditions();
    var result = {};

    for (var condition in lexRulesByConditions) {
      result[condition] = lexRulesByConditions[condition].map(function (lexRule) {
        return lexGrammar.getRuleIndex(lexRule);
      });
    }

    this.writeData('LEX_RULES_BY_START_CONDITIONS', '' + this._toRubyHash(result));
  },


  /**
   * Replaces global vars like `yytext`, `$$`, etc. to be
   * referred from `YYParse`.
   */
  _scopeVars: function _scopeVars(code) {
    return code.replace(/yytext/g, 'YYParse.yytext').replace(/yyleng/g, 'YYParse.yyleng').replace(/\b__loc\b/g, 'YYParse.__loc').replace(/\b__\b/g, 'YYParse.__');
  },


  /**
   * Converts JS object to Ruby's hash representation.
   */
  _toRubyHash: function _toRubyHash(object) {
    if (typeof object !== 'object') {
      return JSON.stringify(object);
    }
    var result = [];
    for (var k in object) {
      var value = object[k];

      if (Array.isArray(object)) {
        result.push(this._toRubyHash(value));
      } else {
        var key = k.replace(/'/g, "\\'");
        result.push("'" + key + "' => " + this._toRubyHash(value));
      }
    }
    return Array.isArray(object) ? '[' + result.join(', ') + ']' : '{' + result.join(', ') + '}';
  },


  /**
   * Python-specific lex rules handler declarations.
   */
  generateLexHandlers: function generateLexHandlers() {
    var handlers = this._generateHandlers(this._lexHandlers, '_lex_rule');
    this.writeData('LEX_RULE_HANDLERS', handlers.join('\n\n'));
  },


  /**
   * Python-specific handler declarations.
   */
  generateProductionHandlers: function generateProductionHandlers() {
    var handlers = this._generateHandlers(this._productionHandlers, '_handler',
    /* isStatic */true);
    this.writeData('PRODUCTION_HANDLERS', handlers.join('\n\n'));
  },


  /**
   * Generates Ruby's `def` methods for handlers.
   */
  _generateHandlers: function _generateHandlers(handlers, name) {
    var isStatic = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;

    return handlers.map(function (_ref, index) {
      var args = _ref.args,
          action = _ref.action;

      return 'def ' + (isStatic ? 'self.' : '') + name + (index + 1) + '(' + args + ')\n' + (action + '\nend');
    });
  }
};

module.exports = RubyParserGeneratorTrait;