#Concat

[![Build Status](https://travis-ci.org/puppetlabs/puppetlabs-concat.png?branch=master)](https://travis-ci.org/puppetlabs/puppetlabs-concat)

####Table of Contents

1. [Overview](#overview)
2. [Module Description - What the module does and why it is useful](#module-description)
3. [Setup - The basics of getting started with concat](#setup)
    * [What concat affects](#what-concat-affects)
    * [Setup requirements](#setup-requirements)
    * [Beginning with concat](#beginning-with-concat)
4. [Usage - Configuration options and additional functionality](#usage)
    * [API _deprecations_](#api-deprecations)
5. [Reference - An under-the-hood peek at what the module is doing and how](#reference)
5. [Limitations - OS compatibility, etc.](#limitations)
6. [Development - Guide for contributing to the module](#development)

##Overview

This module constructs files from multiple fragments in an ordered way.

##Module Description

This module lets you use many concat::fragment{} resources throughout
your modules to construct a single file at the end.  It does this through
a shell (or ruby) script and a temporary holding space for the fragments.

##Setup

###What concat affects

* Installs concatfragments.[sh|rb] based on platform.
* Adds a concat/ directory into Puppets `vardir`.

###Beginning with concat

To start using concat you need to create:

* A concat{} resource for the final file.
* One or more concat::fragment{}'s. 

A minimal example might be:

```puppet
concat { '/tmp/file':
  ensure => present,
}

concat::fragment { 'tmpfile':
  target  => '/tmp/file',
  content => 'test contents',
  order   => '01'
}
```

##Usage

Please be aware that there have been a number of [API
_deprecations_](#api-deprecations).

If you wanted a /etc/motd file that listed all the major modules
on the machine.  And that would be maintained automatically even
if you just remove the include lines for other modules you could
use code like below, a sample /etc/motd would be:

<pre>
Puppet modules on this server:

    -- Apache
    -- MySQL
</pre>

Local sysadmins can also append to the file by just editing /etc/motd.local
their changes will be incorporated into the puppet managed motd.

```puppet
class motd {
  $motd = '/etc/motd'

  concat { $motd:
    owner => 'root',
    group => 'root',
    mode  => '0644'
  }

  concat::fragment{ 'motd_header':
    target  => $motd,
    content => "\nPuppet modules on this server:\n\n",
    order   => '01'
  }

  # local users on the machine can append to motd by just creating
  # /etc/motd.local
  concat::fragment{ 'motd_local':
    target => $motd,
    source => '/etc/motd.local',
    order  => '15'
  }
}

# used by other modules to register themselves in the motd
define motd::register($content="", $order='10') {
  if $content == "" {
    $body = $name
  } else {
    $body = $content
  }

  concat::fragment{ "motd_fragment_$name":
    target  => '/etc/motd',
    order   => $order,
    content => "    -- $body\n"
  }
}
```

To use this you'd then do something like:

```puppet
class apache {
  include apache::install, apache::config, apache::service

  motd::register{ 'Apache': }
}
```

##Reference

###Classes

####Public classes

####Private classes
* `concat::setup`: Sets up the concat script/directories.

###Parameters

###Defines

####concat

#####`ensure`
Controls if the combined file is present or absent.

######Example
- ensure => present
- ensure => absent

#####`path`
Controls the destination of the file to create.

######Example
- path => '/tmp/filename'

#####`owner`
Set the owner of the combined file.

######Example
- owner => 'root'

#####`group`
Set the group of the combined file.

######Example
- group => 'root'

#####`mode`
Set the mode of the combined file.

######Example
- mode => '0644'

#####`warn`
Determine if a warning message should be added at the top of the file to let
users know it was autogenerated by Puppet.  It should be a boolean or a string
containing the contents of the warning message.

######Example
- warn => true
- warn => false
- warn => '# This file is autogenerated!'

#####`force`
Determine if empty files are allowed when no fragments were added.

######Example
- force => true
- force => false

#####`backup`
Controls the filebucket behavior used for the file.

######Example
- backup => 'puppet'

#####`replace`
Controls if Puppet should replace the destination file if it already exists.

######Example
- replace => true
- replace => false

#####`order`
Controls the way in which the shell script chooses to sort the files.  It's
rare you'll need to adjust this.

######Allowed Values
- order => 'alpha'
- order => 'numeric'

#####`ensure_newline`
Ensure there's a newline at the end of the fragments.

######Example
- ensure_newline => true
- ensure_newline => false

####concat::fragment

#####`target`
Choose the destination file of the fragment.

######Example
- target => '/tmp/testfile'

#####`content`
Create the content of the fragment.

######Example
- content => 'test file contents'

#####`source`
Find the sources within Puppet of the fragment.

######Example
- source => 'puppet:///modules/test/testfile'
- source => ['puppet:///modules/test/1', 'puppet:///modules/test/2']

#####`order`
Order the fragments.

######Example
- order => '01'

Best practice is to pass a string to this parameter but integer values are accepted.

#####`ensure`
Control the file of fragment created.

######Example
- ensure => 'present'
- ensure => 'absent'

#####`mode`
Set the mode of the fragment.

######Example
- mode => '0644'

#####`owner`
Set the owner of the fragment.

######Example
- owner => 'root'

#####`group`
Set the group of the fragment.

######Example
- group => 'root'

#####`backup`
Control the filebucket behavior for the fragment.

######Example
- backup => 'puppet'

### API _deprecations_

#### Since version `1.0.0`

##### `concat{}` `warn` parameter

```puppet
concat { '/tmp/file':
  ensure => present,
  warn   => 'true',  # generates stringified boolean value warning
}
```

Using stringified Boolean values as the `warn` parameter to `concat` is
deprecated, generates a catalog compile time warning, and will be silently
treated as the concatenated file header/warning message in a future release.

The following strings are considered a stringified Boolean value:

 * `'true'`
 * `'yes'`
 * `'on'`
 * `'false'`
 * `'no'`
 * `'off'`

Please migrate to using the Puppet DSL's native [Boolean data
type](http://docs.puppetlabs.com/puppet/3/reference/lang_datatypes.html#booleans).

##### `concat{}` `gnu` parameter

```puppet
concat { '/tmp/file':
  ensure => present,
  gnu    => $foo,    # generates deprecation warning
}
```

The `gnu` parameter to `concat` is deprecated, generates a catalog compile time
warning, and has no effect.  This parameter will be removed in a future
release.

Note that this parameter was silently ignored in the `1.0.0` release.

##### `concat::fragment{}` `ensure` parameter

```puppet
concat::fragment { 'cpuinfo':
  ensure => '/proc/cpuinfo', # generates deprecation warning
  target => '/tmp/file',
}
```

Passing a value other than `'present'` or `'absent'` as the `ensure` parameter
to `concat::fragment` is deprecated and generates a catalog compile time
warning.  The warning will become a catalog compilation failure in a future
release.

This type emulates the Puppet core `file` type's disfavored [`ensure`
semantics](http://docs.puppetlabs.com/references/latest/type.html#file-attribute-ensure)
of treating a file path as a directive to create a symlink.  This feature is
problematic in several ways.  It copies an API semantic of another type that is
both frowned upon and not generally well known.  It's behavior may be
surprising in that the target concatenated file will not be a symlink nor is
there any common file system that has a concept of a section of a plain file
being symbolically linked to another file.  Additionally, the behavior is
generally inconsistent with most Puppet types in that a missing source file
will be silently ignored.

If you want to use the content of a file as a fragment please use the `source`
parameter.

##### `concat::fragment{}` `mode/owner/group` parameters

```puppet
concat::fragment { 'foo':
  target  => '/tmp/file',
  content => 'foo',
  mode    => $mode,       # generates deprecation warning
  owner   => $owner,      # generates deprecation warning
  group   => $group,      # generates deprecation warning
}
```

The `mode` parameter to `concat::fragment` is deprecated, generates a catalog compile time warning, and has no effect.

The `owner` parameter to `concat::fragment` is deprecated, generates a catalog
compile time warning, and has no effect.

The `group` parameter to `concat::fragment` is deprecated, generates a catalog
compile time warning, and has no effect.

These parameters had no user visible effect in version `1.0.0` and will be
removed in a future release.

##### `concat::fragment{}` `backup` parameter

```puppet
concat::fragment { 'foo':
  target  => '/tmp/file',
  content => 'foo',
  backup  => 'bar',       # generates deprecation warning
}
```

The `backup` parameter to `concat::fragment` is deprecated, generates a catalog
compile time warning, and has no effect.  It will be removed in a future
release.

In the `1.0.0` release this parameter controlled file bucketing of the file
fragment.  Bucketting the fragment(s) is redundant with bucketting the final
concatenated file and this feature has been removed.

##### `class { 'concat::setup': }`

```puppet
include concat::setup     # generates deprecation warning

class { 'concat::setup': } # generates deprecation warning
```

The `concat::setup` class is deprecated as a public API of this module and
should no longer be directly included in the manifest.  This class may be
removed in a future release.

##### Parameter validation

While not an API depreciation, users should be aware that all public parameters
in this module are now validated for at least variable type.  This may cause
validation errors in a manifest that was previously silently misbehaving.

##Limitations

This module has been tested on:

* RedHat Enterprise Linux (and Centos) 5/6
* Debian 6/7
* Ubuntu 12.04

Testing on other platforms has been light and cannot be guaranteed.

#Development

Puppet Labs modules on the Puppet Forge are open projects, and community
contributions are essential for keeping them great. We can’t access the
huge number of platforms and myriad of hardware, software, and deployment
configurations that Puppet is intended to serve.

We want to keep it as easy as possible to contribute changes so that our
modules work in your environment. There are a few guidelines that we need
contributors to follow so that we can have a chance of keeping on top of things.

You can read the complete module contribution guide [on the Puppet Labs wiki.](http://projects.puppetlabs.com/projects/module-site/wiki/Module_contributing)

###Contributors

The list of contributors can be found at:

https://github.com/puppetlabs/puppetlabs-concat/graphs/contributors
