// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.


'use strict';

// TODO(crbug.com/937570): After the RP launches this should be renamed to
// customizationMenu along with the file, and large parts can be refactored/removed.
const customBackgrounds = {};

/**
 * The browser embeddedSearch.newTabPage object.
 * @type {Object}
 */
let ntpApiHandle;

/**
 * The different types of events that are logged from the NTP. This enum is
 * used to transfer information from the NTP JavaScript to the renderer and is
 * not used as a UMA enum histogram's logged value.
 * Note: Keep in sync with common/ntp_logging_events.h
 * @enum {number}
 * @const
 */
const BACKGROUND_CUSTOMIZATION_LOG_TYPE = {
  // The 'Chrome backgrounds' menu item was clicked.
  NTP_CUSTOMIZE_CHROME_BACKGROUNDS_CLICKED: 40,
  // The 'Upload an image' menu item was clicked.
  NTP_CUSTOMIZE_LOCAL_IMAGE_CLICKED: 41,
  // The 'Restore default background' menu item was clicked.
  NTP_CUSTOMIZE_RESTORE_BACKGROUND_CLICKED: 42,
  // The attribution link on a customized background image was clicked.
  NTP_CUSTOMIZE_ATTRIBUTION_CLICKED: 43,
  // The 'Restore default shortcuts' menu item was clicked.
  NTP_CUSTOMIZE_RESTORE_SHORTCUTS_CLICKED: 46,
  // A collection was selected in the 'Chrome backgrounds' dialog.
  NTP_CUSTOMIZE_CHROME_BACKGROUND_SELECT_COLLECTION: 47,
  // An image was selected in the 'Chrome backgrounds' dialog.
  NTP_CUSTOMIZE_CHROME_BACKGROUND_SELECT_IMAGE: 48,
  // 'Cancel' was clicked in the 'Chrome backgrounds' dialog.
  NTP_CUSTOMIZE_CHROME_BACKGROUND_CANCEL: 49,
  // NOTE: NTP_CUSTOMIZE_CHROME_BACKGROUND_DONE (50) is logged on the backend
  // when the selected image is saved.
  // 'Cancel' was clicked in the 'Upload an image' dialog.
  NTP_CUSTOMIZE_LOCAL_IMAGE_CANCEL: 51,
  // 'Done' was clicked in the 'Upload an image' dialog.
  NTP_CUSTOMIZE_LOCAL_IMAGE_DONE: 52,
};

/**
 * Enum for key codes.
 * @enum {number}
 * @const
 */
customBackgrounds.KEYCODES = {
  BACKSPACE: 8,
  DOWN: 40,
  ENTER: 13,
  ESC: 27,
  LEFT: 37,
  RIGHT: 39,
  SPACE: 32,
  TAB: 9,
  UP: 38,
};

/**
 * Enum for HTML element ids.
 * @enum {string}
 * @const
 */
customBackgrounds.IDS = {
  ATTR1: 'attr1',
  ATTR2: 'attr2',
  ATTRIBUTIONS: 'custom-bg-attr',
  BACK_CIRCLE: 'bg-sel-back-circle',
  BACKGROUNDS_DEFAULT: 'backgrounds-default',
  BACKGROUNDS_DEFAULT_ICON: 'backgrounds-default-icon',
  BACKGROUNDS_BUTTON: 'backgrounds-button',
  BACKGROUNDS_IMAGE_MENU: 'backgrounds-image-menu',
  BACKGROUNDS_MENU: 'backgrounds-menu',
  BACKGROUNDS_UPLOAD: 'backgrounds-upload',
  CANCEL: 'bg-sel-footer-cancel',
  COLORS_BUTTON: 'colors-button',
  COLORS_MENU: 'colors-menu',
  CUSTOMIZATION_MENU: 'customization-menu',
  CUSTOM_LINKS_RESTORE_DEFAULT: 'custom-links-restore-default',
  CUSTOM_LINKS_RESTORE_DEFAULT_TEXT: 'custom-links-restore-default-text',
  DEFAULT_WALLPAPERS: 'edit-bg-default-wallpapers',
  DEFAULT_WALLPAPERS_TEXT: 'edit-bg-default-wallpapers-text',
  DONE: 'bg-sel-footer-done',
  EDIT_BG: 'edit-bg',
  EDIT_BG_DIALOG: 'edit-bg-dialog',
  EDIT_BG_DIVIDER: 'edit-bg-divider',
  EDIT_BG_ICON: 'edit-bg-icon',
  EDIT_BG_MENU: 'edit-bg-menu',
  EDIT_BG_TEXT: 'edit-bg-text',
  MENU_BACK_CIRCLE: 'menu-back-circle',
  MENU_CANCEL: 'menu-cancel',
  MENU_DONE: 'menu-done',
  MENU_TITLE: 'menu-title',
  LINK_ICON: 'link-icon',
  MENU: 'bg-sel-menu',
  OPTIONS_TITLE: 'edit-bg-title',
  RESTORE_DEFAULT: 'edit-bg-restore-default',
  RESTORE_DEFAULT_TEXT: 'edit-bg-restore-default-text',
  SHORTCUTS_BUTTON: 'shortcuts-button',
  SHORTCUTS_MENU: 'shortcuts-menu',
  UPLOAD_IMAGE: 'edit-bg-upload-image',
  UPLOAD_IMAGE_TEXT: 'edit-bg-upload-image-text',
  TILES: 'bg-sel-tiles',
  TITLE: 'bg-sel-title',
};

/**
 * Enum for classnames.
 * @enum {string}
 * @const
 */
customBackgrounds.CLASSES = {
  ATTR_SMALL: 'attr-small',
  ATTR_COMMON: 'attr-common',
  ATTR_LINK: 'attr-link',
  COLLECTION_DIALOG: 'is-col-sel',
  COLLECTION_SELECTED: 'bg-selected',  // Highlight selected tile
  COLLECTION_TILE: 'bg-sel-tile',  // Preview tile for background customization
  COLLECTION_TILE_BG: 'bg-sel-tile-bg',
  COLLECTION_TITLE: 'bg-sel-tile-title',  // Title of a background image
  // Extended and elevated style for entry point.
  ENTRY_POINT_ENHANCED: 'ep-enhanced',
  IMAGE_DIALOG: 'is-img-sel',
  ON_IMAGE_MENU: 'on-img-menu',
  OPTION: 'bg-option',
  OPTION_DISABLED: 'bg-option-disabled',  // The menu option is disabled.
  MENU_SHOWN: 'menu-shown',
  MOUSE_NAV: 'using-mouse-nav',
  SELECTED: 'selected',
  SELECTED_BORDER: 'selected-border',
  SELECTED_CHECK: 'selected-check',
  SELECTED_CIRCLE: 'selected-circle',
  SINGLE_ATTR: 'single-attr'
};

/**
 * Enum for background sources.
 * @enum {number}
 * @const
 */
customBackgrounds.SOURCES = {
  NONE: -1,
  CHROME_BACKGROUNDS: 0,
  IMAGE_UPLOAD: 1,
};

/**
 * Enum for background option menu entries, in the order they appear in the UI.
 * @enum {number}
 * @const
 */
customBackgrounds.MENU_ENTRIES = {
  CHROME_BACKGROUNDS: 0,
  UPLOAD_IMAGE: 1,
  CUSTOM_LINKS_RESTORE_DEFAULT: 2,
  RESTORE_DEFAULT: 3,
};

customBackgrounds.CUSTOM_BACKGROUND_OVERLAY =
    'linear-gradient(rgba(0, 0, 0, 0), rgba(0, 0, 0, 0.3))';

// These shound match the corresponding values in local_ntp.js, that control the
// mv-notice element.
customBackgrounds.delayedHideNotification = -1;
customBackgrounds.NOTIFICATION_TIMEOUT = 10000;

/* Were the background tiles already created.
 * @type {bool}
 */
customBackgrounds.builtTiles = false;

/* Tile that was selected by the user.
 * @type {HTMLElement}
 */
customBackgrounds.selectedTile = null;

/**
 * Number of rows in the custom background dialog to preload.
 * @type {number}
 * @const
 */
customBackgrounds.ROWS_TO_PRELOAD = 3;

/* Type of collection that is being browsed, needed in order
 * to return from the image dialog.
 * @type {number}
 */
customBackgrounds.dialogCollectionsSource = customBackgrounds.SOURCES.NONE;

/*
 * Called when the error notification should be shown.
 * @type {?Function}
 * @private
 */
customBackgrounds.showErrorNotification = null;

/*
 * Called when the custom link notification should be hidden.
 * @type {?Function}
 * @private
 */
customBackgrounds.hideCustomLinkNotification = null;

/*
 * The currently selected option in the richer picker.
 * @type {?Element}
 * @private
 */
customBackgrounds.richerPicker_selectedOption = null;

/**
 * Sets the visibility of the settings menu and individual options depending on
 * their respective features.
 */
customBackgrounds.setMenuVisibility = function() {
  // Reset all hidden values.
  $(customBackgrounds.IDS.EDIT_BG).hidden = false;
  $(customBackgrounds.IDS.DEFAULT_WALLPAPERS).hidden = false;
  $(customBackgrounds.IDS.UPLOAD_IMAGE).hidden = false;
  $(customBackgrounds.IDS.RESTORE_DEFAULT).hidden = false;
  $(customBackgrounds.IDS.EDIT_BG_DIVIDER).hidden = false;
  $(customBackgrounds.IDS.CUSTOM_LINKS_RESTORE_DEFAULT).hidden =
      configData.hideShortcuts;
  $(customBackgrounds.IDS.COLORS_BUTTON).hidden = !configData.chromeColors;
};

/**
 * Display custom background image attributions on the page.
 * @param {string} attributionLine1 First line of attribution.
 * @param {string} attributionLine2 Second line of attribution.
 * @param {string} attributionActionUrl Url to learn more about the image.
 */
customBackgrounds.setAttribution = function(
    attributionLine1, attributionLine2, attributionActionUrl) {
  const attributionBox = $(customBackgrounds.IDS.ATTRIBUTIONS);
  const attr1 = document.createElement('span');
  attr1.id = customBackgrounds.IDS.ATTR1;
  const attr2 = document.createElement('span');
  attr2.id = customBackgrounds.IDS.ATTR2;

  if (attributionLine1 !== '') {
    // Shouldn't be changed from textContent for security assurances.
    attr1.textContent = attributionLine1;
    attr1.classList.add(customBackgrounds.CLASSES.ATTR_COMMON);
    $(customBackgrounds.IDS.ATTRIBUTIONS).appendChild(attr1);
  }
  if (attributionLine2 !== '') {
    // Shouldn't be changed from textContent for security assurances.
    attr2.textContent = attributionLine2;
    attr2.classList.add(customBackgrounds.CLASSES.ATTR_SMALL);
    attr2.classList.add(customBackgrounds.CLASSES.ATTR_COMMON);
    attributionBox.appendChild(attr2);
  }
  if (attributionActionUrl !== '') {
    const attr = (attributionLine2 !== '' ? attr2 : attr1);
    attr.classList.add(customBackgrounds.CLASSES.ATTR_LINK);

    const linkIcon = document.createElement('div');
    linkIcon.id = customBackgrounds.IDS.LINK_ICON;
    // Enlarge link-icon when there is only one line of attribution
    if (attributionLine2 === '') {
      linkIcon.classList.add(customBackgrounds.CLASSES.SINGLE_ATTR);
    }
    attr.insertBefore(linkIcon, attr.firstChild);

    attributionBox.classList.add(customBackgrounds.CLASSES.ATTR_LINK);
    attributionBox.href = attributionActionUrl;
    attributionBox.onclick = function() {
      ntpApiHandle.logEvent(
          BACKGROUND_CUSTOMIZATION_LOG_TYPE.NTP_CUSTOMIZE_ATTRIBUTION_CLICKED);
    };
    attributionBox.style.cursor = 'pointer';
  }
};

customBackgrounds.clearAttribution = function() {
  const attributions = $(customBackgrounds.IDS.ATTRIBUTIONS);
  attributions.removeAttribute('href');
  attributions.className = '';
  attributions.style.cursor = 'none';
  while (attributions.firstChild) {
    attributions.removeChild(attributions.firstChild);
  }
};

customBackgrounds.unselectTile = function() {
  $(customBackgrounds.IDS.DONE).disabled = true;
  customBackgrounds.selectedTile = null;
  $(customBackgrounds.IDS.DONE).tabIndex = -1;
};

/**
 * Remove all collection tiles from the container when the dialog
 * is closed.
 */
customBackgrounds.resetSelectionDialog = function() {
  $(customBackgrounds.IDS.TILES).scrollTop = 0;
  const tileContainer = $(customBackgrounds.IDS.TILES);
  while (tileContainer.firstChild) {
    tileContainer.removeChild(tileContainer.firstChild);
  }
  customBackgrounds.unselectTile();
};

/**
 * Apply selected styling to |button| and make corresponding |menu| visible.
 * @param {?Element} button The button element to apply styling to.
 * @param {?Element} menu The menu element to apply styling to.
 */
customBackgrounds.richerPicker_selectMenuOption = function(button, menu) {
  if (!button || !menu) {
    return;
  }
  button.classList.toggle(customBackgrounds.CLASSES.SELECTED, true);
  customBackgrounds.richerPicker_selectedOption = button;
  menu.classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, true);
};

/**
 * Remove image tiles and maybe swap back to main background menu.
 * @param {boolean} showMenu Whether the main background menu should be shown.
 */
customBackgrounds.richerPicker_resetImageMenu = function(showMenu) {
  const backgroundMenu = $(customBackgrounds.IDS.BACKGROUNDS_MENU);
  const imageMenu = $(customBackgrounds.IDS.BACKGROUNDS_IMAGE_MENU);
  const menu = $(customBackgrounds.IDS.CUSTOMIZATION_MENU);
  const menuTitle = $(customBackgrounds.IDS.MENU_TITLE);

  imageMenu.innerHTML = '';
  imageMenu.classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, false);
  menuTitle.textContent = menuTitle.dataset.mainTitle;
  menu.classList.toggle(customBackgrounds.CLASSES.ON_IMAGE_MENU, false);
  backgroundMenu.classList.toggle(
      customBackgrounds.CLASSES.MENU_SHOWN, showMenu);
  backgroundMenu.scrollTop = 0;

  // Reset done button state.
  $(customBackgrounds.IDS.MENU_DONE).disabled = true;
  customBackgrounds.richerPicker_deselectTile(customBackgrounds.selectedTile);
  customBackgrounds.selectedTile = null;
  $(customBackgrounds.IDS.MENU_DONE).tabIndex = -1;
};

/* Close the collection selection dialog and cleanup the state
 * @param {dialog} menu The dialog to be closed
 */
customBackgrounds.closeCollectionDialog = function(menu) {
  menu.close();
  customBackgrounds.dialogCollectionsSource = customBackgrounds.SOURCES.NONE;
  customBackgrounds.resetSelectionDialog();
};

/* Close and reset the dialog, and set the background.
 * @param {string} url The url of the selected background.
 */
customBackgrounds.setBackground = function(
    url, attributionLine1, attributionLine2, attributionActionUrl) {
  if (configData.richerPicker) {
    $(customBackgrounds.IDS.CUSTOMIZATION_MENU).close();
    customBackgrounds.richerPicker_resetImageMenu(false);
  } else {
    customBackgrounds.closeCollectionDialog($(customBackgrounds.IDS.MENU));
  }
  window.chrome.embeddedSearch.newTabPage.setBackgroundURLWithAttributions(
      url, attributionLine1, attributionLine2, attributionActionUrl);
};

/**
 * Create a tile for a Chrome Backgrounds collection.
 */
customBackgrounds.createChromeBackgroundTile = function(data) {
  const tile = document.createElement('div');
  tile.style.backgroundImage = 'url(' + data.previewImageUrl + ')';
  tile.dataset.id = data.collectionId;
  tile.dataset.name = data.collectionName;
  fadeInImageTile(tile, data.previewImageUrl, null);
  return tile;
};

/**
 * Get the number of tiles in a row according to current window width.
 * @return {number} the number of tiles per row
 */
customBackgrounds.getTilesWide = function() {
  // Browser window can only fit two columns. Should match "#bg-sel-menu" width.
  if ($(customBackgrounds.IDS.MENU).offsetWidth < 517) {
    return 2;
  } else if ($(customBackgrounds.IDS.MENU).offsetWidth < 356) {
    // Browser window can only fit one column. Should match @media (max-width:
    // 356) "#bg-sel-menu" width.
    return 1;
  }

  return 3;
};

/* Get the next tile when the arrow keys are used to navigate the grid.
 * Returns null if the tile doesn't exist.
 * @param {number} deltaX Change in the x direction.
 * @param {number} deltaY Change in the y direction.
 * @param {string} current Number of the current tile.
 */
customBackgrounds.getNextTile = function(deltaX, deltaY, current) {
  let idPrefix = 'coll_tile_';
  if ($(customBackgrounds.IDS.MENU)
          .classList.contains(customBackgrounds.CLASSES.IMAGE_DIALOG)) {
    idPrefix = 'img_tile_';
  }

  if (deltaX != 0) {
    const target = parseInt(current, /*radix=*/ 10) + deltaX;
    return $(idPrefix + target);
  } else if (deltaY != 0) {
    let target = parseInt(current, /*radix=*/ 10);
    let nextTile = $(idPrefix + target);
    const startingTop = nextTile.getBoundingClientRect().top;
    const startingLeft = nextTile.getBoundingClientRect().left;

    // Search until a tile in a different row and the same column is found.
    while (nextTile &&
           (nextTile.getBoundingClientRect().top == startingTop ||
            nextTile.getBoundingClientRect().left != startingLeft)) {
      target += deltaY;
      nextTile = $(idPrefix + target);
    }
    return nextTile;
  }
};

/**
 * Show dialog for selecting a Chrome background.
 * @param {number} collectionsSource The enum value of the source to fetch
 *              collection data from.
 */
customBackgrounds.showCollectionSelectionDialog = function(collectionsSource) {
  const tileContainer = configData.richerPicker ?
      $(customBackgrounds.IDS.BACKGROUNDS_MENU) :
      $(customBackgrounds.IDS.TILES);
  if (configData.richerPicker && customBackgrounds.builtTiles) {
    return;
  }
  customBackgrounds.builtTiles = true;
  const menu = configData.richerPicker ?
      $(customBackgrounds.IDS.CUSTOMIZATION_MENU) :
      $(customBackgrounds.IDS.MENU);
  if (collectionsSource != customBackgrounds.SOURCES.CHROME_BACKGROUNDS) {
    console.log(
        'showCollectionSelectionDialog() called with invalid source=' +
        collectionsSource);
    return;
  }
  customBackgrounds.dialogCollectionsSource = collectionsSource;

  if (!menu.open) {
    menu.showModal();
  }

  // Create dialog header.
  $(customBackgrounds.IDS.TITLE).textContent =
      configData.translatedStrings.selectChromeWallpaper;
  if (!configData.richerPicker) {
    menu.classList.toggle(customBackgrounds.CLASSES.COLLECTION_DIALOG);
    menu.classList.remove(customBackgrounds.CLASSES.IMAGE_DIALOG);
  }

  // Create dialog tiles.
  for (let i = 0; i < coll.length; ++i) {
    const tileBackground = document.createElement('div');
    tileBackground.classList.add(
        customBackgrounds.CLASSES.COLLECTION_TILE_BG);
    const tile = customBackgrounds.createChromeBackgroundTile(coll[i]);
    tile.classList.add(customBackgrounds.CLASSES.COLLECTION_TILE);
    tile.id = 'coll_tile_' + i;
    tile.dataset.tile_num = i;
    tile.tabIndex = -1;
    // Accessibility support for screen readers.
    tile.setAttribute('role', 'button');

    const title = document.createElement('div');
    title.classList.add(customBackgrounds.CLASSES.COLLECTION_TITLE);
    title.textContent = tile.dataset.name;

    const tileInteraction = function(event) {
      let tile = event.target;
      if (tile.classList.contains(customBackgrounds.CLASSES.COLLECTION_TITLE)) {
        tile = tile.parentNode;
      }

      // Load images for selected collection.
      const imgElement = $('ntp-images-loader');
      if (imgElement) {
        imgElement.parentNode.removeChild(imgElement);
      }
      const imgScript = document.createElement('script');
      imgScript.id = 'ntp-images-loader';
      imgScript.src = 'chrome-search://local-ntp/ntp-background-images.js?' +
          'collection_id=' + tile.dataset.id;
      ntpApiHandle.logEvent(
          BACKGROUND_CUSTOMIZATION_LOG_TYPE
              .NTP_CUSTOMIZE_CHROME_BACKGROUND_SELECT_COLLECTION);

      document.body.appendChild(imgScript);

      imgScript.onload = function() {
        // Verify that the individual image data was successfully loaded.
        const imageDataLoaded =
            (collImg.length > 0 && collImg[0].collectionId == tile.dataset.id);

        // Dependent upon the success of the load, populate the image selection
        // dialog or close the current dialog.
        if (imageDataLoaded) {
          $(customBackgrounds.IDS.BACKGROUNDS_MENU)
              .classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, false);
          $(customBackgrounds.IDS.BACKGROUNDS_IMAGE_MENU)
              .classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, true);

          // In the RP the upload or default tile may be selected.
          if (configData.richerPicker) {
            customBackgrounds.richerPicker_deselectTile(
                customBackgrounds.selectedTile);
          } else {
            customBackgrounds.resetSelectionDialog();
          }
          customBackgrounds.showImageSelectionDialog(tile.dataset.name);
        } else {
          customBackgrounds.handleError(collImgErrors);
        }
      };
    };

    tile.onclick = tileInteraction;
    tile.onkeydown = function(event) {
      if (event.keyCode === customBackgrounds.KEYCODES.ENTER) {
        event.preventDefault();
        event.stopPropagation();
        tileInteraction(event);
      } else if (
          event.keyCode === customBackgrounds.KEYCODES.LEFT ||
          event.keyCode === customBackgrounds.KEYCODES.UP ||
          event.keyCode === customBackgrounds.KEYCODES.RIGHT ||
          event.keyCode === customBackgrounds.KEYCODES.DOWN) {
        // Handle arrow key navigation.
        event.preventDefault();
        event.stopPropagation();

        let target = null;
        if (event.keyCode === customBackgrounds.KEYCODES.LEFT) {
          target = customBackgrounds.getNextTile(
              document.documentElement.classList.contains('rtl') ? 1 : -1, 0,
              event.currentTarget.dataset.tile_num);
        } else if (event.keyCode === customBackgrounds.KEYCODES.UP) {
          target = customBackgrounds.getNextTile(
              0, -1, event.currentTarget.dataset.tile_num);
        } else if (event.keyCode === customBackgrounds.KEYCODES.RIGHT) {
          target = customBackgrounds.getNextTile(
              document.documentElement.classList.contains('rtl') ? -1 : 1, 0,
              event.currentTarget.dataset.tile_num);
        } else if (event.keyCode === customBackgrounds.KEYCODES.DOWN) {
          target = customBackgrounds.getNextTile(
              0, 1, event.currentTarget.dataset.tile_num);
        }
        if (target) {
          target.focus();
        } else {
          event.currentTarget.focus();
        }
      }
    };

    tile.appendChild(title);
    tileBackground.appendChild(tile);
    tileContainer.appendChild(tileBackground);
  }

  $(customBackgrounds.IDS.TILES).focus();
};

/**
 * Apply styling to a selected tile in the richer picker and enable the done
 * button.
 * @param {!Element} tile The tile to apply styling to.
 */
customBackgrounds.richerPicker_selectTile = function(tile) {
  tile.parentElement.classList.toggle(customBackgrounds.CLASSES.SELECTED, true);
  $(customBackgrounds.IDS.MENU_DONE).disabled = false;
  customBackgrounds.selectedTile = tile;
  $(customBackgrounds.IDS.MENU_DONE).tabIndex = 0;

  // Create and append selected check.
  const selectedCircle = document.createElement('div');
  const selectedCheck = document.createElement('div');
  selectedCircle.classList.add(customBackgrounds.CLASSES.SELECTED_CIRCLE);
  selectedCheck.classList.add(customBackgrounds.CLASSES.SELECTED_CHECK);
  selectedCircle.appendChild(selectedCheck);
  tile.appendChild(selectedCircle);
};

/**
 * Remove styling from a selected tile in the richer picker and disable the
 * done button.
 * @param {?Element} tile The tile to remove styling from.
 */
customBackgrounds.richerPicker_deselectTile = function(tile) {
  if (tile === null) {
    return;
  }
  tile.parentElement.classList.toggle(
      customBackgrounds.CLASSES.SELECTED, false);
  $(customBackgrounds.IDS.MENU_DONE).disabled = true;
  customBackgrounds.selectedTile = null;
  $(customBackgrounds.IDS.MENU_DONE).tabIndex = -1;

  // Remove selected check and circle.
  for (let i = 0; i < tile.children.length; ++i) {
    if (tile.children[i].classList.contains(
            customBackgrounds.CLASSES.SELECTED_CHECK) ||
        tile.children[i].classList.contains(
            customBackgrounds.CLASSES.SELECTED_CIRCLE)) {
      tile.removeChild(tile.children[i]);
      --i;
    }
  }
};


/**
 * Apply border and checkmark when a tile is selected
 * @param {!Element} tile The tile to apply styling to.
 */
customBackgrounds.applySelectedState = function(tile) {
  tile.classList.add(customBackgrounds.CLASSES.COLLECTION_SELECTED);
  const selectedBorder = document.createElement('div');
  const selectedCircle = document.createElement('div');
  const selectedCheck = document.createElement('div');
  selectedBorder.classList.add(customBackgrounds.CLASSES.SELECTED_BORDER);
  selectedCircle.classList.add(customBackgrounds.CLASSES.SELECTED_CIRCLE);
  selectedCheck.classList.add(customBackgrounds.CLASSES.SELECTED_CHECK);
  selectedBorder.appendChild(selectedCircle);
  selectedBorder.appendChild(selectedCheck);
  tile.appendChild(selectedBorder);
  tile.dataset.oldLabel = tile.getAttribute('aria-label');
  tile.setAttribute(
      'aria-label',
      tile.dataset.oldLabel + ' ' + configData.translatedStrings.selectedLabel);
};

/**
 * Remove border and checkmark when a tile is un-selected
 * @param {!Element} tile The tile to remove styling from.
 */
customBackgrounds.removeSelectedState = function(tile) {
  tile.classList.remove(customBackgrounds.CLASSES.COLLECTION_SELECTED);
  tile.removeChild(tile.firstChild);
  tile.setAttribute('aria-label', tile.dataset.oldLabel);
};

/**
 * Show dialog for selecting an image. Image data should previous have been
 * loaded into collImg via
 * chrome-search://local-ntp/ntp-background-images.js?collection_id=<collection_id>
 * @param {string} dialogTitle The title to be displayed at the top of the
 *                 dialog.
 */
customBackgrounds.showImageSelectionDialog = function(dialogTitle) {
  const firstNTile = customBackgrounds.ROWS_TO_PRELOAD
      * customBackgrounds.getTilesWide();
  const tileContainer = configData.richerPicker ?
      $(customBackgrounds.IDS.BACKGROUNDS_IMAGE_MENU) :
      $(customBackgrounds.IDS.TILES);
  const menu = configData.richerPicker ?
      $(customBackgrounds.IDS.CUSTOMIZATION_MENU) :
      $(customBackgrounds.IDS.MENU);

  if (configData.richerPicker) {
    $(customBackgrounds.IDS.MENU_TITLE).textContent = dialogTitle;
    menu.classList.toggle(customBackgrounds.CLASSES.ON_IMAGE_MENU, true);
  } else {
    $(customBackgrounds.IDS.TITLE).textContent = dialogTitle;
    menu.classList.remove(customBackgrounds.CLASSES.COLLECTION_DIALOG);
    menu.classList.add(customBackgrounds.CLASSES.IMAGE_DIALOG);
  }

  const preLoadTiles = [];
  const postLoadTiles = [];

  for (let i = 0; i < collImg.length; ++i) {
    const tileBackground = document.createElement('div');
    tileBackground.classList.add(
        customBackgrounds.CLASSES.COLLECTION_TILE_BG);
    const tile = document.createElement('div');
    tile.classList.add(customBackgrounds.CLASSES.COLLECTION_TILE);
    // Accessibility support for screen readers.
    tile.setAttribute('role', 'button');

    // TODO(crbug.com/854028): Remove this hardcoded check when wallpaper
    // previews are supported.
    if (collImg[i].collectionId === 'solidcolors') {
      tile.dataset.attributionLine1 = '';
      tile.dataset.attributionLine2 = '';
      tile.dataset.attributionActionUrl = '';
    } else {
      tile.dataset.attributionLine1 =
          (collImg[i].attributions[0] !== undefined ?
               collImg[i].attributions[0] :
               '');
      tile.dataset.attributionLine2 =
          (collImg[i].attributions[1] !== undefined ?
               collImg[i].attributions[1] :
               '');
      tile.dataset.attributionActionUrl = collImg[i].attributionActionUrl;
    }
    tile.setAttribute('aria-label', collImg[i].attributions[0]);
    tile.dataset.url = collImg[i].imageUrl;

    tile.id = 'img_tile_' + i;
    tile.dataset.tile_num = i;
    tile.tabIndex = -1;

    // Load the first |ROWS_TO_PRELOAD| rows of tiles.
    if (i < firstNTile) {
      preLoadTiles.push(tile);
    } else {
      postLoadTiles.push(tile);
    }

    const tileInteraction = function(tile) {
      if (customBackgrounds.selectedTile) {
        if (configData.richerPicker) {
          const id = customBackgrounds.selectedTile.id;
          customBackgrounds.richerPicker_deselectTile(
              customBackgrounds.selectedTile);
          if (id === tile.id) {
            return;
          }
        } else {
          customBackgrounds.removeSelectedState(customBackgrounds.selectedTile);
          if (customBackgrounds.selectedTile.id === tile.id) {
            customBackgrounds.unselectTile();
            return;
          }
        }
      }

      if (configData.richerPicker) {
        customBackgrounds.richerPicker_selectTile(tile);
      } else {
        customBackgrounds.applySelectedState(tile);
        customBackgrounds.selectedTile = tile;
      }

      $(customBackgrounds.IDS.DONE).tabIndex = 0;

      // Turn toggle off when an image is selected.
      $(customBackgrounds.IDS.DONE).disabled = false;
      ntpApiHandle.logEvent(BACKGROUND_CUSTOMIZATION_LOG_TYPE
                                .NTP_CUSTOMIZE_CHROME_BACKGROUND_SELECT_IMAGE);
    };

    tile.onclick = function(event) {
      const clickCount = event.detail;
      // Control + option + space will fire the onclick event with 0 clickCount.
      if (clickCount <= 1) {
        tileInteraction(event.currentTarget);
      } else if (
          clickCount === 2 &&
          customBackgrounds.selectedTile === event.currentTarget) {
        customBackgrounds.setBackground(
            event.currentTarget.dataset.url,
            event.currentTarget.dataset.attributionLine1,
            event.currentTarget.dataset.attributionLine2,
            event.currentTarget.dataset.attributionActionUrl);
      }
    };
    tile.onkeydown = function(event) {

      if (event.keyCode === customBackgrounds.KEYCODES.ENTER) {
        event.preventDefault();
        event.stopPropagation();
        tileInteraction(event.currentTarget);
      } else if (
          event.keyCode === customBackgrounds.KEYCODES.LEFT ||
          event.keyCode === customBackgrounds.KEYCODES.UP ||
          event.keyCode === customBackgrounds.KEYCODES.RIGHT ||
          event.keyCode === customBackgrounds.KEYCODES.DOWN) {
        // Handle arrow key navigation.
        event.preventDefault();
        event.stopPropagation();

        let target = null;
        if (event.keyCode == customBackgrounds.KEYCODES.LEFT) {
          target = customBackgrounds.getNextTile(
              document.documentElement.classList.contains('rtl') ? 1 : -1, 0,
              event.currentTarget.dataset.tile_num);
        } else if (event.keyCode == customBackgrounds.KEYCODES.UP) {
          target = customBackgrounds.getNextTile(
              0, -1, event.currentTarget.dataset.tile_num);
        } else if (event.keyCode == customBackgrounds.KEYCODES.RIGHT) {
          target = customBackgrounds.getNextTile(
              document.documentElement.classList.contains('rtl') ? -1 : 1, 0,
              event.currentTarget.dataset.tile_num);
        } else if (event.keyCode == customBackgrounds.KEYCODES.DOWN) {
          target = customBackgrounds.getNextTile(
              0, 1, event.currentTarget.dataset.tile_num);
        }
        if (target) {
          target.focus();
        } else {
          event.currentTarget.focus();
        }
      }
    };

    tileBackground.appendChild(tile);
    tileContainer.appendChild(tileBackground);
  }
  let tileGetsLoaded = 0;
  for (const tile of preLoadTiles) {
    loadTile(tile, collImg, () => {
      // After the preloaded tiles finish loading, the rest of the tiles start
      // loading.
      if (++tileGetsLoaded === preLoadTiles.length) {
        postLoadTiles.forEach((tile) => loadTile(tile, collImg, null));
      }
    });
  }

  $(customBackgrounds.IDS.TILES).focus();
};

/**
 * Add background image src to the tile and add animation for the tile once it
 * successfully loaded.
 * @param {!Object} tile the tile that needs to be loaded.
 * @param {!Object} imageData the source imageData.
 * @param {?Function} countLoad If not null, called after the tile finishes
 * loading.
 */
const loadTile = function(tile, imageData, countLoad) {
  if (imageData[tile.dataset.tile_num].collectionId === 'solidcolors') {
    tile.style.backgroundImage = [customBackgrounds.CUSTOM_BACKGROUND_OVERLAY,
      'url(' + imageData[tile.dataset.tile_num].thumbnailImageUrl + ')'].join(
        ',').trim();
  } else {
    tile.style.backgroundImage =
        'url(' + imageData[tile.dataset.tile_num].thumbnailImageUrl + ')';
  }
  fadeInImageTile(
      tile, imageData[tile.dataset.tile_num].thumbnailImageUrl, countLoad);
};

/**
 * Fade in effect for both collection and image tile. Once the image
 * successfully loads, we can assume the background image with the same source
 * has also loaded. Then, we set opacity for the tile to start the animation.
 * @param {!Object} tile The tile to add the fade in animation to.
 * @param {string} imageUrl the image url for the tile
 * @param {?Function} countLoad If not null, called after the tile finishes
 * loading.
 */
const fadeInImageTile = function(tile, imageUrl, countLoad) {
  const image = new Image();
  image.onload = () => {
    tile.style.opacity = '1';
    if (countLoad) {
      countLoad();
    }
  };
  image.src = imageUrl;
};

/**
 * Load the NTPBackgroundCollections script. It'll create a global
 * variable name "coll" which is a dict of background collections data.
 * @private
 */
customBackgrounds.loadChromeBackgrounds = function() {
  const collElement = $('ntp-collection-loader');
  if (collElement) {
    collElement.parentNode.removeChild(collElement);
  }
  const collScript = document.createElement('script');
  collScript.id = 'ntp-collection-loader';
  collScript.src = 'chrome-search://local-ntp/ntp-background-collections.js?' +
      'collection_type=background';
  collScript.onload = function() {
    if (configData.richerPicker) {
      customBackgrounds.showCollectionSelectionDialog(
          customBackgrounds.SOURCES.CHROME_BACKGROUNDS);
    }
  };
  document.body.appendChild(collScript);
};

/* Close dialog when an image is selected via the file picker. */
customBackgrounds.closeCustomizationDialog = function() {
  if (configData.richerPicker) {
    $(customBackgrounds.IDS.CUSTOMIZATION_MENU).close();
  } else {
    $(customBackgrounds.IDS.EDIT_BG_DIALOG).close();
  }
};

/*
 * Get the next visible option. There are times when various combinations of
 * options are hidden.
 * @param {number} current_index Index of the option the key press occurred on.
 * @param {number} deltaY Direction to search in, -1 for up, 1 for down.
 */
customBackgrounds.getNextOption = function(current_index, deltaY) {
  // Create array corresponding to the menu. Important that this is in the same
  // order as the MENU_ENTRIES enum, so we can index into it.
  const entries = [];
  entries.push($(customBackgrounds.IDS.DEFAULT_WALLPAPERS));
  entries.push($(customBackgrounds.IDS.UPLOAD_IMAGE));
  entries.push($(customBackgrounds.IDS.CUSTOM_LINKS_RESTORE_DEFAULT));
  entries.push($(customBackgrounds.IDS.RESTORE_DEFAULT));

  let idx = current_index;
  do {
    idx = idx + deltaY;
    if (idx === -1) {
      idx = 3;
    }
    if (idx === 4) {
      idx = 0;
    }
  } while (idx !== current_index && (entries[idx].hidden ||
           entries[idx].classList.contains(
               customBackgrounds.CLASSES.OPTION_DISABLED)));
  return entries[idx];
};

/* Hide custom background options based on the network state
 * @param {bool} online The current state of the network
 */
customBackgrounds.networkStateChanged = function(online) {
  $(customBackgrounds.IDS.DEFAULT_WALLPAPERS).hidden = !online;
};

/**
 * Set customization menu to default options (custom backgrounds).
 */
customBackgrounds.richerPicker_setCustomizationMenuToDefaultState = function() {
  customBackgrounds.richerPicker_resetCustomizationMenu();
  $(customBackgrounds.IDS.BACKGROUNDS_MENU)
      .classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, true);
  customBackgrounds.richerPicker_selectedOption =
      $(customBackgrounds.IDS.BACKGROUNDS_BUTTON);
};

/**
 * Resets customization menu options.
 */
customBackgrounds.richerPicker_resetCustomizationMenu = function() {
  customBackgrounds.richerPicker_resetImageMenu(false);
  $(customBackgrounds.IDS.BACKGROUNDS_MENU)
      .classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, false);
  $(customBackgrounds.IDS.SHORTCUTS_MENU)
      .classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, false);
  $(customBackgrounds.IDS.COLORS_MENU)
      .classList.toggle(customBackgrounds.CLASSES.MENU_SHOWN, false);
  if (customBackgrounds.richerPicker_selectedOption) {
    customBackgrounds.richerPicker_selectedOption.classList.toggle(
        customBackgrounds.CLASSES.SELECTED, false);
    customBackgrounds.richerPicker_selectedOption = null;
  }
};

/**
 * Initialize the settings menu, custom backgrounds dialogs, and custom
 * links menu items. Set the text and event handlers for the various
 * elements.
 * @param {!Function} showErrorNotification Called when the error notification
 *                    should be displayed.
 * @param {!Function} hideCustomLinkNotification Called when the custom link
 *                    notification should be hidden.
 */
customBackgrounds.init = function(
    showErrorNotification, hideCustomLinkNotification) {
  ntpApiHandle = window.chrome.embeddedSearch.newTabPage;
  const editDialog = $(customBackgrounds.IDS.EDIT_BG_DIALOG);
  const menu = $(customBackgrounds.IDS.MENU);

  $(customBackgrounds.IDS.OPTIONS_TITLE).textContent =
      configData.translatedStrings.customizeBackground;

  // Store the main menu title so it can be restored if needed.
  $(customBackgrounds.IDS.MENU_TITLE).dataset.mainTitle =
      $(customBackgrounds.IDS.MENU_TITLE).textContent;

  $(customBackgrounds.IDS.EDIT_BG_ICON)
      .setAttribute(
          'aria-label', configData.translatedStrings.customizeThisPage);

  $(customBackgrounds.IDS.EDIT_BG_ICON)
      .setAttribute('title', configData.translatedStrings.customizeBackground);

  // Edit gear icon interaction events.
  const editBackgroundInteraction = function() {
    if (configData.richerPicker) {
      customBackgrounds.richerPicker_setCustomizationMenuToDefaultState();
      customBackgrounds.loadChromeBackgrounds();
      $(customBackgrounds.IDS.CUSTOMIZATION_MENU).showModal();
    } else {
      editDialog.showModal();
    }
  };
  $(customBackgrounds.IDS.EDIT_BG).onclick = function(event) {
    editDialog.classList.add(customBackgrounds.CLASSES.MOUSE_NAV);
    editBackgroundInteraction();
  };

  $(customBackgrounds.IDS.MENU_CANCEL).onclick = function(event) {
    $(customBackgrounds.IDS.CUSTOMIZATION_MENU).close();
    customBackgrounds.richerPicker_resetCustomizationMenu();
  };


  // Find the first menu option that is not hidden or disabled.
  const findFirstMenuOption = () => {
    const editMenu = $(customBackgrounds.IDS.EDIT_BG_MENU);
    for (let i = 1; i < editMenu.children.length; i++) {
      const option = editMenu.children[i];
      if (option.classList.contains(customBackgrounds.CLASSES.OPTION)
          && !option.hidden && !option.classList.contains(
              customBackgrounds.CLASSES.OPTION_DISABLED)) {
        option.focus();
        return;
      }
    }
  };

  $(customBackgrounds.IDS.EDIT_BG).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER ||
        event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      // no default behavior for ENTER
      event.preventDefault();
      editDialog.classList.remove(customBackgrounds.CLASSES.MOUSE_NAV);
      editBackgroundInteraction();
      findFirstMenuOption();
    }
  };

  // Interactions to close the customization option dialog.
  const editDialogInteraction = function() {
    editDialog.close();
  };
  editDialog.onclick = function(event) {
    editDialog.classList.add(customBackgrounds.CLASSES.MOUSE_NAV);
    if (event.target === editDialog) {
      editDialogInteraction();
    }
  };
  editDialog.onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ESC) {
      editDialogInteraction();
    } else if (
        editDialog.classList.contains(customBackgrounds.CLASSES.MOUSE_NAV) &&
        (event.keyCode === customBackgrounds.KEYCODES.TAB ||
         event.keyCode === customBackgrounds.KEYCODES.UP ||
         event.keyCode === customBackgrounds.KEYCODES.DOWN)) {
      // When using tab in mouse navigation mode, select the first option
      // available.
      event.preventDefault();
      findFirstMenuOption();
      editDialog.classList.remove(customBackgrounds.CLASSES.MOUSE_NAV);
    } else if (event.keyCode === customBackgrounds.KEYCODES.TAB) {
      // If keyboard navigation is attempted, remove mouse-only mode.
      editDialog.classList.remove(customBackgrounds.CLASSES.MOUSE_NAV);
    } else if (
        event.keyCode === customBackgrounds.KEYCODES.LEFT ||
        event.keyCode === customBackgrounds.KEYCODES.UP ||
        event.keyCode === customBackgrounds.KEYCODES.RIGHT ||
        event.keyCode === customBackgrounds.KEYCODES.DOWN) {
      event.preventDefault();
      editDialog.classList.remove(customBackgrounds.CLASSES.MOUSE_NAV);
    }
  };

  customBackgrounds.initCustomLinksItems(hideCustomLinkNotification);
  customBackgrounds.initCustomBackgrounds(showErrorNotification);
};

/**
 * Initialize custom link items in the settings menu dialog. Set the text
 * and event handlers for the various elements.
 * @param {!Function} hideCustomLinkNotification Called when the custom link
 *                    notification should be hidden.
 */
customBackgrounds.initCustomLinksItems = function(hideCustomLinkNotification) {
  customBackgrounds.hideCustomLinkNotification = hideCustomLinkNotification;

  const editDialog = $(customBackgrounds.IDS.EDIT_BG_DIALOG);
  const menu = $(customBackgrounds.IDS.MENU);

  $(customBackgrounds.IDS.CUSTOM_LINKS_RESTORE_DEFAULT_TEXT).textContent =
      configData.translatedStrings.restoreDefaultLinks;

  // Interactions with the "Restore default shortcuts" option.
  const customLinksRestoreDefaultInteraction = function() {
    editDialog.close();
    customBackgrounds.hideCustomLinkNotification();
    window.chrome.embeddedSearch.newTabPage.resetCustomLinks();
    ntpApiHandle.logEvent(BACKGROUND_CUSTOMIZATION_LOG_TYPE
                              .NTP_CUSTOMIZE_RESTORE_SHORTCUTS_CLICKED);
  };
  $(customBackgrounds.IDS.CUSTOM_LINKS_RESTORE_DEFAULT).onclick = () => {
    if (!$(customBackgrounds.IDS.CUSTOM_LINKS_RESTORE_DEFAULT).classList.
        contains(customBackgrounds.CLASSES.OPTION_DISABLED)) {
      customLinksRestoreDefaultInteraction();
    }
  };
  $(customBackgrounds.IDS.CUSTOM_LINKS_RESTORE_DEFAULT).onkeydown = function(
      event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER) {
      customLinksRestoreDefaultInteraction();
    } else if (event.keyCode === customBackgrounds.KEYCODES.UP) {
      // Handle arrow key navigation.
      event.preventDefault();
      customBackgrounds
          .getNextOption(
              customBackgrounds.MENU_ENTRIES.CUSTOM_LINKS_RESTORE_DEFAULT, -1)
          .focus();
    } else if (event.keyCode === customBackgrounds.KEYCODES.DOWN) {
      event.preventDefault();
      customBackgrounds
          .getNextOption(
              customBackgrounds.MENU_ENTRIES.CUSTOM_LINKS_RESTORE_DEFAULT, 1)
          .focus();
    }
  };
};

/**
 * Initialize the settings menu and custom backgrounds dialogs. Set the
 * text and event handlers for the various elements.
 * @param {!Function} showErrorNotification Called when the error notification
 *                    should be displayed.
 */
customBackgrounds.initCustomBackgrounds = function(showErrorNotification) {
  customBackgrounds.showErrorNotification = showErrorNotification;

  const editDialog = $(customBackgrounds.IDS.EDIT_BG_DIALOG);
  const menu = $(customBackgrounds.IDS.MENU);

  $(customBackgrounds.IDS.DEFAULT_WALLPAPERS_TEXT).textContent =
      configData.translatedStrings.defaultWallpapers;
  $(customBackgrounds.IDS.UPLOAD_IMAGE_TEXT).textContent =
      configData.translatedStrings.uploadImage;
  $(customBackgrounds.IDS.RESTORE_DEFAULT_TEXT).textContent =
      configData.translatedStrings.restoreDefaultBackground;
  $(customBackgrounds.IDS.DONE).textContent =
      configData.translatedStrings.selectionDone;
  $(customBackgrounds.IDS.CANCEL).textContent =
      configData.translatedStrings.selectionCancel;

  window.addEventListener('online', function(event) {
    customBackgrounds.networkStateChanged(true);
  });

  window.addEventListener('offline', function(event) {
    customBackgrounds.networkStateChanged(false);
  });

  if (!window.navigator.onLine) {
    customBackgrounds.networkStateChanged(false);
  }

  $(customBackgrounds.IDS.BACK_CIRCLE)
      .setAttribute('aria-label', configData.translatedStrings.backLabel);
  $(customBackgrounds.IDS.CANCEL)
      .setAttribute('aria-label', configData.translatedStrings.selectionCancel);
  $(customBackgrounds.IDS.DONE)
      .setAttribute('aria-label', configData.translatedStrings.selectionDone);

  $(customBackgrounds.IDS.DONE).disabled = true;

  // Interactions with the "Upload an image" option.
  const uploadImageInteraction = function() {
    window.chrome.embeddedSearch.newTabPage.selectLocalBackgroundImage();
    ntpApiHandle.logEvent(
        BACKGROUND_CUSTOMIZATION_LOG_TYPE.NTP_CUSTOMIZE_LOCAL_IMAGE_CLICKED);
  };

  $(customBackgrounds.IDS.UPLOAD_IMAGE).onclick = (event) => {
    if (!$(customBackgrounds.IDS.UPLOAD_IMAGE).classList.contains(
        customBackgrounds.CLASSES.OPTION_DISABLED)) {
      uploadImageInteraction();
    }
  };
  $(customBackgrounds.IDS.UPLOAD_IMAGE).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER) {
      uploadImageInteraction();
    }

    // Handle arrow key navigation.
    if (event.keyCode === customBackgrounds.KEYCODES.UP) {
      event.preventDefault();
      customBackgrounds
          .getNextOption(customBackgrounds.MENU_ENTRIES.UPLOAD_IMAGE, -1)
          .focus();
    }
    if (event.keyCode === customBackgrounds.KEYCODES.DOWN) {
      event.preventDefault();
      customBackgrounds
          .getNextOption(customBackgrounds.MENU_ENTRIES.UPLOAD_IMAGE, 1)
          .focus();
    }
  };

  // Interactions with the "Restore default background" option.
  const restoreDefaultInteraction = function() {
    editDialog.close();
    customBackgrounds.clearAttribution();
    window.chrome.embeddedSearch.newTabPage.setBackgroundURL('');
    ntpApiHandle.logEvent(BACKGROUND_CUSTOMIZATION_LOG_TYPE
                              .NTP_CUSTOMIZE_RESTORE_BACKGROUND_CLICKED);
  };
  $(customBackgrounds.IDS.RESTORE_DEFAULT).onclick = (event) => {
    if (!$(customBackgrounds.IDS.RESTORE_DEFAULT).classList.contains(
        customBackgrounds.CLASSES.OPTION_DISABLED)) {
      restoreDefaultInteraction();
    }
  };
  $(customBackgrounds.IDS.RESTORE_DEFAULT).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER) {
      restoreDefaultInteraction();
    }

    // Handle arrow key navigation.
    if (event.keyCode === customBackgrounds.KEYCODES.UP) {
      event.preventDefault();
      customBackgrounds
          .getNextOption(customBackgrounds.MENU_ENTRIES.RESTORE_DEFAULT, -1)
          .focus();
    }
    if (event.keyCode === customBackgrounds.KEYCODES.DOWN) {
      event.preventDefault();
      customBackgrounds
          .getNextOption(customBackgrounds.MENU_ENTRIES.RESTORE_DEFAULT, 1)
          .focus();
    }
  };

  // Interactions with the "Chrome backgrounds" option.
  const defaultWallpapersInteraction = function(event) {
    customBackgrounds.loadChromeBackgrounds();
    $('ntp-collection-loader').onload = function() {
      editDialog.close();
      if (typeof coll != 'undefined' && coll.length > 0) {
        customBackgrounds.showCollectionSelectionDialog(
            customBackgrounds.SOURCES.CHROME_BACKGROUNDS);
      } else {
        customBackgrounds.handleError(collErrors);
      }
    };
    ntpApiHandle.logEvent(BACKGROUND_CUSTOMIZATION_LOG_TYPE
                              .NTP_CUSTOMIZE_CHROME_BACKGROUNDS_CLICKED);
  };
  $(customBackgrounds.IDS.DEFAULT_WALLPAPERS).onclick = function(event) {
    $(customBackgrounds.IDS.MENU)
        .classList.add(customBackgrounds.CLASSES.MOUSE_NAV);
    defaultWallpapersInteraction(event);
  };
  $(customBackgrounds.IDS.DEFAULT_WALLPAPERS).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER) {
      $(customBackgrounds.IDS.MENU)
          .classList.remove(customBackgrounds.CLASSES.MOUSE_NAV);
      defaultWallpapersInteraction(event);
    }

    // Handle arrow key navigation.
    if (event.keyCode === customBackgrounds.KEYCODES.UP) {
      event.preventDefault();
      customBackgrounds
          .getNextOption(customBackgrounds.MENU_ENTRIES.CHROME_BACKGROUNDS, -1)
          .focus();
    }
    if (event.keyCode === customBackgrounds.KEYCODES.DOWN) {
      event.preventDefault();
      customBackgrounds
          .getNextOption(customBackgrounds.MENU_ENTRIES.CHROME_BACKGROUNDS, 1)
          .focus();
    }
  };

  // Escape and Backspace handling for the background picker dialog.
  menu.onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      $(customBackgrounds.IDS.TILES).scrollTop +=
          $(customBackgrounds.IDS.TILES).offsetHeight;
      event.stopPropagation();
      event.preventDefault();
    }
    if (event.keyCode === customBackgrounds.KEYCODES.ESC ||
        event.keyCode === customBackgrounds.KEYCODES.BACKSPACE) {
      event.preventDefault();
      event.stopPropagation();
      if (menu.classList.contains(
              customBackgrounds.CLASSES.COLLECTION_DIALOG)) {
        menu.close();
        customBackgrounds.resetSelectionDialog();
      } else {
        customBackgrounds.resetSelectionDialog();
        customBackgrounds.showCollectionSelectionDialog(
            customBackgrounds.dialogCollectionsSource);
      }
    }

    // If keyboard navigation is attempted, remove mouse-only mode.
    if (event.keyCode === customBackgrounds.KEYCODES.TAB ||
        event.keyCode === customBackgrounds.KEYCODES.LEFT ||
        event.keyCode === customBackgrounds.KEYCODES.UP ||
        event.keyCode === customBackgrounds.KEYCODES.RIGHT ||
        event.keyCode === customBackgrounds.KEYCODES.DOWN) {
      menu.classList.remove(customBackgrounds.CLASSES.MOUSE_NAV);
    }
  };

  // Interactions with the back arrow on the image selection dialog.
  const backInteraction = function(event) {
    if (configData.richerPicker) {
      customBackgrounds.richerPicker_resetImageMenu(true);
    }
    customBackgrounds.resetSelectionDialog();
    customBackgrounds.showCollectionSelectionDialog(
        customBackgrounds.dialogCollectionsSource);
  };
  $(customBackgrounds.IDS.BACK_CIRCLE).onclick = backInteraction;
  $(customBackgrounds.IDS.MENU_BACK_CIRCLE).onclick = backInteraction;
  $(customBackgrounds.IDS.BACK_CIRCLE).onkeyup = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER ||
        event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      backInteraction(event);
    }
  };
  // Pressing Spacebar on the back arrow shouldn't scroll the dialog.
  $(customBackgrounds.IDS.BACK_CIRCLE).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      event.stopPropagation();
    }
  };

  // Interactions with the cancel button on the background picker dialog.
  $(customBackgrounds.IDS.CANCEL).onclick = function(event) {
    customBackgrounds.closeCollectionDialog(menu);
    ntpApiHandle.logEvent(BACKGROUND_CUSTOMIZATION_LOG_TYPE
                              .NTP_CUSTOMIZE_CHROME_BACKGROUND_CANCEL);
  };
  $(customBackgrounds.IDS.CANCEL).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER ||
        event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      customBackgrounds.closeCollectionDialog(menu);
      ntpApiHandle.logEvent(BACKGROUND_CUSTOMIZATION_LOG_TYPE
                                .NTP_CUSTOMIZE_CHROME_BACKGROUND_CANCEL);
    }
  };

  // Interactions with the done button on the background picker dialog.
  const doneInteraction = function(event) {
    const done = configData.richerPicker ? $(customBackgrounds.IDS.MENU_DONE) :
                                           $(customBackgrounds.IDS.DONE);
    if (done.disabled) {
      return;
    }
    customBackgrounds.setBackground(
        customBackgrounds.selectedTile.dataset.url,
        customBackgrounds.selectedTile.dataset.attributionLine1,
        customBackgrounds.selectedTile.dataset.attributionLine2,
        customBackgrounds.selectedTile.dataset.attributionActionUrl);
  };
  $(customBackgrounds.IDS.DONE).onclick = doneInteraction;
  $(customBackgrounds.IDS.MENU_DONE).onclick = doneInteraction;
  $(customBackgrounds.IDS.DONE).onkeyup = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER) {
      doneInteraction(event);
    }
  };

  // On any arrow key event in the tiles area, focus the first tile.
  $(customBackgrounds.IDS.TILES).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.LEFT ||
        event.keyCode === customBackgrounds.KEYCODES.UP ||
        event.keyCode === customBackgrounds.KEYCODES.RIGHT ||
        event.keyCode === customBackgrounds.KEYCODES.DOWN) {
      event.preventDefault();
      if ($(customBackgrounds.IDS.MENU)
              .classList.contains(
                  customBackgrounds.CLASSES.COLLECTION_DIALOG)) {
        $('coll_tile_0').focus();
      } else {
        $('img_tile_0').focus();
      }
    }
  };

  $(customBackgrounds.IDS.BACKGROUNDS_UPLOAD).onclick = uploadImageInteraction;
  $(customBackgrounds.IDS.BACKGROUNDS_UPLOAD).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER ||
        event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      uploadImageInteraction();
    }
  };

  $(customBackgrounds.IDS.BACKGROUNDS_DEFAULT).onclick = function() {
    const tile = $(customBackgrounds.IDS.BACKGROUNDS_DEFAULT_ICON);
    tile.dataset.url = '';
    tile.dataset.attributionLine1 = '';
    tile.dataset.attributionLine2 = '';
    tile.dataset.attributionActionUrl = '';
    customBackgrounds.richerPicker_selectTile(tile);
  };

  const richerPickerOpenBackgrounds = function() {
    customBackgrounds.richerPicker_resetCustomizationMenu();
    customBackgrounds.richerPicker_selectMenuOption(
        $(customBackgrounds.IDS.BACKGROUNDS_BUTTON),
        $(customBackgrounds.IDS.BACKGROUNDS_MENU));
  };

  $(customBackgrounds.IDS.BACKGROUNDS_BUTTON).onclick =
      richerPickerOpenBackgrounds;
  $(customBackgrounds.IDS.BACKGROUNDS_BUTTON).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER ||
        event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      richerPickerOpenBackgrounds();
    }
  };

  const richerPickerOpenShortcuts = function() {
    customBackgrounds.richerPicker_resetCustomizationMenu();
    customBackgrounds.richerPicker_selectMenuOption(
        $(customBackgrounds.IDS.SHORTCUTS_BUTTON),
        $(customBackgrounds.IDS.SHORTCUTS_MENU));
  };

  $(customBackgrounds.IDS.SHORTCUTS_BUTTON).onclick = richerPickerOpenShortcuts;
  $(customBackgrounds.IDS.SHORTCUTS_BUTTON).onkeydown = function(event) {
    if (event.keyCode === customBackgrounds.KEYCODES.ENTER ||
        event.keyCode === customBackgrounds.KEYCODES.SPACE) {
      richerPickerOpenShortcuts();
    }
  };

  $(customBackgrounds.IDS.COLORS_BUTTON).onclick = function() {
    customBackgrounds.richerPicker_resetCustomizationMenu();
    customBackgrounds.richerPicker_selectMenuOption(
        $(customBackgrounds.IDS.COLORS_BUTTON),
        $(customBackgrounds.IDS.COLORS_MENU));
  };
};

customBackgrounds.handleError = function(errors) {
  const unavailableString = configData.translatedStrings.backgroundsUnavailable;

  if (errors != 'undefined') {
    // Network errors.
    if (errors.net_error) {
      if (errors.net_error_no != 0) {
        const onClick = () => {
          window.open(
              'https://chrome://network-error/' + errors.net_error_no,
              '_blank');
        };
        customBackgrounds.showErrorNotification(
            configData.translatedStrings.connectionError,
            configData.translatedStrings.moreInfo, onClick);
      } else {
        customBackgrounds.showErrorNotification(
            configData.translatedStrings.connectionErrorNoPeriod);
      }
    } else if (errors.service_error) {  // Service errors.
      customBackgrounds.showErrorNotification(unavailableString);
    }
    return;
  }

  // Generic error when we can't tell what went wrong.
  customBackgrounds.showErrorNotification(unavailableString);
};
