// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/manifest_web_app_browser_controller.h"

#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/ssl/security_state_tab_helper.h"
#include "chrome/browser/ui/browser.h"
#include "content/public/browser/navigation_entry.h"
#include "content/public/common/origin_util.h"
#include "content/public/common/url_constants.h"
#include "extensions/common/constants.h"
#include "ui/gfx/favicon_size.h"
#include "ui/gfx/image/image_skia.h"
#include "url/gurl.h"

ManifestWebAppBrowserController::ManifestWebAppBrowserController(
    Browser* browser)
    : AppBrowserController(browser), app_launch_url_(GURL()) {}

ManifestWebAppBrowserController::~ManifestWebAppBrowserController() = default;

base::Optional<std::string> ManifestWebAppBrowserController::GetAppId() const {
  return base::nullopt;
}

bool ManifestWebAppBrowserController::ShouldShowToolbar() const {
  content::WebContents* web_contents =
      browser()->tab_strip_model()->GetActiveWebContents();

  // Don't show a toolbar until a navigation has occurred.
  if (!web_contents || web_contents->GetLastCommittedURL().is_empty())
    return false;

  // Show toolbar if the web_contents is not on a secure origin.
  if (!content::IsOriginSecure(app_launch_url_))
    return true;

  // Show toolbar if web_contents is not on the same origin as it was originally
  // launched on.
  if (!url::IsSameOriginWith(app_launch_url_,
                             web_contents->GetLastCommittedURL()) ||
      !url::IsSameOriginWith(app_launch_url_, web_contents->GetVisibleURL())) {
    return true;
  }

  // Show toolbar if on a insecure external website. This checks the security
  // level, different from IsOriginSecure which just checks the origin itself.
  if (!IsSiteSecure(web_contents))
    return true;

  return false;
}

bool ManifestWebAppBrowserController::ShouldShowHostedAppButtonContainer()
    const {
  return true;
}

gfx::ImageSkia ManifestWebAppBrowserController::GetWindowAppIcon() const {
  gfx::ImageSkia page_icon = browser()->GetCurrentPageIcon().AsImageSkia();
  if (!page_icon.isNull())
    return page_icon;

  // The extension icon may be loading still. Return a transparent icon rather
  // than using a placeholder to avoid flickering.
  SkBitmap bitmap;
  bitmap.allocN32Pixels(gfx::kFaviconSize, gfx::kFaviconSize);
  bitmap.eraseColor(SK_ColorTRANSPARENT);
  return gfx::ImageSkia::CreateFrom1xBitmap(bitmap);
}

gfx::ImageSkia ManifestWebAppBrowserController::GetWindowIcon() const {
  return browser()->GetCurrentPageIcon().AsImageSkia();
}

std::string ManifestWebAppBrowserController::GetAppShortName() const {
  return std::string();
}

base::string16 ManifestWebAppBrowserController::GetFormattedUrlOrigin() const {
  return FormatUrlOrigin(GetAppLaunchURL());
}

GURL ManifestWebAppBrowserController::GetAppLaunchURL() const {
  return app_launch_url_;
}

void ManifestWebAppBrowserController::OnTabInserted(
    content::WebContents* contents) {
  if (app_launch_url_.is_empty())
    app_launch_url_ = contents->GetURL();
  AppBrowserController::OnTabInserted(contents);
  UpdateToolbarVisibility(false);
}
