/*
 * tcmsg_qdisc.c - traffic control qdisc message parser
 * Copyright (C) 2014 Tetsumune KISO <t2mune@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "nield.h"
#include "rtnetlink.h"

#define TIME_UNITS_PER_SEC 1000000

static double us2tick = 1;
static double clock_factor = 1;

/*
 * parse traffic control qdisc messages
 */
int parse_tcmsg_qdisc(struct nlmsghdr *nlh)
{
    struct tcmsg *tcm;
    int tcm_len;
    struct rtattr *tca[__TCA_MAX];
    char msg[MAX_MSG_SIZE] = "";
    char *mp = msg;
    char ifname[IFNAMSIZ];
    char parent[MAX_STR_SIZE] = "";
    char handle[MAX_STR_SIZE] = "";
    char kind[IFNAMSIZ] = "(unknown)";
    int log_opts = get_log_opts();

    /* debug nlmsghdr */
    if(log_opts & L_DEBUG)
        debug_nlmsg(0, nlh);

    /* get tcmsg */
    tcm_len = NLMSG_PAYLOAD(nlh, 0);
    if(tcm_len < sizeof(*tcm)) {
        rec_log("error: %s: tcmsg: length too short", __func__);
        return(1);
    }
    tcm = (struct tcmsg *)NLMSG_DATA(nlh);

    /* parse traffic control message attributes */
    parse_tc(tca, nlh);

    /* debug tcmsg */
    if(log_opts & L_DEBUG)
        debug_tcmsg(0, nlh, tcm, tca, tcm_len);

    /* kind of message */
    switch(nlh->nlmsg_type) {
        case RTM_NEWQDISC:
            mp = add_log(msg, mp, "tc qdisc added: ");
            break;
        case RTM_DELQDISC:
            mp = add_log(msg, mp, "tc qdisc deleted: ");
            break;
        case RTM_NEWTCLASS:
            mp = add_log(msg, mp, "tc class added: ");
            break;
        case RTM_DELTCLASS:
            mp = add_log(msg, mp, "tc class deleted: ");
            break;
        default:
            rec_log("error: %s: nlmsg_type: unknown message", __func__);
            return(1);
    }

    /* get interface name */
    if_indextoname_from_lists(tcm->tcm_ifindex, ifname);

    mp = add_log(msg, mp, "interface=%s ", ifname); 

    /* get parent qdisc handle */
    parse_tc_classid(parent, sizeof(parent), tcm->tcm_parent);
    mp = add_log(msg, mp, "parent=%s ", parent);

    /* get qdisc handle */
    parse_tc_classid(handle, sizeof(handle), tcm->tcm_handle);
    mp = add_log(msg, mp, "classid=%s ", handle);

    /* get qdisc kind */
    if(tca[TCA_KIND])
        strncpy(kind, (char *)RTA_DATA(tca[TCA_KIND]), sizeof(kind));

    mp = add_log(msg, mp, "qdisc=%s ", kind);

    /* get qdisc options */
    if(tca[TCA_OPTIONS]) {
        if(!strncmp(kind, "pfifo_fast", sizeof(kind))) {
            if(parse_tca_options_prio(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "pfifo", sizeof(kind))) {
            if(parse_tca_options_pfifo(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "bfifo", sizeof(kind))) {
            if(parse_tca_options_bfifo(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "prio", sizeof(kind))) {
            if(parse_tca_options_prio(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#ifdef HAVE_STRUCT_TC_MULTIQ_QOPT_BANDS
        } else if(!strncmp(kind, "multiq", sizeof(kind))) {
            if(parse_tca_options_multiq(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
        } else if(!strncmp(kind, "plug", sizeof(kind))) {
            if(parse_tca_options_plug(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
        } else if(!strncmp(kind, "sfq", sizeof(kind))) {
            if(parse_tca_options_sfq(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "tbf", sizeof(kind))) {
            if(parse_tca_options_tbf(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "red", sizeof(kind))) {
            if(parse_tca_options_red(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "gred", sizeof(kind))) {
            if(parse_tca_options_gred(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
#if HAVE_DECL_TCA_CHOKE_UNSPEC
        } else if(!strncmp(kind, "choke", sizeof(kind))) {
            if(parse_tca_options_choke(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
        } else if(!strncmp(kind, "htb", sizeof(kind))) {
            if(parse_tca_options_htb(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "hfsc", sizeof(kind))) {
            if(parse_tca_options_hfsc(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "cbq", sizeof(kind))) {
            if(parse_tca_options_cbq(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "dsmark", sizeof(kind))) {
            if(parse_tca_options_dsmark(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
        } else if(!strncmp(kind, "netem", sizeof(kind))) {
            if(parse_tca_options_netem(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#if HAVE_DECL_TCA_DRR_UNSPEC
        } else if(!strncmp(kind, "drr", sizeof(kind))) {
            if(parse_tca_options_drr(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
#if HAVE_DECL_TCA_SFB_UNSPEC
        } else if(!strncmp(kind, "sfb", sizeof(kind))) {
            if(parse_tca_options_sfb(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
#if HAVE_DECL_TCA_QFQ_UNSPEC
        } else if(!strncmp(kind, "qfq", sizeof(kind))) {
            if(parse_tca_options_qfq(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
#if HAVE_DECL_TCA_CODEL_UNSPEC
        } else if(!strncmp(kind, "codel", sizeof(kind))) {
            if(parse_tca_options_codel(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
        } else if(!strncmp(kind, "fq_codel", sizeof(kind))) {
            if(parse_tca_options_fq_codel(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
#endif
        }
    }

    /* write log */
    rec_log("%s", msg);

    return(0);
}

/*
 * parse qdisc handle
 */
void parse_tc_classid(char *p, int len, unsigned id)
{
    if (id == TC_H_ROOT)
        snprintf(p, len, "root");
    else if(id == TC_H_INGRESS)
        snprintf(p, len, "ingress");
    else if(id == TC_H_UNSPEC)
        snprintf(p, len, "none");
    else if(!TC_H_MAJ(id))
        snprintf(p, len, ":%x", TC_H_MIN(id));
    else if(!TC_H_MIN(id))
        snprintf(p, len, "%x:", TC_H_MAJ(id)>>16);
    else if(id)
        snprintf(p, len, "%x:%x", TC_H_MAJ(id)>>16, TC_H_MIN(id));
}

/*
 * parse prio options
 */
int parse_tca_options_prio(char *msg, char **mp, struct rtattr *tca)
{
    struct tc_prio_qopt *qopt;
    int i;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_log("error: %s: TCA_OPTIONS: payload too short", __func__);
        return(1);
    }
    qopt = (struct tc_prio_qopt *)RTA_DATA(tca);
    *mp = add_log(msg, *mp, "bands=%d priomap=", qopt->bands);

    for(i = 0; i < TC_PRIO_MAX + 1; i++)
        if(i == TC_PRIO_MAX)
            *mp = add_log(msg, *mp, "%d ", qopt->priomap[i]);
        else
            *mp = add_log(msg, *mp, "%d-", qopt->priomap[i]);

    return(0);
}

/*
 * parse pfifo options
 */
int parse_tca_options_pfifo(char *msg, char **mp, struct rtattr *tca)
{
    struct tc_fifo_qopt *qopt;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_log("error: %s: TCA_OPTIONS: payload too short", __func__);
        return(1);
    }
    qopt = (struct tc_fifo_qopt *)RTA_DATA(tca);
    *mp = add_log(msg, *mp, "limit=%d(packet) ", qopt->limit);

    return(0);
}

/*
 * parse bfifo options
 */
int parse_tca_options_bfifo(char *msg, char **mp, struct rtattr *tca)
{
    struct tc_fifo_qopt *qopt;
    char limit[MAX_STR_SIZE];

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_log("error: %s: TCA_OPTIONS: payload too short", __func__);
        return(1);
    }
    qopt = (struct tc_fifo_qopt *)RTA_DATA(tca);
    convert_unit_size(limit, sizeof(limit), qopt->limit);
    *mp = add_log(msg, *mp, "limit=%s", limit);

    return(0);
}

#ifdef HAVE_STRUCT_TC_MULTIQ_QOPT_BANDS
/*
 * parse multiq options
 */
int parse_tca_options_multiq(char *msg, char **mp, struct rtattr *tca)
{
    struct tc_multiq_qopt *qopt;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_log("error: %s: TCA_OPTIONS: payload too short", __func__);
        return(1);
    }
    qopt = (struct tc_multiq_qopt *)RTA_DATA(tca);

    *mp = add_log(msg, *mp, "bands=%d max=%d ", qopt->bands, qopt->max_bands);

    return(0);
};
#endif

#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
/*
 * parse plug options
 */
int parse_tca_options_plug(char *msg, char **mp, struct rtattr *tca)
{
    struct tc_plug_qopt *qopt;
    char action[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_log("error: %s: TCA_OPTIONS: payload too short", __func__);
        return(1);
    }
    qopt = (struct tc_plug_qopt *)RTA_DATA(tca);

    if(qopt->action == TCQ_PLUG_BUFFER)
        strncpy(action, "buffer", sizeof(action));
    else if(qopt->action == TCQ_PLUG_RELEASE_ONE)
        strncpy(action, "release_one", sizeof(action));
    else if(qopt->action == TCQ_PLUG_RELEASE_INDEFINITE)
        strncpy(action, "release_indefinite", sizeof(action));

    *mp = add_log(msg, *mp, "action=%s limit=%u ", action, qopt->limit);

    return(0);
}
#endif

/*
 * parse tbf options
 */
int parse_tca_options_tbf(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *tbf[__TCA_TBF_MAX];

    parse_tbf(tbf, tca);

    if(tbf[TCA_TBF_PARMS]) {
        struct tc_tbf_qopt *qopt;
        char rate[MAX_STR_SIZE];
        char burst[MAX_STR_SIZE];
        char peakrate[MAX_STR_SIZE];
        char mtu[MAX_STR_SIZE];
        double rate_latency = 0;
        double peakrate_latency = 0;
        char latency[MAX_STR_SIZE];

        if(RTA_PAYLOAD(tbf[TCA_TBF_PARMS]) < sizeof(*qopt)) {
            rec_log("error: %s: TCA_TBF_PARMS: payload too short", __func__);
            return(1);
        }
        qopt = (struct tc_tbf_qopt *)RTA_DATA(tbf[TCA_TBF_PARMS]);

        get_us2tick();
        convert_unit_rate(rate, sizeof(rate), qopt->rate.rate);
        convert_unit_size(burst, sizeof(burst),
            get_burst_size(qopt->rate.rate, qopt->buffer));
        rate_latency = get_latency(qopt->rate.rate, qopt->buffer, qopt->limit);

        *mp = add_log(msg, *mp, "rate=%s burst=%s ", rate, burst);

        if(qopt->peakrate.rate) {
            convert_unit_rate(peakrate, sizeof(peakrate), qopt->peakrate.rate);
            convert_unit_size(mtu, sizeof(mtu),
                get_burst_size(qopt->peakrate.rate, qopt->mtu));
            peakrate_latency = get_latency(qopt->peakrate.rate, qopt->mtu, qopt->limit);

            *mp = add_log(msg, *mp, "peakrate=%s minburst=%s ", peakrate, mtu);
        }

        if(rate_latency < peakrate_latency)
            convert_unit_usec(latency, sizeof(latency), peakrate_latency);
        else
            convert_unit_usec(latency, sizeof(latency), rate_latency);

        *mp = add_log(msg, *mp, "latency=%s ", latency);
    }

    return(0);
}

/*
 * parse sfq options
 */
int parse_tca_options_sfq(char *msg, char **mp, struct rtattr *tca)
{
    struct tc_sfq_qopt *qopt;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_log("error: %s: TCA_OPTIONS: payaload too short", __func__);
        return(1);
    }
    qopt = (struct tc_sfq_qopt *)RTA_DATA(tca);

    *mp = add_log(msg, *mp,
        "quantum=%u(byte) perturb=%d(second) limit=%u(packet) divisor=%u flows=%u ",
        qopt->quantum, qopt->perturb_period, qopt->limit, qopt->divisor, qopt->flows);

#ifdef HAVE_STRUCT_TC_SFQ_QOPT_V1_V0
    struct tc_sfq_qopt_v1 *qopt_v1 = NULL;

    if(RTA_PAYLOAD(tca) >= sizeof(*qopt_v1))
        qopt_v1 = (struct tc_sfq_qopt_v1 *)RTA_DATA(tca);

    if(qopt_v1) {
        char list[MAX_STR_SIZE] = "";
        char min[MAX_STR_SIZE] = "";
        char max[MAX_STR_SIZE] = "";

        convert_unit_size(min, sizeof(min), qopt_v1->qth_min);
        convert_unit_size(max, sizeof(max), qopt_v1->qth_max);

        *mp = add_log(msg, *mp, "depth=%u(packet) headdrop=%s min=%s max=%s ",
            qopt_v1->depth, qopt_v1->headdrop ? "on" : "off", min, max);

        if(qopt_v1->flags) {
            convert_tc_red_flags(qopt_v1->flags, list, sizeof(list), 0);
            *mp = add_log(msg, *mp, "flag=%s ", list);
        }
        *mp = add_log(msg, *mp, "probability=%g(%%) ", qopt_v1->max_P / pow(2, 32) * 100);
    }
#endif

    return(0);
}

/*
 * parse red options
 */
int parse_tca_options_red(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *red[__TCA_RED_MAX];

    parse_red(red, tca);

    if(red[TCA_RED_PARMS]) {
        struct tc_red_qopt *qopt;
        char limit[MAX_STR_SIZE] = "";
        char min[MAX_STR_SIZE] = "";
        char max[MAX_STR_SIZE] = "";
        char list[MAX_STR_SIZE] = "";

        if(RTA_PAYLOAD(red[TCA_RED_PARMS]) < sizeof(*qopt)) {
            rec_log("error: %s: TCA_RED_PARMS: payload too short", __func__);
            return(1);
        }
        qopt = (struct tc_red_qopt *)RTA_DATA(red[TCA_RED_PARMS]);

        convert_unit_size(limit, sizeof(limit), qopt->limit);
        convert_unit_size(min, sizeof(min), qopt->qth_min);
        convert_unit_size(max, sizeof(max), qopt->qth_max);

        *mp = add_log(msg, *mp, "limit=%s min=%s max=%s ", limit, min, max);
        if(qopt->flags) {
            convert_tc_red_flags(qopt->flags, list, sizeof(list), 0);
            *mp = add_log(msg, *mp, "flag=%s ", list);
        }
    }

#if HAVE_DECL_TCA_RED_MAX_P
    if(red[TCA_RED_MAX_P]) {
        if(RTA_PAYLOAD(red[TCA_RED_MAX_P]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_RED_MAX_P: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "probability=%g(%%) ",
            *(unsigned *)RTA_DATA(red[TCA_RED_MAX_P]) / pow(2, 32) * 100);
    }
#endif

    return(0);
}

/*
 * parse gred options
 */
int parse_tca_options_gred(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *gred[__TCA_GRED_MAX];
    struct tc_gred_sopt *sopt = NULL;
    struct tc_gred_qopt *qopt = NULL;
    unsigned *max_p = NULL;
    int i, flag = 0;
    char *mp_tmp = *mp;
    char limit[MAX_STR_SIZE] = "";
    char min[MAX_STR_SIZE] = "";
    char max[MAX_STR_SIZE] = "";

    parse_gred(gred, tca);

    if(gred[TCA_GRED_PARMS]) {
        if(RTA_PAYLOAD(gred[TCA_GRED_PARMS]) < sizeof(*qopt)) {
            rec_log("error: %s: TCA_GRED_PARMS: payload too short", __func__);
            return(1);
        }
        qopt = (struct tc_gred_qopt *)RTA_DATA(gred[TCA_GRED_PARMS]);
    }

    if(gred[TCA_GRED_DPS]) {
        if(RTA_PAYLOAD(gred[TCA_GRED_DPS]) < sizeof(*sopt)) {
            rec_log("error: %s: TCA_GRED_DPS: payload too short", __func__);
            return(1);
        }
        sopt = (struct tc_gred_sopt *)RTA_DATA(gred[TCA_GRED_DPS]);
    }

    if(!sopt || !qopt) {
        rec_log("%s", msg);
        return(0);
    }

#if HAVE_DECL_TCA_GRED_MAX_P
    if(gred[TCA_GRED_MAX_P]) {
        if(RTA_PAYLOAD(gred[TCA_GRED_MAX_P]) < sizeof(*max_p)) {
            rec_log("error: %s: TCA_GRED_MAX_P: payload too short", __func__);
            return(1);
        }
        max_p = (unsigned *)RTA_DATA(gred[TCA_GRED_MAX_P]);
    }
#endif

    for(i = 0; i < sopt->DPs; i++, qopt++) {
        if(qopt->DP >= sopt->DPs)
            continue;

        convert_unit_size(limit, sizeof(limit), qopt->limit);
        convert_unit_size(min, sizeof(min), qopt->qth_min);
        convert_unit_size(max, sizeof(max), qopt->qth_max);

        *mp = add_log(msg, *mp, "DP=%u limit=%s min=%s max=%s prio=%d ",
            qopt->DP, limit, min, max, qopt->prio);
        if(max_p)
            *mp = add_log(msg, *mp, "probability=%g(%%) ", max_p[i] / pow(2, 32) * 100);
        rec_log("%s", msg);
        *mp = mp_tmp;
        flag = 1;
    }

    if(!flag || strstr(msg, "qdisc deleted")) {
        *mp = add_log(msg, *mp, "DPs=%u default-DP=%u ", sopt->DPs, sopt->def_DP);
        rec_log("%s", msg);
    }

    return(0);
}

#if HAVE_DECL_TCA_CHOKE_UNSPEC
/*
 * parse choke options
 */
int parse_tca_options_choke(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *choke[__TCA_CHOKE_MAX];

    parse_choke(choke, tca);

    if(choke[TCA_CHOKE_PARMS]) {
        struct tc_choke_qopt *qopt;
        char list[MAX_STR_SIZE] = "";

        if(RTA_PAYLOAD(choke[TCA_CHOKE_PARMS]) < sizeof(*qopt)) {
            rec_log("error: %s: TCA_CHOKE_PARMS: payload too short", __func__);
            return(1);
        }
        qopt = (struct tc_choke_qopt *)RTA_DATA(choke[TCA_CHOKE_PARMS]);

        *mp = add_log(msg, *mp, "limit=%u(packet) min=%u(packet) max=%u(packet) ",
            qopt->limit, qopt->qth_min, qopt->qth_max);
        if(qopt->flags) {
            convert_tc_red_flags(qopt->flags, list, sizeof(list), 0);
            *mp = add_log(msg, *mp, "flag=%s ", list);
        }
    }

    if(choke[TCA_CHOKE_MAX_P]) {
        if(RTA_PAYLOAD(choke[TCA_CHOKE_MAX_P]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_CHOKE_MAX_P: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "probability=%g(%%) ",
            *(unsigned *)RTA_DATA(choke[TCA_CHOKE_MAX_P]) / pow(2, 32) * 100);
    }

    return(0);
}
#endif

/*
 * parse htb options
 */
int parse_tca_options_htb(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *htb[__TCA_HTB_MAX];

    parse_htb(htb, tca);

    if(htb[TCA_HTB_INIT]) {
        struct tc_htb_glob *glob;

        if(RTA_PAYLOAD(htb[TCA_HTB_INIT]) < sizeof(*glob)) {
            rec_log("error: %s: TCA_HTB_INIT: payload too short", __func__);
            return(1);
        }
        glob = (struct tc_htb_glob *)RTA_DATA(htb[TCA_HTB_INIT]);

        *mp = add_log(msg, *mp, "rate2quantum=%u ", glob->rate2quantum);
        *mp = add_log(msg, *mp, "default-class=0x%x ", glob->defcls);
    }

    if(htb[TCA_HTB_PARMS]) {
        struct tc_htb_opt *opt;
        char rate[MAX_STR_SIZE];
        char ceil[MAX_STR_SIZE];
        char burst[MAX_STR_SIZE];
        char cburst[MAX_STR_SIZE];

        if(RTA_PAYLOAD(htb[TCA_HTB_PARMS]) < sizeof(*opt)) {
            rec_log("error: %s: TCA_HTB_PARMS: payload too short", __func__);
            return(1);
        }
        opt = (struct tc_htb_opt *)RTA_DATA(htb[TCA_HTB_PARMS]);

        get_us2tick();
        convert_unit_rate(rate, sizeof(rate), opt->rate.rate);
        convert_unit_size(burst, sizeof(burst),
            get_burst_size(opt->rate.rate, opt->buffer));
        convert_unit_rate(ceil, sizeof(ceil), opt->ceil.rate);
        convert_unit_size(cburst, sizeof(cburst),
            get_burst_size(opt->ceil.rate, opt->cbuffer));

        *mp = add_log(msg, *mp, "rate=%s burst=%s ceil=%s cburst=%s level=%u prio=%u ",
            rate, burst, ceil, cburst, opt->level, opt->prio);
    }

    return(0);
}

/*
 * parse hfsc options
 *
 * rt : realtime service curve
 * ls : linkshare service curve
 * sc : rt+ls service curve
 * ul : upperlimit service curve
 *
 */
int parse_tca_options_hfsc(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *hfsc[__TCA_HFSC_MAX];
    struct tc_hfsc_qopt *qopt;
    struct tc_service_curve *rsc = NULL, *fsc = NULL, *usc = NULL;

    if(RTA_PAYLOAD(tca) == sizeof(*qopt)) {
        qopt = (struct tc_hfsc_qopt *)RTA_DATA(tca);
        *mp = add_log(msg, *mp, "default-class=0x%x ", qopt->defcls);

        return(0);
    }

    parse_hfsc(hfsc, tca);

    if(hfsc[TCA_HFSC_RSC]) {
        if(RTA_PAYLOAD(hfsc[TCA_HFSC_RSC]) < sizeof(*rsc)) {
            rec_log("error: %s: TCA_HFSC_RSC: payload too short", __func__);
            return(1);
        }
        rsc = (struct tc_service_curve *)RTA_DATA(hfsc[TCA_HFSC_RSC]);
    }

    if(hfsc[TCA_HFSC_FSC]) {
        if(RTA_PAYLOAD(hfsc[TCA_HFSC_FSC]) < sizeof(*fsc)) {
            rec_log("error: %s: TCA_HFSC_FSC: payload too short", __func__);
            return(1);
        }
        fsc = (struct tc_service_curve *)RTA_DATA(hfsc[TCA_HFSC_FSC]);
    }

    if(hfsc[TCA_HFSC_USC]) {
        if(RTA_PAYLOAD(hfsc[TCA_HFSC_USC]) < sizeof(*usc)) {
            rec_log("error: %s: TCA_HFSC_USC: payload too short", __func__);
            return(1);
        }
        usc = (struct tc_service_curve *)RTA_DATA(hfsc[TCA_HFSC_USC]);
    }

    if(rsc && fsc && memcmp(rsc, fsc, sizeof(*rsc)) == 0)
        print_hfsc_sc(msg, mp, "realtime+linkshare", rsc);
    else {
        if(rsc)
            print_hfsc_sc(msg, mp, "realtime", rsc);
        if(fsc)
            print_hfsc_sc(msg, mp, "linkshare", fsc);
    }

    if(usc)
        print_hfsc_sc(msg, mp, "upperlimit", usc);

    return(0);
}

/*
 * parse hfsc survice curve
 */
int print_hfsc_sc(char *msg, char **mp, char *name, struct tc_service_curve *sc)
{
    char m1[MAX_STR_SIZE], m2[MAX_STR_SIZE], d[MAX_STR_SIZE];

    convert_unit_rate(m1, sizeof(m1), sc->m1);
    convert_unit_rate(m2, sizeof(m2), sc->m2);
    convert_unit_usec(d, sizeof(d), sc->d);

    *mp = add_log(msg, *mp, "%s(m1/d/m2)=%s/%s/%s ", name, m1, d, m2);

    return(0);
}

/*
 * parse cbq options
 */
int parse_tca_options_cbq(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *cbq[__TCA_CBQ_MAX];
    struct tc_ratespec *rspec = NULL;
    struct tc_cbq_lssopt *lss = NULL;
    struct tc_cbq_wrropt *wrr = NULL;
    struct tc_cbq_fopt *fopt = NULL;
    struct tc_cbq_ovl *ovl = NULL;

    parse_cbq(cbq, tca);

    if(cbq[TCA_CBQ_LSSOPT]) {
        if(RTA_PAYLOAD(cbq[TCA_CBQ_LSSOPT]) < sizeof(*lss)) {
            rec_log("error: %s: TCA_CBQ_LSSOPT: payload too short", __func__);
            return(1);
        }
        lss = (struct tc_cbq_lssopt *)RTA_DATA(cbq[TCA_CBQ_LSSOPT]);
    }

    if(cbq[TCA_CBQ_WRROPT]) {
        if(RTA_PAYLOAD(cbq[TCA_CBQ_WRROPT]) < sizeof(*wrr)) {
            rec_log("error: %s: TCA_CBQ_WRROPT: payload too short", __func__);
            return(1);
        }
        wrr = (struct tc_cbq_wrropt *)RTA_DATA(cbq[TCA_CBQ_WRROPT]);
    }

    if(cbq[TCA_CBQ_FOPT]) {
        if(RTA_PAYLOAD(cbq[TCA_CBQ_FOPT]) < sizeof(*fopt)) {
            rec_log("error: %s: TCA_CBQ_FOPT: payload too short", __func__);
            return(1);
        }
        fopt = (struct tc_cbq_fopt *)RTA_DATA(cbq[TCA_CBQ_FOPT]);
    }

    if(cbq[TCA_CBQ_OVL_STRATEGY]) {
        if(RTA_PAYLOAD(cbq[TCA_CBQ_OVL_STRATEGY]) < sizeof(*ovl)) {
            rec_log("error: %s: TCA_CBQ_OVL_STRATEGY: payload too short", __func__);
            return(1);
        }
        ovl = (struct tc_cbq_ovl *)RTA_DATA(cbq[TCA_CBQ_OVL_STRATEGY]);
    }

    if(cbq[TCA_CBQ_RATE]) {
        if(RTA_PAYLOAD(cbq[TCA_CBQ_RATE]) < sizeof(*rspec)) {
            rec_log("error: %s: TCA_CBQ_RATE: payload too short", __func__);
            return(1);
        }
        rspec = (struct tc_ratespec *)RTA_DATA(cbq[TCA_CBQ_RATE]);
    }

    if(rspec) {
        char rate[MAX_STR_SIZE];

        convert_unit_rate(rate, sizeof(rate), rspec->rate);
        *mp = add_log(msg, *mp, "rate=%s ", rate);
    }

    if(lss) {
        char maxidle[MAX_STR_SIZE];
        char minidle[MAX_STR_SIZE];

        get_us2tick();
        convert_unit_usec(maxidle, sizeof(maxidle),
            (lss->maxidle >> lss->ewma_log) / us2tick);
        *mp = add_log(msg, *mp, "maxidle=%s ", maxidle);

        if(lss->minidle != 0x7fffffff) {
            convert_unit_usec(minidle, sizeof(minidle),
                (lss->minidle >> lss->ewma_log) / us2tick);
            *mp = add_log(msg, *mp, "minidle=%s ", minidle);
        }

        *mp = add_log(msg, *mp, "level=%u avpkt=%u(byte) ", lss->level, lss->avpkt);
    }

    if(wrr) {
        if(wrr->priority != TC_CBQ_MAXPRIO)
            *mp = add_log(msg, *mp, "prio=%u ", wrr->priority);
        else
            *mp = add_log(msg, *mp, "prio=%u(no-transmit) ", wrr->priority);
    }

    return(0);
}

/*
 * parse dsmark options
 */
int parse_tca_options_dsmark(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *dsmark[__TCA_DSMARK_MAX];

    parse_dsmark(dsmark, tca);

    if(dsmark[TCA_DSMARK_INDICES]) {
        if(RTA_PAYLOAD(dsmark[TCA_DSMARK_INDICES]) < sizeof(unsigned short)) {
            rec_log("error: %s: TCA_DSMARK_INDICES: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "indices=0x%04x ",
            *(unsigned short *)RTA_DATA(dsmark[TCA_DSMARK_INDICES]));
    }

    if(dsmark[TCA_DSMARK_DEFAULT_INDEX]) {
        if(RTA_PAYLOAD(dsmark[TCA_DSMARK_DEFAULT_INDEX]) < sizeof(unsigned short)) {
            rec_log("error: %s: TCA_DSMARK_DEFAULT_INDEX: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "default_index=0x%04x ",
            *(unsigned short *)RTA_DATA(dsmark[TCA_DSMARK_DEFAULT_INDEX]));
    }

    if(dsmark[TCA_DSMARK_SET_TC_INDEX])
        *mp = add_log(msg, *mp, "set_tc_index=on ");

    if(dsmark[TCA_DSMARK_VALUE]) {
        if(RTA_PAYLOAD(dsmark[TCA_DSMARK_VALUE]) < sizeof(unsigned char)) {
            rec_log("error: %s: TCA_DSMARK_VALUE: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "value=0x%02x ",
            *(unsigned char *)RTA_DATA(dsmark[TCA_DSMARK_VALUE]));
    }

    if(dsmark[TCA_DSMARK_MASK]) {
        if(RTA_PAYLOAD(dsmark[TCA_DSMARK_MASK]) < sizeof(unsigned char)) {
            rec_log("error: %s: TCA_DSMARK_MASK: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "mask=0x%02x ",
            *(unsigned char *)RTA_DATA(dsmark[TCA_DSMARK_MASK]));
    }

    return(0);
}

/*
 * parse netem options
 */
int parse_tca_options_netem(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *netem[__TCA_NETEM_MAX];
    struct tc_netem_qopt *qopt;
    struct tc_netem_corr *corr = NULL;
    struct tc_netem_reorder *reorder = NULL;
    struct tc_netem_corrupt *corrupt = NULL;
    const double max_percent_value = 0xffffffff;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_log("error: %s: TCA_OPTIONS: payload too short", __func__);
        return(1);
    }
    qopt = (struct tc_netem_qopt *)RTA_DATA(tca);

    parse_netem(netem, tca);

    if(netem[TCA_NETEM_CORR]) {
        if(RTA_PAYLOAD(netem[TCA_NETEM_CORR]) < sizeof(*corr)) {
            rec_log("error: %s: TCA_NETEM_CORR: payload too short", __func__);
            return(1);
        }
        corr = (struct tc_netem_corr *)RTA_DATA(netem[TCA_NETEM_CORR]);
    }

    if(netem[TCA_NETEM_REORDER]) {
        if(RTA_PAYLOAD(netem[TCA_NETEM_REORDER]) < sizeof(*reorder)) {
            rec_log("error: %s: TCA_NETEM_REORDER: payload too short", __func__);
            return(1);
        }
        reorder = (struct tc_netem_reorder *)RTA_DATA(netem[TCA_NETEM_REORDER]);
    }

    if(netem[TCA_NETEM_CORRUPT]) {
        if(RTA_PAYLOAD(netem[TCA_NETEM_REORDER]) < sizeof(*corrupt)) {
            rec_log("error: %s: TCA_NETEM_REORDER: payload too short", __func__);
            return(1);
        }
        corrupt = (struct tc_netem_corrupt *)RTA_DATA(netem[TCA_NETEM_CORRUPT]);
    }

#if HAVE_DECL_TCA_NETEM_LOSS
    struct rtattr *loss[__NETEM_LOSS_MAX];
    struct tc_netem_gimodel *gimodel = NULL;
    struct tc_netem_gemodel *gemodel = NULL;

    if(netem[TCA_NETEM_LOSS]) {

        parse_netem_loss(loss, netem[TCA_NETEM_LOSS]);

        if(loss[NETEM_LOSS_GI]) {
            if(RTA_PAYLOAD(loss[NETEM_LOSS_GI]) < sizeof(*gimodel)) {
                rec_log("error: %s: NETEM_LOSS_GI: payload too short",
                    __func__);
                return(1);
            }
            gimodel = (struct tc_netem_gimodel *)RTA_DATA(loss[NETEM_LOSS_GI]);
        }

        if(loss[NETEM_LOSS_GE]) {
            if(RTA_PAYLOAD(loss[NETEM_LOSS_GE]) < sizeof(*gemodel)) {
                rec_log("error: %s: NETEM_LOSS_GE: payload too short",
                    __func__);
                return(1);
            }
            gemodel = (struct tc_netem_gemodel *)RTA_DATA(loss[NETEM_LOSS_GE]);
        }
    }
#endif

#if HAVE_DECL_TCA_NETEM_RATE
    struct tc_netem_rate *rate = NULL;

    if(netem[TCA_NETEM_RATE]) {

        if(RTA_PAYLOAD(netem[TCA_NETEM_RATE]) < sizeof(*rate)) {
            rec_log("error: %s: TCA_NETEM_RATE: payload too short", __func__);
            return(1);
        }
        rate = (struct tc_netem_rate *)RTA_DATA(netem[TCA_NETEM_RATE]);
    }
#endif

    if(qopt->limit)
        *mp = add_log(msg, *mp, "limit=%u(packet) ", qopt->limit);

    if(qopt->latency) {
        char latency[MAX_STR_SIZE];

        get_us2tick();
        convert_unit_usec(latency, sizeof(latency), qopt->latency / us2tick);
        *mp = add_log(msg, *mp, "delay=%s ", latency);
        if(corr && corr->delay_corr)
            *mp = add_log(msg, *mp, "delay-correlation=%g(%%) ",
                (double)corr->delay_corr / max_percent_value * 100.);
    }

    if(qopt->jitter) {
        char jitter[MAX_STR_SIZE];

        get_us2tick();
        convert_unit_usec(jitter, sizeof(jitter), qopt->jitter / us2tick);
        *mp = add_log(msg, *mp, "jitter=%s ", jitter);
    }

    if(qopt->loss) {
        *mp = add_log(msg, *mp, "loss=%g(%%) ",
            (double)qopt->loss / max_percent_value * 100.);
        if(corr && corr->loss_corr)
            *mp = add_log(msg, *mp, "loss-correlation=%g(%%) ",
                (double)corr->loss_corr / max_percent_value * 100.);
    }

#if HAVE_DECL_TCA_NETEM_LOSS
    if(gimodel)
        *mp = add_log(msg, *mp, "loss-state(p13/p31/p32/p23/p14)="
            "%g(%%)/%g(%%)/%g(%%)/%g(%%)/%g(%%) ",
            (double)gimodel->p13 / max_percent_value * 100.,
            (double)gimodel->p31 / max_percent_value * 100.,
            (double)gimodel->p32 / max_percent_value * 100.,
            (double)gimodel->p23 / max_percent_value * 100.,
            (double)gimodel->p14 / max_percent_value * 100.);

    if(gemodel)
        *mp = add_log(msg, *mp, "loss-gemodel(p/r/1-h/1-k)="
            "%g(%%)/%g(%%)/%g(%%)/%g(%%) ",
            (double)gemodel->p / max_percent_value * 100.,
            (double)gemodel->r / max_percent_value * 100.,
            (double)gemodel->h / max_percent_value * 100.,
            (double)gemodel->k1 / max_percent_value * 100.);
#endif

    if(qopt->duplicate) {
        *mp = add_log(msg, *mp, "duplicate=%g(%%) ",
            (double)qopt->duplicate / max_percent_value * 100.);
        if(corr && corr->dup_corr)
            *mp = add_log(msg, *mp, "duplicate-correlation=%g(%%) ",
                (double)corr->dup_corr / max_percent_value * 100.);
    }

    if(reorder && reorder->probability) {
        *mp = add_log(msg, *mp, "reorder=%g(%%) ",
            (double)reorder->probability / max_percent_value * 100.);
        if(reorder->correlation)
            *mp = add_log(msg, *mp, "reorder-correlation=%g(%%) ",
                (double)reorder->correlation / max_percent_value * 100.);
    }

    if(corrupt && corrupt->probability) {
        *mp = add_log(msg, *mp, "corrupt=%g(%%) ",
            (double)corrupt->probability / max_percent_value * 100.);
        if(corrupt->correlation)
            *mp = add_log(msg, *mp, "corrupt-correlation=%g(%%) ",
                (double)corrupt->correlation / max_percent_value * 100.);
    }

#if HAVE_DECL_TCA_NETEM_RATE
    if(rate && rate->rate) {
        char netem_rate[MAX_STR_SIZE];

        convert_unit_rate(netem_rate, sizeof(netem_rate), rate->rate);
        *mp = add_log(msg, *mp, "rate=%s ", netem_rate);
                if(rate->packet_overhead)
            *mp = add_log(msg, *mp, "packet-overhead=%u(byte) ", rate->packet_overhead);

                if(rate->cell_size)
            *mp = add_log(msg, *mp, "cell-size=%u(byte) ", rate->cell_size);

                if(rate->cell_overhead)
            *mp = add_log(msg, *mp, "cell-overhead=%u(byte) ", rate->cell_overhead);
    }
#endif

    if(qopt->gap)
        *mp = add_log(msg, *mp, "gap=%u(packet) ", qopt->gap);

#if HAVE_DECL_TCA_NETEM_ECN
    if(netem[TCA_NETEM_ECN] && *(unsigned *)RTA_DATA(netem[TCA_NETEM_ECN]))
        *mp = add_log(msg, *mp, "ecn=on ");
#endif

    return(0);
}

#if HAVE_DECL_TCA_DRR_UNSPEC
/*
 * parse drr options
 */
int parse_tca_options_drr(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *drr[__TCA_NETEM_MAX];
    char quantum[MAX_STR_SIZE];

    parse_drr(drr, tca);

    if(drr[TCA_DRR_QUANTUM]) {
        if(RTA_PAYLOAD(drr[TCA_DRR_QUANTUM]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_DRR_QUANTUM: payload too short", __func__);
            return(1);
        }
        convert_unit_size(quantum, sizeof(quantum),
            (double)*(unsigned *)RTA_DATA(drr[TCA_DRR_QUANTUM]));
        *mp = add_log(msg, *mp, "quantum=%s ", quantum);
    }

    return(0);
}
#endif

#if HAVE_DECL_TCA_SFB_UNSPEC
/*
 * parse sfb options
 */
int parse_tca_options_sfb(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *sfb[__TCA_NETEM_MAX];

    parse_sfb(sfb, tca);

    if(sfb[TCA_SFB_PARMS]) {
        struct tc_sfb_qopt *qopt;
        char rehash[MAX_STR_SIZE];
        char warmup[MAX_STR_SIZE];

        if(RTA_PAYLOAD(sfb[TCA_SFB_PARMS]) < sizeof(*qopt)) {
            rec_log("error: %s: TCA_SFB_PARMS: payload too short", __func__);
            return(1);
        }
        qopt = (struct tc_sfb_qopt *)RTA_DATA(sfb[TCA_SFB_PARMS]);

        convert_unit_usec(rehash, sizeof(rehash), qopt->rehash_interval * 1000);
        convert_unit_usec(warmup, sizeof(warmup), qopt->warmup_time * 1000);

        *mp = add_log(msg, *mp, "limit=%u(packet) max=%u(packet) "
            "target=%u(packet) increment=%.5f decrement=%.5f "
            "penalty-rate=%u(packet/s) penalty-burst=%u(packet) "
            "rehash=%s warmup=%s ",
            qopt->limit, qopt->max, qopt->bin_size,
            (double)qopt->increment / SFB_MAX_PROB,
            (double)qopt->decrement / SFB_MAX_PROB,
            qopt->penalty_rate, qopt->penalty_burst,
            rehash, warmup);
    }

    return(0);
}
#endif

#if HAVE_DECL_TCA_QFQ_UNSPEC
/*
 * parse qfq options
 */
int parse_tca_options_qfq(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *qfq[__TCA_QFQ_MAX];

    parse_qfq(qfq, tca);

    if(qfq[TCA_QFQ_WEIGHT]) {
        if(RTA_PAYLOAD(qfq[TCA_QFQ_WEIGHT]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_QFQ_WEIGHT: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "weight=%u ",
            *(unsigned *)RTA_DATA(qfq[TCA_QFQ_WEIGHT]));
    }

    if(qfq[TCA_QFQ_LMAX]) {
        if(RTA_PAYLOAD(qfq[TCA_QFQ_MAX]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_QFQ_WEIGHT: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "maxpkt=%u(byte) ",
            *(unsigned *)RTA_DATA(qfq[TCA_QFQ_LMAX]));
    }

    return(0);
}
#endif

#if HAVE_DECL_TCA_CODEL_UNSPEC
/*
 * parse codel options
 */
int parse_tca_options_codel(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *codel[__TCA_CODEL_MAX];

    parse_codel(codel, tca);

    if(codel[TCA_CODEL_LIMIT]) {
        if(RTA_PAYLOAD(codel[TCA_CODEL_LIMIT]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_CODEL_LIMIT: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "limit=%u(packet) ",
            *(unsigned *)RTA_DATA(codel[TCA_CODEL_LIMIT]));
    }

    if(codel[TCA_CODEL_TARGET]) {
        char target[MAX_STR_SIZE];

        if(RTA_PAYLOAD(codel[TCA_CODEL_TARGET]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_CODEL_TARGET: payload too short", __func__);
            return(1);
        }
        convert_unit_usec(target, sizeof(target),
            (double)*(unsigned *)RTA_DATA(codel[TCA_CODEL_TARGET]));
        *mp = add_log(msg, *mp, "target=%s ", target);
    }

    if(codel[TCA_CODEL_INTERVAL]) {
        char interval[MAX_STR_SIZE];

        if(RTA_PAYLOAD(codel[TCA_CODEL_INTERVAL]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_CODEL_INTERVAL: payload too short", __func__);
            return(1);
        }
        convert_unit_usec(interval, sizeof(interval),
            (double)*(unsigned *)RTA_DATA(codel[TCA_CODEL_INTERVAL]));
        *mp = add_log(msg, *mp, "interval=%s ", interval);
    }

    if(codel[TCA_CODEL_ECN]) {
        if(RTA_PAYLOAD(codel[TCA_CODEL_ECN]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_CODEL_ECN: payload too short", __func__);
            return(1);
        }
        if(*(unsigned *)RTA_DATA(codel[TCA_CODEL_ECN]))
            *mp = add_log(msg, *mp, "ecn=on ");
    }

    return(0);
}
#endif

#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
/*
 * parse fq_codel options
 */
int parse_tca_options_fq_codel(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *fq_codel[__TCA_FQ_CODEL_MAX];

    parse_fq_codel(fq_codel, tca);

    if(fq_codel[TCA_FQ_CODEL_LIMIT]) {
        if(RTA_PAYLOAD(fq_codel[TCA_FQ_CODEL_LIMIT]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_FQ_CODEL_LIMIT: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "limit=%u(packet) ",
            *(unsigned *)RTA_DATA(fq_codel[TCA_FQ_CODEL_LIMIT]));
    }

    if(fq_codel[TCA_FQ_CODEL_FLOWS]) {
        if(RTA_PAYLOAD(fq_codel[TCA_FQ_CODEL_FLOWS]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_FQ_CODEL_FLOWS: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "flows=%u ",
            *(unsigned *)RTA_DATA(fq_codel[TCA_FQ_CODEL_FLOWS]));
    }

    if(fq_codel[TCA_FQ_CODEL_QUANTUM]) {
        if(RTA_PAYLOAD(fq_codel[TCA_FQ_CODEL_QUANTUM]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_FQ_CODEL_QUANTUM: payload too short", __func__);
            return(1);
        }
        *mp = add_log(msg, *mp, "quantum=%u(byte) ",
            *(unsigned *)RTA_DATA(fq_codel[TCA_FQ_CODEL_QUANTUM]));
    }

    if(fq_codel[TCA_FQ_CODEL_TARGET]) {
        char target[MAX_STR_SIZE];

        if(RTA_PAYLOAD(fq_codel[TCA_FQ_CODEL_TARGET]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_FQ_CODEL_TARGET: payload too short", __func__);
            return(1);
        }
        convert_unit_usec(target, sizeof(target),
            (double)*(unsigned *)RTA_DATA(fq_codel[TCA_FQ_CODEL_TARGET]));
        *mp = add_log(msg, *mp, "target=%s ", target);
    }

    if(fq_codel[TCA_FQ_CODEL_INTERVAL]) {
        char interval[MAX_STR_SIZE];

        if(RTA_PAYLOAD(fq_codel[TCA_FQ_CODEL_INTERVAL]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_FQ_CODEL_INTERVAL: payload too short", __func__);
            return(1);
        }
        convert_unit_usec(interval, sizeof(interval),
            (double)*(unsigned *)RTA_DATA(fq_codel[TCA_FQ_CODEL_INTERVAL]));
        *mp = add_log(msg, *mp, "interval=%s ", interval);
    }

    if(fq_codel[TCA_FQ_CODEL_ECN]) {
        if(RTA_PAYLOAD(fq_codel[TCA_FQ_CODEL_ECN]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_FQ_CODEL_ECN: payload too short", __func__);
            return(1);
        }
        if(*(unsigned *)RTA_DATA(fq_codel[TCA_FQ_CODEL_ECN]))
            *mp = add_log(msg, *mp, "ecn=on ");
    }

    return(0);
}
#endif

/*
 * convert unit data rate
 */
void convert_unit_rate(char *str, int len, double num)
{
    num *= 8.0;

    if(num >= 1000.0 * 1000.0 * 1000.0)
        snprintf(str, len, "%.3f(Gbit/s)", num / (1000.0 * 1000.0 * 1000.0));
    else if(num >= 1000.0 * 1000.0)
        snprintf(str, len, "%.3f(Mbit/s)", num / (1000.0 * 1000.0));
    else if(num >= 1000.0)
        snprintf(str, len, "%.3f(kbit/s)", num / 1000.0);
    else
        snprintf(str, len, "%.3f(bit/s)", num);
}

/*
 * convert unit data size
 */
void convert_unit_size(char *str, int len, double num)
{
    if(num >= 1024.0 * 1024.0 * 1024.0)
        snprintf(str, len, "%.3f(Gbyte)", num / (1024.0 * 1024.0 * 1024.0));
    else if(num >= 1024.0 * 1024.0)
        snprintf(str, len, "%.3f(Mbyte)", num / (1024.0 * 1024.0));
    else if(num >= 1024.0)
        snprintf(str, len, "%.3f(Kbyte)", num / 1024.0);
    else
        snprintf(str, len, "%.3f(byte)", num);
}

/*
 * convert unit of time
 */
void convert_unit_usec(char *str, int len, double usec)
{
    if(usec >= 1000.0 * 1000.0)
        snprintf(str, len, "%.3f(s)", usec / (1000.0 * 1000.0));
    else if(usec >= 1000.0)
        snprintf(str, len, "%.3f(ms)", usec / 1000.0);
    else if(usec < 0)
        snprintf(str, len, "%.3f(us)", 0.0);
    else
        snprintf(str, len, "%.3f(us)", usec);
}

/*
 * get ticks per usec
 */
int get_us2tick(void)
{
    FILE *fp;
    __u32 clock_res;
    __u32 us2ns;
    __u32 tick2ns;

    fp = fopen("/proc/net/psched", "r");
    if(fp == NULL)
        return -1;

    if(fscanf(fp, "%08x%08x%08x", &us2ns, &tick2ns, &clock_res) != 3) {
        fclose(fp);
        return(-1);
    }

    fclose(fp);

    if(clock_res == 1000000000)
        us2ns = tick2ns;

    clock_factor  = (double)clock_res / TIME_UNITS_PER_SEC;
    us2tick = (double)us2ns / tick2ns * clock_factor;

    return(0);
}

/*
 * get burst size
 *
 * (rate(byte/sec) * (buffer(tick) / us2tick(tick/usec))) / TIME_UNITS_PER_SEC(usec/sec)
 */
double get_burst_size(unsigned rate, unsigned buffer)
{
    return((double)rate * (buffer / us2tick)) / TIME_UNITS_PER_SEC;
}

/*
 * get latency
 */
double get_latency(unsigned rate, unsigned buffer, unsigned limit)
{
    return(TIME_UNITS_PER_SEC * (limit / (double)rate) - (buffer / us2tick));
}

/*
 * debug traffic control message
 */
void debug_tcmsg(int lev, struct nlmsghdr *nlh, struct tcmsg *tcm, struct rtattr *tca[], int tcm_len)
{
    /* debug tcmsg */
    char ifname[IFNAMSIZ] = "";
    char handle[MAX_STR_SIZE] = "";
    char parent[MAX_STR_SIZE] = "";
    char kind[IFNAMSIZ] = "";

    if_indextoname_from_lists(tcm->tcm_ifindex, ifname);

    switch(nlh->nlmsg_type) {
        case RTM_NEWQDISC:
        case RTM_DELQDISC:
        case RTM_NEWTCLASS:
        case RTM_DELTCLASS:
            parse_tc_classid(handle, sizeof(handle), tcm->tcm_handle);
            break;
        case RTM_NEWTFILTER:
        case RTM_DELTFILTER:
            parse_u32_handle(handle, sizeof(handle), tcm->tcm_handle);
            break;

    }
    parse_tc_classid(parent, sizeof(parent), tcm->tcm_parent);

    rec_dbg(lev, "*********************************************************************");

    rec_dbg(lev, "[ tcmsg(%d) ]",
        NLMSG_ALIGN(sizeof(*tcm)));
    rec_dbg(lev, "    tcm_family(%d): %d(%s)",
        sizeof(tcm->tcm_family), tcm->tcm_family,
        convert_af_type(tcm->tcm_family));
    rec_dbg(lev, "    tcm__pad1(%d): %d",
        sizeof(tcm->tcm__pad1), tcm->tcm__pad1);
    rec_dbg(lev, "    tcm__pad2(%d): %d",
        sizeof(tcm->tcm__pad2), tcm->tcm__pad2);
    rec_dbg(lev, "    tcm_ifindex(%d): %d(%s)",
        sizeof(tcm->tcm_ifindex), tcm->tcm_ifindex, ifname);
    rec_dbg(lev, "    tcm_handle(%d): 0x%08x(%s)",
        sizeof(tcm->tcm_handle), tcm->tcm_handle, handle);
    rec_dbg(lev, "    tcm_parent(%d): 0x%08x(%s)",
        sizeof(tcm->tcm_parent), tcm->tcm_parent, parent);
    rec_dbg(lev, "    tcm_info(%d): 0x%08x(%u, %s)",
        sizeof(tcm->tcm_info), tcm->tcm_info,
        TC_H_MAJ(tcm->tcm_info)>>16,
        convert_eth_p(TC_H_MIN(tcm->tcm_info), 1));

    /* debug traffic control attributes */
    rec_dbg(lev,"*********************************************************************");
    rec_dbg(lev, "[ tcmsg attributes(%d) ]",
            NLMSG_ALIGN(tcm_len - NLMSG_ALIGN(sizeof(*tcm))));

    if(tca[TCA_KIND])
        debug_tca_kind(lev+1, tca[TCA_KIND], kind, sizeof(kind));

    if(tca[TCA_OPTIONS])
        debug_tca_options(lev+1, tcm, tca[TCA_OPTIONS], kind, sizeof(kind));

    if(tca[TCA_STATS])
        debug_tca_stats(lev+1, tca[TCA_STATS]);

    if(tca[TCA_XSTATS])
        debug_tca_xstats(lev+1, tca[TCA_XSTATS], kind, sizeof(kind));

    if(tca[TCA_RATE])
        debug_tca_rate(lev+1, tca[TCA_RATE]);

    if(tca[TCA_FCNT])
        debug_tca_fcnt(lev+1, tca[TCA_FCNT]);

    if(tca[TCA_STATS2])
        debug_tca_stats2(lev+1, tca[TCA_STATS2]);

#if HAVE_DECL_TCA_STAB_UNSPEC
    if(tca[TCA_STAB])
        debug_tca_stab(lev+1, tca[TCA_STAB]);
#endif

    rec_dbg(lev, "");
}

/*
 * debug attribute TCA_KIND
 */
void debug_tca_kind(int lev, struct rtattr *tca, char *kind, int len)
{
    if(RTA_PAYLOAD(tca) > len) {
        rec_dbg(lev, "TCA_KIND(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    strncpy(kind, (char *)RTA_DATA(tca), len);

    rec_dbg(lev, "TCA_KIND(%hu): %s", RTA_ALIGN(tca->rta_len), kind);
}

/*
 * debug attribute TCA_OPTIONS
 */
void debug_tca_options(int lev, struct tcmsg *tcm, struct rtattr *tca, char *kind, int len)
{
    /* kinds of qdisc */
    if(!strncmp(kind, "pfifo_fast", len))
        debug_tca_options_prio(lev, tca);
    else if(!strncmp(kind, "pfifo", len))
        debug_tca_options_fifo(lev, tca);
    else if(!strncmp(kind, "bfifo", len))
        debug_tca_options_fifo(lev, tca);
    else if(!strncmp(kind, "prio", len))
        debug_tca_options_prio(lev, tca);
#ifdef HAVE_STRUCT_TC_MULTIQ_QOPT_BANDS
    else if(!strncmp(kind, "multiq", len))
        debug_tca_options_multiq(lev, tca);
#endif
#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
    else if(!strncmp(kind, "plug", len))
        debug_tca_options_plug(lev, tca);
#endif
    else if(!strncmp(kind, "sfq", len))
        debug_tca_options_sfq(lev, tca);
    else if(!strncmp(kind, "tbf", len))
        debug_tca_options_tbf(lev, tca);
    else if(!strncmp(kind, "red", len))
        debug_tca_options_red(lev, tca);
    else if(!strncmp(kind, "gred", len))
        debug_tca_options_gred(lev, tca);
#if HAVE_DECL_TCA_CHOKE_UNSPEC
    else if(!strncmp(kind, "choke", len))
        debug_tca_options_choke(lev, tca);
#endif
    else if(!strncmp(kind, "htb", len))
        debug_tca_options_htb(lev, tca);
    else if(!strncmp(kind, "hfsc", len))
        debug_tca_options_hfsc(lev, tca);
    else if(!strncmp(kind, "cbq", len))
        debug_tca_options_cbq(lev, tca);
    else if(!strncmp(kind, "dsmark", len))
        debug_tca_options_dsmark(lev, tca);
    else if(!strncmp(kind, "netem", len))
        debug_tca_options_netem(lev, tca);
#if HAVE_DECL_TCA_DRR_UNSPEC
    else if(!strncmp(kind, "drr", len))
        debug_tca_options_drr(lev, tca);
#endif
#if HAVE_DECL_TCA_SFB_UNSPEC
    else if(!strncmp(kind, "sfb", len))
        debug_tca_options_sfb(lev, tca);
#endif
#if HAVE_DECL_TCA_QFQ_UNSPEC
    else if(!strncmp(kind, "qfq", len))
        debug_tca_options_qfq(lev, tca);
#endif
#if HAVE_DECL_TCA_CODEL_UNSPEC
    else if(!strncmp(kind, "codel", len))
        debug_tca_options_codel(lev, tca);
#endif
#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
    else if(!strncmp(kind, "fq_codel", len))
        debug_tca_options_fq_codel(lev, tca);
#endif
    else if(!strncmp(kind, "ingress", len))
        return;
    /* kinds of filter */
    else if(!strncmp(kind, "u32", len))
        debug_tca_options_u32(lev, tca);
    else if(!strncmp(kind, "rsvp", len))
        debug_tca_options_rsvp(lev, tcm, tca);
    else if(!strncmp(kind, "route", len))
        debug_tca_options_route(lev, tca);
    else if(!strncmp(kind, "fw", len))
        debug_tca_options_fw(lev, tca);
    else if(!strncmp(kind, "tcindex", len))
        debug_tca_options_tcindex(lev, tca);
#if HAVE_DECL_TCA_FLOW_UNSPEC
    else if(!strncmp(kind, "flow", len))
        debug_tca_options_flow(lev, tca);
#endif
    else if(!strncmp(kind, "basic", len))
        debug_tca_options_basic(lev, tca);
#if HAVE_DECL_TCA_CGROUP_UNSPEC
    else if(!strncmp(kind, "cgroup", len))
        debug_tca_options_cgroup(lev, tca);
#endif
    else
        rec_dbg(lev, "TCA_OPTIONS(%hu): -- unkonwn option %s --",
            RTA_ALIGN(tca->rta_len), kind);
}
 
/*
 * debug attribute TCA_STATS
 */
void debug_tca_stats(int lev, struct rtattr *tca)
{
    struct tc_stats *stats;

    if(RTA_PAYLOAD(tca) < sizeof(*stats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    stats = (struct tc_stats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_STATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_stats(%d) ]", sizeof(*stats));
    rec_dbg(lev, "        bytes(%d): %lu", sizeof(stats->bytes), stats->bytes);
    rec_dbg(lev, "        packets(%d): %u", sizeof(stats->packets), stats->packets);
    rec_dbg(lev, "        drops(%d): %u", sizeof(stats->drops), stats->drops);
    rec_dbg(lev, "        overlimits(%d): %u", sizeof(stats->overlimits), stats->overlimits);
    rec_dbg(lev, "        bps(%d): %u", sizeof(stats->bps), stats->bps);
    rec_dbg(lev, "        pps(%d): %u", sizeof(stats->pps), stats->pps);
    rec_dbg(lev, "        qlen(%d): %u", sizeof(stats->qlen), stats->qlen);
    rec_dbg(lev, "        backlog(%d): %u", sizeof(stats->backlog), stats->backlog);
}

/*
 * debug attribute TCA_XSTATS
 */
void debug_tca_xstats(int lev, struct rtattr *tca, char *kind, int len)
{
    /* for dummy */
    if(0)
        return;
#ifdef HAVE_STRUCT_TC_SFQ_XSTATS_ALLOT
    else if(!strncmp(kind, "sfq", len))
        debug_tc_sfq_xstats(lev, tca);
#endif
    else if(!strncmp(kind, "red", len))
        debug_tc_red_xstats(lev, tca);
#if HAVE_DECL_TCA_CHOKE_UNSPEC
    else if(!strncmp(kind, "choke", len))
        debug_tc_choke_xstats(lev, tca);
#endif
    else if(!strncmp(kind, "htb", len))
        debug_tc_htb_xstats(lev, tca);
    else if(!strncmp(kind, "cbq", len))
        debug_tc_cbq_xstats(lev, tca);
#if HAVE_DECL_TCA_DRR_UNSPEC
    else if(!strncmp(kind, "drr", len))
        debug_tc_drr_xstats(lev, tca);
#endif
#if HAVE_DECL_TCA_SFB_UNSPEC
    else if(!strncmp(kind, "sfb", len))
        debug_tc_sfb_xstats(lev, tca);
#endif
#if HAVE_DECL_TCA_QFQ_UNSPEC
    else if(!strncmp(kind, "qfq", len))
        debug_tc_qfq_xstats(lev, tca);
#endif
#if HAVE_DECL_TCA_CODEL_UNSPEC
    else if(!strncmp(kind, "codel", len))
        debug_tc_codel_xstats(lev, tca);
#endif
#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
    else if(!strncmp(kind, "fq_codel", len))
        debug_tc_fq_codel_xstats(lev, tca);
#endif
    else if(!strncmp(kind, "ingress", len))
        return;
    else
        rec_dbg(lev, "TCA_XSTATS(%hu): -- unkown kind %s --",
            RTA_ALIGN(tca->rta_len), kind);
}

/*
 * debug attribute TCA_RATE
 */
void debug_tca_rate(int lev, struct rtattr *tca)
{
    struct tc_estimator *rate;

    if(RTA_PAYLOAD(tca) < sizeof(*rate)) {
        rec_dbg(lev, "TCA_RATE(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    rate = (struct tc_estimator *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_RATE(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_estimator(%d) ]", sizeof(*rate));
    rec_dbg(lev, "        interval(%d): %d", sizeof(rate->interval), rate->interval);
    rec_dbg(lev, "        ewma_log(%d): %d", sizeof(rate->ewma_log), rate->ewma_log);
}

/*
 * debug attribute TCA_FCNT
 */
void debug_tca_fcnt(int lev, struct rtattr *tca)
{
    if(RTA_PAYLOAD(tca) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_RATE(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FCNT(%hu): 0x%08x",
        RTA_ALIGN(tca->rta_len), *(unsigned *)RTA_DATA(tca));
}

/*
 * debug attribute TCA_STATS2
 */
void debug_tca_stats2(int lev, struct rtattr *tca)
{
    rec_dbg(lev, "TCA_STATS2(%hu):",
        RTA_ALIGN(tca->rta_len));

    debug_tca_stats2_attr(lev, tca);
}

/*
 * debug attributes TCA_STATS_*
 */
void debug_tca_stats2_attr(int lev, struct rtattr *tca)
{
    struct rtattr *stats2[__TCA_STATS_MAX];

    parse_stats2(stats2, tca);

    if(stats2[TCA_STATS_BASIC])
        debug_tca_stats_basic(lev+1, stats2[TCA_STATS_BASIC]);

    if(stats2[TCA_STATS_RATE_EST])
        debug_tca_stats_rate_est(lev+1, stats2[TCA_STATS_RATE_EST]);

    if(stats2[TCA_STATS_QUEUE])
        debug_tca_stats_queue(lev+1, stats2[TCA_STATS_QUEUE]);

    if(stats2[TCA_STATS_APP])
        debug_tca_stats_app(lev, stats2[TCA_STATS_APP]);
}

/*
 * debug attribute TCA_STATS_BASIC
 */
void debug_tca_stats_basic(int lev, struct rtattr *stats2)
{
    struct gnet_stats_basic *basic;

    if(RTA_PAYLOAD(stats2) < sizeof(*basic)) {
        rec_dbg(lev, "TCA_STATS_BASIC(%hu): -- payload too short --",
            RTA_ALIGN(stats2->rta_len));
        return;
    }
    basic = (struct gnet_stats_basic *)RTA_DATA(stats2);

    rec_dbg(lev, "TCA_STATS_BASIC(%hu):", RTA_ALIGN(stats2->rta_len));
    rec_dbg(lev, "    [ gnet_stats_basic(%d) ]", sizeof(*basic));
    rec_dbg(lev, "        bytes(%d): %lu", sizeof(basic->bytes), basic->bytes);
    rec_dbg(lev, "        packets(%d): %u", sizeof(basic->packets), basic->packets);
}

/*
 * debug attribute TCA_STATS_RATE_EST
 */
void debug_tca_stats_rate_est(int lev, struct rtattr *stats2)
{
    struct gnet_stats_rate_est *rate;

    if(RTA_PAYLOAD(stats2) < sizeof(*rate)) {
        rec_dbg(lev, "TCA_STATS_RATE_EST(%hu): -- payload too short --",
            RTA_ALIGN(stats2->rta_len));
        return;
    }
    rate = (struct gnet_stats_rate_est *)RTA_DATA(stats2);

    rec_dbg(lev, "TCA_STATS_RATE_EST(%hu):", RTA_ALIGN(stats2->rta_len));
    rec_dbg(lev, "    [ gnet_stats_rate_est(%d) ]", sizeof(*rate));
    rec_dbg(lev, "        bps(%d): %u", sizeof(rate->bps), rate->bps);
    rec_dbg(lev, "        pps(%d): %u", sizeof(rate->pps), rate->pps);
}

/*
 * debug attribute TCA_STATS_QUEUE
 */
void debug_tca_stats_queue(int lev, struct rtattr *stats2)
{
    struct gnet_stats_queue *queue;

    if(RTA_PAYLOAD(stats2) < sizeof(*queue)) {
        rec_dbg(lev, "TCA_STATS_QUEUE(%hu): -- payload too short --",
            RTA_ALIGN(stats2->rta_len));
        return;
    }
    queue = (struct gnet_stats_queue *)RTA_DATA(stats2);

    rec_dbg(lev, "TCA_STATS_QUEUE(%hu):", RTA_ALIGN(stats2->rta_len));
    rec_dbg(lev, "    [ gnet_stats_queue(%d) ]", sizeof(*queue));
    rec_dbg(lev, "        qlen(%d): %u", sizeof(queue->qlen), queue->qlen);
    rec_dbg(lev, "        backlog(%d): %u", sizeof(queue->backlog), queue->backlog);
    rec_dbg(lev, "        drops(%d): %u", sizeof(queue->drops), queue->drops);
    rec_dbg(lev, "        requeue(%d): %u", sizeof(queue->requeues), queue->requeues);
    rec_dbg(lev, "        overlimits(%d): %u", sizeof(queue->overlimits), queue->overlimits);
}

/*
 * debug attribute TCA_STATS_APP
 */
void debug_tca_stats_app(int lev, struct rtattr *stats2)
{
    struct gnet_estimator *est;

    if(RTA_PAYLOAD(stats2) < sizeof(*est)) {
        rec_dbg(lev, "TCA_STATS_APP(%hu): -- payload too short --",
            RTA_ALIGN(stats2->rta_len));
        return;
    }
    est = (struct gnet_estimator *)RTA_DATA(stats2);

    rec_dbg(lev, "TCA_STATS_APP(%hu):", RTA_ALIGN(stats2->rta_len));
    rec_dbg(lev, "    [ gnet_estimator(%d) ]", sizeof(*est));
    rec_dbg(lev, "        interval(%d): %d", sizeof(est->interval), est->interval);
    rec_dbg(lev, "        ewma_log(%d): %d", sizeof(est->ewma_log), est->ewma_log);
}

#if HAVE_DECL_TCA_STAB_UNSPEC
/*
 * debug attribute TCA_STAB
 */
void debug_tca_stab(int lev, struct rtattr *tca)
{
    struct rtattr *stab[__TCA_STAB_MAX];

    rec_dbg(lev, "TCA_STAB(%hu):",
        RTA_ALIGN(tca->rta_len));

    parse_stab(stab, tca);

    if(stab[TCA_STAB_BASE])
        debug_tca_stab_base(lev+1, stab[TCA_STAB_BASE]);

    if(stab[TCA_STAB_DATA])
        debug_tca_stab_data(lev+1, stab[TCA_STAB_DATA]);
}


/*
 * debug attribute TCA_STAB_BASE
 */
void debug_tca_stab_base(int lev, struct rtattr *stab)
{
    struct tc_sizespec *base;

    if(RTA_PAYLOAD(stab) < sizeof(*base)) {
        rec_dbg(lev, "TCA_STAB_BASE(%hu): -- payload too short --",
            RTA_ALIGN(stab->rta_len));
        return;
    }
    base = (struct tc_sizespec *)RTA_DATA(stab);

    rec_dbg(lev, "TCA_STATS_BASE(%hu):", RTA_ALIGN(stab->rta_len));
    rec_dbg(lev, "    [ tc_sizespec(%d) ]", sizeof(*base));
    rec_dbg(lev, "        cell_log(%d): %d", sizeof(base->cell_log), base->cell_log);
    rec_dbg(lev, "        size_log(%d): %d", sizeof(base->size_log), base->size_log);
    rec_dbg(lev, "        cell_align(%d): %hd", sizeof(base->cell_align), base->cell_align);
    rec_dbg(lev, "        overhead(%d): %d", sizeof(base->overhead), base->overhead);
    rec_dbg(lev, "        linklayer(%d): %u", sizeof(base->linklayer), base->linklayer);
    rec_dbg(lev, "        mpu(%d): %u", sizeof(base->mpu), base->mpu);
    rec_dbg(lev, "        mtu(%d): %u", sizeof(base->mtu), base->mtu);
    rec_dbg(lev, "        tsize(%d): %u", sizeof(base->tsize), base->tsize);
}

/*
 * debug attribute TCA_STAB_DATA
 */
void debug_tca_stab_data(int lev, struct rtattr *stab)
{
    rec_dbg(lev, "TCA_STATS_DATA(%hu): -- ignored --",
        RTA_ALIGN(stab->rta_len));
}
#endif

/*
 * debug prio options
 */
void debug_tca_options_prio(int lev, struct rtattr *tca)
{
    struct tc_prio_qopt *qopt;
    char prio[MAX_STR_SIZE] = "";
    char *p = prio;
    int i, len = sizeof(prio);

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_OPTIONS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    qopt = (struct tc_prio_qopt *)RTA_DATA(tca);

    for(i = 0; i < TC_PRIO_MAX + 1; i++) {
        if(i == TC_PRIO_MAX)
            p += snprintf(p, len - strlen(prio), "%d ", qopt->priomap[i]);
        else
            p += snprintf(p, len - strlen(prio), "%d-", qopt->priomap[i]);
        if(len < p - prio) {
            rec_dbg(lev, "TCA_OPTIONS(%hu): -- priomap too long --",
                RTA_ALIGN(tca->rta_len));
            return;
        }
    }

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_prio_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        bands(%d): %d", sizeof(qopt->bands), qopt->bands);
    rec_dbg(lev, "        priomap(%d): %s", sizeof(qopt->priomap), prio);
}

/*
 * debug fifo options
 */
void debug_tca_options_fifo(int lev, struct rtattr *tca)
{
    struct tc_fifo_qopt *qopt;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_OPTIONS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    qopt = (struct tc_fifo_qopt *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_fifo_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        limit(%d): %d", sizeof(qopt->limit), qopt->limit);
}

#ifdef HAVE_STRUCT_TC_MULTIQ_QOPT_BANDS
/*
 * debug multiq options
 */
void debug_tca_options_multiq(int lev, struct rtattr *tca)
{
    struct tc_multiq_qopt *qopt;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_OPTIONS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    qopt = (struct tc_multiq_qopt *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_multiq_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        bands(%d): %d", sizeof(qopt->bands), qopt->bands);
    rec_dbg(lev, "        max_bands(%d): %d", sizeof(qopt->max_bands), qopt->max_bands);
};
#endif

#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
/*
 * debug plug options
 */
void debug_tca_options_plug(int lev, struct rtattr *tca)
{
    struct tc_plug_qopt *qopt;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_OPTIONS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    qopt = (struct tc_plug_qopt *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_plug_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        action(%d): %d(%s)",
        sizeof(qopt->action), qopt->action, convert_tcq_plug_action(qopt->action));
    rec_dbg(lev, "        limit(%d): %u", qopt->limit);
}
#endif

/*
 * debug tbf options
 */
void debug_tca_options_tbf(int lev, struct rtattr *tca)
{
    struct rtattr *tbf[__TCA_TBF_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):",
        RTA_ALIGN(tca->rta_len));

    parse_tbf(tbf, tca);

    if(tbf[TCA_TBF_PARMS])
        debug_tca_tbf_parms(lev+1, tbf[TCA_TBF_PARMS]);

    if(tbf[TCA_TBF_RTAB])
        debug_tca_tbf_rtab(lev+1, tbf[TCA_TBF_RTAB]);

    if(tbf[TCA_TBF_PTAB])
        debug_tca_tbf_ptab(lev+1, tbf[TCA_TBF_PTAB]);
}

/*
 * debug attribute TCA_TBF_PARMS
 */
void debug_tca_tbf_parms(int lev, struct rtattr *tbf)
{
    struct tc_tbf_qopt *qopt;
    struct tc_ratespec *rate;
    struct tc_ratespec *peak;

    if(RTA_PAYLOAD(tbf) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_TBF_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(tbf->rta_len));
        return;
    }
    qopt = (struct tc_tbf_qopt *)RTA_DATA(tbf);
    rate = &(qopt->rate);
    peak = &(qopt->peakrate);

    rec_dbg(lev, "TCA_TBF_PARMS(%hu):", RTA_ALIGN(tbf->rta_len));
    rec_dbg(lev, "    [ tc_tbf_qopt(%d) ]", sizeof(*qopt));

    debug_tc_ratespec(lev+2, rate, "rate");
    debug_tc_ratespec(lev+2, peak, "peakrate");

    rec_dbg(lev, "        limit(%d): %u", sizeof(qopt->limit), qopt->limit);
    rec_dbg(lev, "        buffer(%d): %u", sizeof(qopt->buffer), qopt->buffer);
    rec_dbg(lev, "        mtu(%d): %u", sizeof(qopt->mtu), qopt->mtu);
}

/*
 * debug attribute TCA_TBF_RTAB
 */
void debug_tca_tbf_rtab(int lev, struct rtattr *tbf)
{
    rec_dbg(lev, "TCA_TBF_RTAB(%hu): -- ignored --", RTA_ALIGN(tbf->rta_len));
}

/*
 * debug attribute TCA_TBF_PTAB
 */
void debug_tca_tbf_ptab(int lev, struct rtattr *tbf)
{
    rec_dbg(lev, "TCA_TBF_PTAB(%hu): -- ignored --", 
        RTA_ALIGN(tbf->rta_len));
}

/*
 * debug sfq options
 */
void debug_tca_options_sfq(int lev, struct rtattr *tca)
{
    struct tc_sfq_qopt *qopt;
    struct tc_sfqred_stats *stats = NULL;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_OPTIONS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    qopt = (struct tc_sfq_qopt *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

#ifdef HAVE_STRUCT_TC_SFQ_QOPT_V1_V0
    struct tc_sfq_qopt_v1 *qopt_v1 = NULL;

    if(RTA_PAYLOAD(tca) >= sizeof(*qopt_v1)) {
        qopt_v1 = (struct tc_sfq_qopt_v1 *)RTA_DATA(tca);
        stats = &(qopt_v1->stats);
    }

    if(qopt_v1) {
        char list[MAX_STR_SIZE] = "";

        convert_tc_red_flags(qopt_v1->flags, list, sizeof(list), 1);

        rec_dbg(lev, "    [ tc_sfq_qopt_v1(%d) ]", sizeof(*qopt_v1));
        rec_dbg(lev, "        [ tc_sfq_qopt v0(%d) ]", sizeof(qopt_v1->v0));
        rec_dbg(lev, "            quantum(%d): %u", sizeof(qopt->quantum), qopt->quantum);
        rec_dbg(lev, "            perturb_period(%d): %d",
            sizeof(qopt->perturb_period), qopt->perturb_period);
        rec_dbg(lev, "            limit(%d): %u", sizeof(qopt->limit), qopt->limit);
        rec_dbg(lev, "            divisor(%d): %u", sizeof(qopt->divisor), qopt->divisor);
        rec_dbg(lev, "            flows(%d): %u", sizeof(qopt->flows), qopt->flows);
        rec_dbg(lev, "        depth(%d): %u", sizeof(qopt_v1->depth), qopt_v1->depth);
        rec_dbg(lev, "        headdrop(%d): %u", sizeof(qopt_v1->headdrop), qopt_v1->headdrop);
        rec_dbg(lev, "        limit(%d): %u", sizeof(qopt_v1->limit), qopt_v1->limit);
        rec_dbg(lev, "        qth_min(%d): %u", sizeof(qopt_v1->qth_min), qopt_v1->qth_min);
        rec_dbg(lev, "        qth_max(%d): %u", sizeof(qopt_v1->qth_max), qopt_v1->qth_max);
        rec_dbg(lev, "        Wlog(%d): %d", sizeof(qopt_v1->Wlog), qopt_v1->Wlog);
        rec_dbg(lev, "        Plog(%d): %d", sizeof(qopt_v1->Plog), qopt_v1->Plog);
        rec_dbg(lev, "        Scell_log(%d): %d",
            sizeof(qopt_v1->Scell_log), qopt_v1->Scell_log);
        rec_dbg(lev, "        flags(%d): %d(%s)",
            sizeof(qopt_v1->flags), qopt_v1->flags, list);
        rec_dbg(lev, "        max_P(%d): %u",
            sizeof(qopt_v1->max_P), qopt_v1->max_P);
        rec_dbg(lev, "        [ tc_sfqred_stats stats(%d) ]", sizeof(qopt_v1->stats));
        rec_dbg(lev, "            prob_drop(%d): %u",
            sizeof(stats->prob_drop), stats->prob_drop);
        rec_dbg(lev, "            forced_drop(%d): %u",
            sizeof(stats->forced_drop), stats->forced_drop);
        rec_dbg(lev, "            prob_mark(%d): %u",
            sizeof(stats->prob_mark), stats->prob_mark);
        rec_dbg(lev, "            forced_mark(%d): %u",
            sizeof(stats->forced_mark), stats->forced_mark);
        rec_dbg(lev, "            prob_mark_head(%d): %u",
            sizeof(stats->prob_mark_head), stats->prob_mark_head);
        rec_dbg(lev, "            forced_mark_head(%d): %u",
            sizeof(stats->forced_mark_head), stats->forced_mark_head);

        return;
    }
#endif

    rec_dbg(lev, "    [ tc_sfq_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        quantum(%d): %u", sizeof(qopt->quantum), qopt->quantum);
    rec_dbg(lev, "        perturb_period(%d): %d",
        sizeof(qopt->perturb_period), qopt->perturb_period);
    rec_dbg(lev, "        limit(%d): %u", sizeof(qopt->limit), qopt->limit);
    rec_dbg(lev, "        divisor(%d): %u", sizeof(qopt->divisor), qopt->divisor);
    rec_dbg(lev, "        flows(%d): %u", sizeof(qopt->flows), qopt->flows);
}

#ifdef HAVE_STRUCT_TC_SFQ_XSTATS_ALLOT
/*
 * debug tc_sfq_xstats
 */
void debug_tc_sfq_xstats(int lev, struct rtattr *tca)
{
    struct tc_sfq_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_sfq_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        allot(%d): %d", sizeof(xstats->allot), xstats->allot);
}
#endif

/*
 * debug red options
 */
void debug_tca_options_red(int lev, struct rtattr *tca)
{
    struct rtattr *red[__TCA_RED_MAX];

    parse_red(red, tca);

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    if(red[TCA_RED_PARMS])
        debug_tca_red_parms(lev+1, red[TCA_RED_PARMS]);

    if(red[TCA_RED_STAB])
        debug_tca_red_stab(lev+1, red[TCA_RED_STAB]);

#if HAVE_DECL_TCA_RED_MAX_P
    if(red[TCA_RED_MAX_P])
        debug_tca_red_max_p(lev+1, red[TCA_RED_MAX_P]);
#endif
}

/*
 * debug attribute TCA_RED_PARMS
 */
void debug_tca_red_parms(int lev, struct rtattr *red)
{
    struct tc_red_qopt *qopt;
    char list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(red) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_RED_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(red->rta_len));
        return;
    }
    qopt = (struct tc_red_qopt *)RTA_DATA(red);

    convert_tc_red_flags(qopt->flags, list, sizeof(list), 1);

    rec_dbg(lev, "TCA_RED_PARMS(%hu):", RTA_ALIGN(red->rta_len));
    rec_dbg(lev, "    [ tc_red_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        limit(%d): %d", sizeof(qopt->limit), qopt->limit);
    rec_dbg(lev, "        qth_min(%d): %u", sizeof(qopt->qth_min), qopt->qth_min);
    rec_dbg(lev, "        qth_max(%d): %u", sizeof(qopt->qth_max), qopt->qth_max);
    rec_dbg(lev, "        Wlog(%d): %d", sizeof(qopt->Wlog), qopt->Wlog);
    rec_dbg(lev, "        Plog(%d): %d", sizeof(qopt->Plog), qopt->Plog);
    rec_dbg(lev, "        Scell_log(%d): %d", sizeof(qopt->Scell_log), qopt->Scell_log);
    rec_dbg(lev, "        flags(%d): %d(%s)", sizeof(qopt->flags), qopt->flags, list);
}

/*
 * debug attribute TCA_RED_STAB
 */
void debug_tca_red_stab(int lev, struct rtattr *red)
{
    rec_dbg(lev, "TCA_RED_STAB(%hu): -- ignored --", RTA_ALIGN(red->rta_len));
}

#if HAVE_DECL_TCA_RED_MAX_P
/*
 * debug attribute TCA_RED_MAX_P
 */
void debug_tca_red_max_p(int lev, struct rtattr *red)
{
    if(RTA_PAYLOAD(red) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_RED_MAX_P(%hu): -- payload too short --",
            RTA_ALIGN(red->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_RED_MAX_P(%hu): %u",
        RTA_ALIGN(red->rta_len),
        *(unsigned *)RTA_DATA(red));
}
#endif

/*
 * debug tc_red_xstats
 */
void debug_tc_red_xstats(int lev, struct rtattr *tca)
{
    struct tc_red_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_red_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_red_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        early(%d): %u", sizeof(xstats->early), xstats->early);
    rec_dbg(lev, "        pdrop(%d): %u", sizeof(xstats->pdrop), xstats->pdrop);
    rec_dbg(lev, "        other(%d): %u", sizeof(xstats->other), xstats->other);
    rec_dbg(lev, "        marked(%d): %u", sizeof(xstats->marked), xstats->marked);
}

/*
 * debug gred options
 */
void debug_tca_options_gred(int lev, struct rtattr *tca)
{
    struct rtattr *gred[__TCA_GRED_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_gred(gred, tca);

    if(gred[TCA_GRED_PARMS])
        debug_tca_gred_parms(lev+1, gred[TCA_GRED_PARMS]);

    if(gred[TCA_GRED_STAB])
        debug_tca_gred_stab(lev+1, gred[TCA_GRED_STAB]);

    if(gred[TCA_GRED_DPS])
        debug_tca_gred_dps(lev+1, gred[TCA_GRED_DPS]);

#if HAVE_DECL_TCA_GRED_MAX_P
    if(gred[TCA_GRED_MAX_P])
        debug_tca_gred_max_p(lev+1, gred[TCA_GRED_MAX_P]);
#endif
}

/*
 * debug attribute TCA_GRED_PARMS
 */
void debug_tca_gred_parms(int lev, struct rtattr *gred)
{
    struct tc_gred_qopt *qopt;
    int i;

    if(RTA_PAYLOAD(gred) < sizeof(*qopt) * MAX_DPs) {
        rec_dbg(lev, "TCA_GRED_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(gred->rta_len));
        return;
    }
    qopt = (struct tc_gred_qopt *)RTA_DATA(gred);

    rec_dbg(lev, "TCA_GRED_PARMS(%hu):", RTA_ALIGN(gred->rta_len));

    for(i = 0; i < MAX_DPs; i++, qopt++) {
        rec_dbg(lev, "    [ tc_gred_qopt(%hu) ]", sizeof(*qopt));
        rec_dbg(lev, "        limit(%d): %u", sizeof(qopt->limit), qopt->limit);
        rec_dbg(lev, "        qth_min(%d): %u", sizeof(qopt->qth_min), qopt->qth_min);
        rec_dbg(lev, "        qth_max(%d): %u", sizeof(qopt->qth_max), qopt->qth_max);
        rec_dbg(lev, "        DP(%d): %u", sizeof(qopt->DP), qopt->DP);
        rec_dbg(lev, "        backlog(%d): %u", sizeof(qopt->backlog), qopt->backlog);
        rec_dbg(lev, "        qave(%d): %u", sizeof(qopt->qave), qopt->qave);
        rec_dbg(lev, "        forced(%d): %u", sizeof(qopt->forced), qopt->forced);
        rec_dbg(lev, "        early(%d): %u", sizeof(qopt->early), qopt->early);
        rec_dbg(lev, "        other(%d): %u", sizeof(qopt->other), qopt->other);
        rec_dbg(lev, "        pdrop(%d): %u", sizeof(qopt->pdrop), qopt->pdrop);
        rec_dbg(lev, "        Wlog(%d): %d", sizeof(qopt->Wlog), qopt->Wlog);
        rec_dbg(lev, "        Plog(%d): %d", sizeof(qopt->Plog), qopt->Plog);
        rec_dbg(lev, "        Scell_log(%d): %d", sizeof(qopt->Scell_log), qopt->Scell_log);
        rec_dbg(lev, "        prio(%d): %d", sizeof(qopt->prio), qopt->prio);
        rec_dbg(lev, "        packets(%d): %u", sizeof(qopt->packets), qopt->packets);
        rec_dbg(lev, "        bytesin(%d): %u", sizeof(qopt->bytesin), qopt->bytesin);
    }
}

/*
 * debug attribute TCA_GRED_STAB
 */
void debug_tca_gred_stab(int lev, struct rtattr *gred)
{
    rec_dbg(lev, "TCA_GRED_STAB(%hu): -- ignored --", RTA_ALIGN(gred->rta_len));
}

/*
 * debug attribute TCA_GRED_DPS
 */
void debug_tca_gred_dps(int lev, struct rtattr *gred)
{
    struct tc_gred_sopt *sopt;

    if(RTA_PAYLOAD(gred) < sizeof(*sopt)) {
        rec_dbg(lev, "TCA_GRED_DPS(%hu): -- payload too short --",
            RTA_ALIGN(gred->rta_len));
        return;
    }
    sopt = (struct tc_gred_sopt *)RTA_DATA(gred);

    rec_dbg(lev, "TCA_GRED_DPS(%hu):", RTA_ALIGN(gred->rta_len));
    rec_dbg(lev, "    [ tc_gred_sopt(%d) ]", sizeof(*sopt));
    rec_dbg(lev, "        DPs(%hu): %u", sizeof(sopt->DPs), sopt->DPs);
    rec_dbg(lev, "        def_DP(%hu): %u", sizeof(sopt->def_DP), sopt->def_DP);
    rec_dbg(lev, "        grio(%hu): %d", sizeof(sopt->grio), sopt->grio);
    rec_dbg(lev, "        flags(%hu): %d", sizeof(sopt->flags), sopt->flags);
    rec_dbg(lev, "        pad1(%hu): %d", sizeof(sopt->pad1), sopt->pad1);
}

#if HAVE_DECL_TCA_GRED_MAX_P
/*
 * debug attribute TCA_GRED_MAX_P
 */
void debug_tca_gred_max_p(int lev, struct rtattr *gred)
{
    unsigned *max_p = (unsigned *)RTA_DATA(gred);
    int i;

    if(RTA_PAYLOAD(gred) < sizeof(__u32) * MAX_DPs) {
        rec_dbg(lev, "TCA_GRED_MAX_P(%hu): -- payload too short --",
            RTA_ALIGN(gred->rta_len));
        return;
    }

    rec_dbg(lev, "TCA_GRED_MAX_P(%hu):", RTA_ALIGN(gred->rta_len));

    for(i = 0; i < MAX_DPs; i++)
        rec_dbg(lev+1, "%d(%hu): %u", i, sizeof(max_p[i]), max_p[i]);

}
#endif

#if HAVE_DECL_TCA_CHOKE_UNSPEC
/*
 * debug choke options
 */
void debug_tca_options_choke(int lev, struct rtattr *tca)
{
    struct rtattr *choke[__TCA_CHOKE_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_choke(choke, tca);

    if(choke[TCA_CHOKE_PARMS])
        debug_tca_choke_parms(lev+1, choke[TCA_CHOKE_PARMS]);

    if(choke[TCA_CHOKE_STAB])
        debug_tca_choke_stab(lev+1, choke[TCA_CHOKE_STAB]);

    if(choke[TCA_CHOKE_MAX_P])
        debug_tca_choke_max_p(lev+1, choke[TCA_CHOKE_MAX_P]);
}

/*
 * debug attribute TCA_CHOKE_PARMS
 */
void debug_tca_choke_parms(int lev, struct rtattr *choke)
{
    struct tc_choke_qopt *qopt;
    char list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(choke) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_CHOKE_PARMS(%hu):",
            RTA_ALIGN(choke->rta_len));
        return;
    }
    qopt = (struct tc_choke_qopt *)RTA_DATA(choke);

    convert_tc_red_flags(qopt->flags, list, sizeof(list), 1);

    rec_dbg(lev, "TCA_CHOKE_PARMS(%hu):", RTA_ALIGN(choke->rta_len));
    rec_dbg(lev, "    [ tc_choke_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        limit(%d): %d", sizeof(qopt->limit), qopt->limit);
    rec_dbg(lev, "        qth_min(%d): %u", sizeof(qopt->qth_min), qopt->qth_min);
    rec_dbg(lev, "        qth_max(%d): %u", sizeof(qopt->qth_max), qopt->qth_max);
    rec_dbg(lev, "        Wlog(%d): %d", sizeof(qopt->Wlog), qopt->Wlog);
    rec_dbg(lev, "        Plog(%d): %d", sizeof(qopt->Plog), qopt->Plog);
    rec_dbg(lev, "        Scell_log(%d): %d", sizeof(qopt->Scell_log), qopt->Scell_log);
    rec_dbg(lev, "        flags(%d): %d(%s)", sizeof(qopt->flags), qopt->flags, list);
}

/*
 * debug attribute TCA_CHOKE_STAB
 */
void debug_tca_choke_stab(int lev, struct rtattr *choke)
{
    rec_dbg(lev, "TCA_CHOKE_STAB(%hu): -- ignored --", RTA_ALIGN(choke->rta_len));
}

/*
 * debug attribute TCA_CHOKE_MAX_P
 */
void debug_tca_choke_max_p(int lev, struct rtattr *choke)
{
    if(RTA_PAYLOAD(choke) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_CHOKE_MAX_P(%hu): -- payload too short --",
            RTA_ALIGN(choke->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_CHOKE_MAX_P(%hu): %u",
        RTA_ALIGN(choke->rta_len), *(unsigned *)RTA_DATA(choke));
}

/*
 * debug tc_choke_xstats
 */
void debug_tc_choke_xstats(int lev, struct rtattr *tca)
{
    struct tc_choke_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_choke_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_choke_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        early(%d): %u", sizeof(xstats->early), xstats->early);
    rec_dbg(lev, "        pdrop(%d): %u", sizeof(xstats->pdrop), xstats->pdrop);
    rec_dbg(lev, "        other(%d): %u", sizeof(xstats->other), xstats->other);
    rec_dbg(lev, "        marked(%d): %u", sizeof(xstats->marked), xstats->marked);
    rec_dbg(lev, "        matched(%d): %u", sizeof(xstats->matched), xstats->matched);
}
#endif

/*
 * debug htb options
 */
void debug_tca_options_htb(int lev, struct rtattr *tca)
{
    struct rtattr *htb[__TCA_HTB_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_htb(htb, tca);

    if(htb[TCA_HTB_PARMS])
        debug_tca_htb_parms(lev+1, htb[TCA_HTB_PARMS]);

    if(htb[TCA_HTB_INIT])
        debug_tca_htb_init(lev+1, htb[TCA_HTB_INIT]);

    if(htb[TCA_HTB_CTAB])
        debug_tca_htb_ctab(lev+1, htb[TCA_HTB_CTAB]);

    if(htb[TCA_HTB_RTAB])
        debug_tca_htb_rtab(lev+1, htb[TCA_HTB_RTAB]);
}

/*
 * debug attribute TCA_HTB_PARMS
 */
void debug_tca_htb_parms(int lev, struct rtattr *htb)
{
    struct tc_htb_opt *opt;
    struct tc_ratespec *rate, *ceil;

    if(RTA_PAYLOAD(htb) < sizeof(*opt)) {
        rec_dbg(lev, "TCA_HTB_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(htb->rta_len));
        return;
    }
    opt = (struct tc_htb_opt *)RTA_DATA(htb);
    rate = &(opt->rate);
    ceil = &(opt->ceil);

    rec_dbg(lev, "TCA_HTB_PARMS(%hu):", RTA_ALIGN(htb->rta_len));
    rec_dbg(lev, "    [ tc_htb_opt(%d) ]", sizeof(*opt));

    debug_tc_ratespec(lev+2, rate, "rate");
    debug_tc_ratespec(lev+2, ceil, "ceil");

    rec_dbg(lev, "        buffer(%d): %u", sizeof(opt->buffer), opt->buffer);
    rec_dbg(lev, "        cbuffer(%d): %u", sizeof(opt->cbuffer), opt->cbuffer);
    rec_dbg(lev, "        quantum(%d): %u", sizeof(opt->quantum), opt->quantum);
    rec_dbg(lev, "        level(%d): %u", sizeof(opt->level), opt->level);
    rec_dbg(lev, "        prio(%d): %u", sizeof(opt->prio), opt->prio);
}

/*
 * debug attribute TCA_HTB_INIT
 */
void debug_tca_htb_init(int lev, struct rtattr *htb)
{
    struct tc_htb_glob *glob;

    if(RTA_PAYLOAD(htb) < sizeof(*glob)) {
        rec_dbg(lev, "TCA_HTB_INIT(%hu): -- payload too short --",
            RTA_ALIGN(htb->rta_len));
        return;
    }
    glob = (struct tc_htb_glob *)RTA_DATA(htb);

    rec_dbg(lev, "TCA_HTB_INIT(%hu):", RTA_ALIGN(htb->rta_len));
    rec_dbg(lev, "    [ tc_htb_glob(%d) ]", sizeof(*glob));
    rec_dbg(lev, "        version(%d): %u", sizeof(glob->version), glob->version);
    rec_dbg(lev, "        rate2quantum(%d): %u", sizeof(glob->rate2quantum), glob->rate2quantum);
    rec_dbg(lev, "        defcls(%d): 0x%x", sizeof(glob->defcls), glob->defcls);
    rec_dbg(lev, "        debug(%d): %u", sizeof(glob->debug), glob->debug);
    rec_dbg(lev, "        direct_pkts(%d): %u", sizeof(glob->direct_pkts), glob->direct_pkts);
}

/*
 * debug attribute TCA_HTB_CTAB
 */
void debug_tca_htb_ctab(int lev, struct rtattr *htb)
{
    rec_dbg(lev, "TCA_HTB_CTAB(%hu): -- ignored --", RTA_ALIGN(htb->rta_len));
}

/*
 * debug attribute TCA_HTB_RTAB
 */
void debug_tca_htb_rtab(int lev, struct rtattr *htb)
{
    rec_dbg(lev, "TCA_HTB_RTAB(%hu): -- ignored --", RTA_ALIGN(htb->rta_len));
}

/*
 * debug tc_htb_xstats
 */
void debug_tc_htb_xstats(int lev, struct rtattr *tca)
{
    struct tc_htb_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_htb_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_htb_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        lends(%d): %u", sizeof(xstats->lends), xstats->lends);
    rec_dbg(lev, "        borrows(%d): %u", sizeof(xstats->borrows), xstats->borrows);
    rec_dbg(lev, "        giants(%d): %u", sizeof(xstats->giants), xstats->giants);
    rec_dbg(lev, "        tokens(%d): %u", sizeof(xstats->tokens), xstats->tokens);
    rec_dbg(lev, "        ctokens(%d): %u", sizeof(xstats->ctokens), xstats->ctokens);
}

/*
 * debug hfsc options
 */
void debug_tca_options_hfsc(int lev, struct rtattr *tca)
{
    struct tc_hfsc_qopt *qopt;
    struct rtattr *hfsc[__TCA_HFSC_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    if(RTA_PAYLOAD(tca) == sizeof(*qopt)) {
        qopt = (struct tc_hfsc_qopt *)RTA_DATA(tca);
        rec_dbg(lev, "    [ tc_hfsc_qopt(%d) ]", sizeof(*qopt));
        rec_dbg(lev, "        defcls(%d): 0x%x", sizeof(qopt->defcls), qopt->defcls);

        return;
    }

    parse_hfsc(hfsc, tca);

    if(hfsc[TCA_HFSC_RSC])
        debug_tca_hfsc_rsc(lev+1, hfsc[TCA_HFSC_RSC]);

    if(hfsc[TCA_HFSC_FSC])
        debug_tca_hfsc_fsc(lev+1, hfsc[TCA_HFSC_FSC]);

    if(hfsc[TCA_HFSC_USC])
        debug_tca_hfsc_usc(lev+1, hfsc[TCA_HFSC_USC]);
}

/*
 * debug attribute TCA_HFSC_RSC
 */
void debug_tca_hfsc_rsc(int lev, struct rtattr *hfsc)
{
    struct tc_service_curve *sc;

    if(RTA_PAYLOAD(hfsc) < sizeof(*sc)) {
        rec_dbg(lev, "TCA_HFSC_RSC(%hu): -- payload too short --",
            RTA_ALIGN(hfsc->rta_len));
        return;
    }
    sc = (struct tc_service_curve *)RTA_DATA(hfsc);

    rec_dbg(lev, "TCA_HFSC_RSC(%hu):", RTA_ALIGN(hfsc->rta_len));
    rec_dbg(lev, "    [ tc_service_curve(%d) ]", sizeof(*sc));
    rec_dbg(lev, "        m1(%d): %u", sizeof(sc->m1), sc->m1);
    rec_dbg(lev, "        d(%d): %u", sizeof(sc->d), sc->d);
    rec_dbg(lev, "        m2(%d): %u", sizeof(sc->m2), sc->m2);
}

/*
 * debug attribute TCA_HFSC_FSC
 */
void debug_tca_hfsc_fsc(int lev, struct rtattr *hfsc)
{
    struct tc_service_curve *sc;

    if(RTA_PAYLOAD(hfsc) < sizeof(*sc)) {
        rec_dbg(lev, "TCA_HFSC_FSC(%hu): -- payload too short --",
            RTA_ALIGN(hfsc->rta_len));
        return;
    }
    sc = (struct tc_service_curve *)RTA_DATA(hfsc);

    rec_dbg(lev, "TCA_HFSC_FSC(%hu):", RTA_ALIGN(hfsc->rta_len));
    rec_dbg(lev, "    [ tc_service_curve(%d) ]", sizeof(*sc));
    rec_dbg(lev, "        m1(%d): %u", sizeof(sc->m1), sc->m1);
    rec_dbg(lev, "        d(%d): %u", sizeof(sc->d), sc->d);
    rec_dbg(lev, "        m2(%d): %u", sizeof(sc->m2), sc->m2);
}

/*
 * debug attribute TCA_HFSC_USC
 */
void debug_tca_hfsc_usc(int lev, struct rtattr *hfsc)
{
    struct tc_service_curve *sc;

    if(RTA_PAYLOAD(hfsc) < sizeof(*sc)) {
        rec_dbg(lev, "TCA_HFSC_FSC(%hu): -- payload too short --",
            RTA_ALIGN(hfsc->rta_len));
        return;
    }
    sc = (struct tc_service_curve *)RTA_DATA(hfsc);

    rec_dbg(lev, "TCA_HFSC_USC(%hu):", RTA_ALIGN(hfsc->rta_len));
    rec_dbg(lev, "    [ tc_service_curve(%d) ]", sizeof(*sc));
    rec_dbg(lev, "        m1(%d): %u", sizeof(sc->m1), sc->m1);
    rec_dbg(lev, "        d(%d): %u", sizeof(sc->d), sc->d);
    rec_dbg(lev, "        m2(%d): %u", sizeof(sc->m2), sc->m2);
}

/*
 * debug cbq options
 */
void debug_tca_options_cbq(int lev, struct rtattr *tca)
{
    struct rtattr *cbq[__TCA_CBQ_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):",
        RTA_ALIGN(tca->rta_len));

    parse_cbq(cbq, tca);

    if(cbq[TCA_CBQ_LSSOPT])
        debug_tca_cbq_lssopt(lev+1, cbq[TCA_CBQ_LSSOPT]);

    if(cbq[TCA_CBQ_WRROPT])
        debug_tca_cbq_wrropt(lev+1, cbq[TCA_CBQ_WRROPT]);

    if(cbq[TCA_CBQ_FOPT])
        debug_tca_cbq_fopt(lev+1, cbq[TCA_CBQ_FOPT]);

    if(cbq[TCA_CBQ_OVL_STRATEGY])
        debug_tca_cbq_ovl_strategy(lev+1, cbq[TCA_CBQ_OVL_STRATEGY]);

    if(cbq[TCA_CBQ_RATE])
        debug_tca_cbq_rate(lev+1, cbq[TCA_CBQ_RATE]);

    if(cbq[TCA_CBQ_RTAB])
        debug_tca_cbq_rtab(lev+1, cbq[TCA_CBQ_RTAB]);

    if(cbq[TCA_CBQ_POLICE])
        debug_tca_cbq_police(lev+1, cbq[TCA_CBQ_POLICE]);
}

/*
 * debug attribute TCA_CBQ_LSSOPT
 */
void debug_tca_cbq_lssopt(int lev, struct rtattr *cbq)
{
    struct tc_cbq_lssopt *lss;
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(cbq) < sizeof(*lss)) {
        rec_dbg(lev, "TCA_CBQ_LSSOPT(%hu): -- payload too short --",
            RTA_ALIGN(cbq->rta_len));
        return;
    }
    lss = (struct tc_cbq_lssopt *)RTA_DATA(cbq);

    convert_tcf_cbq_lss_flags(lss->flags, flags_list, sizeof(flags_list));

    rec_dbg(lev, "TCA_CBQ_LSSOPT(%hu):", RTA_ALIGN(cbq->rta_len));
    rec_dbg(lev, "    [ tc_cbq_lssopt(%d) ]", sizeof(*lss));
    rec_dbg(lev, "        change(%d): %d", sizeof(lss->change), lss->change);
    rec_dbg(lev, "        flags(%d): %d(%s)", sizeof(lss->flags), lss->flags, flags_list);
    rec_dbg(lev, "        ewma_log(%d): %d", sizeof(lss->ewma_log), lss->ewma_log);
    rec_dbg(lev, "        level(%d): %d", sizeof(lss->level), lss->level);
    rec_dbg(lev, "        maxidle(%d): %u", sizeof(lss->maxidle), lss->maxidle);
    rec_dbg(lev, "        minidle(%d): %u", sizeof(lss->minidle), lss->minidle);
    rec_dbg(lev, "        offtime(%d): %u", sizeof(lss->offtime), lss->offtime);
    rec_dbg(lev, "        avpkt(%d): %u", sizeof(lss->avpkt), lss->avpkt);
}

/*
 * debug attribute TCA_CBQ_WRROPT
 */
void debug_tca_cbq_wrropt(int lev, struct rtattr *cbq)
{
    struct tc_cbq_wrropt *wrr;

    if(RTA_PAYLOAD(cbq) < sizeof(*wrr)) {
        rec_dbg(lev, "TCA_CBQ_WRROPT(%hu): -- payload too short --",
            RTA_ALIGN(cbq->rta_len));
        return;
    }
    wrr = (struct tc_cbq_wrropt *)RTA_DATA(cbq);

    rec_dbg(lev, "TCA_CBQ_WRROPT(%hu):", RTA_ALIGN(cbq->rta_len));
    rec_dbg(lev, "    [ tc_cbq_wrropt(%d) ]", sizeof(*wrr));
    rec_dbg(lev, "        flags(%d): %d", sizeof(wrr->flags), wrr->flags);
    rec_dbg(lev, "        priority(%d): %d", sizeof(wrr->priority), wrr->priority);
    rec_dbg(lev, "        cpriority(%d): %d", sizeof(wrr->cpriority), wrr->cpriority);
    rec_dbg(lev, "        __reserved(%d): %d", sizeof(wrr->__reserved), wrr->__reserved);
    rec_dbg(lev, "        allot(%d): %u", sizeof(wrr->allot), wrr->allot);
    rec_dbg(lev, "        weight(%d): %u", sizeof(wrr->weight), wrr->weight);
}

/*
 * debug attribute TCA_CBQ_FOPT
 */
void debug_tca_cbq_fopt(int lev, struct rtattr *cbq)
{
    struct tc_cbq_fopt *fopt;
    char split[MAX_STR_SIZE];

    if(RTA_PAYLOAD(cbq) < sizeof(*fopt)) {
        rec_dbg(lev, "TCA_CBQ_FOPT(%hu): -- payload too short --",
            RTA_ALIGN(cbq->rta_len));
        return;
    }
    fopt = (struct tc_cbq_fopt *)RTA_DATA(cbq);

    parse_tc_classid(split, sizeof(split), fopt->split);

    rec_dbg(lev, "TCA_CBQ_FOPT(%hu):", RTA_ALIGN(cbq->rta_len));
    rec_dbg(lev, "    [ tc_cbq_fopt(%d) ]", sizeof(*fopt));
    rec_dbg(lev, "        split(%d): %x:%x(%s)",
        sizeof(fopt->split), TC_H_MAJ(fopt->split)>>16,
        TC_H_MIN(fopt->split), split);
    rec_dbg(lev, "        defmap(%d): 0x%x", sizeof(fopt->defmap), fopt->defmap);
    rec_dbg(lev, "        defchange(%d): 0x%x", sizeof(fopt->defchange), fopt->defchange);
}

/*
 * debug attribute TCA_CBQ_OVL_STRATEGY
 */
void debug_tca_cbq_ovl_strategy(int lev, struct rtattr *cbq)
{
    struct tc_cbq_ovl *ovl;
    char strategy_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(cbq) < sizeof(*ovl)) {
        rec_dbg(lev, "TCA_CBQ_OVL_STRATEGY(%hu): -- payload too short --",
            RTA_ALIGN(cbq->rta_len));
        return;
    }
    ovl = (struct tc_cbq_ovl *)RTA_DATA(cbq);

    convert_tc_cbq_ovl_strategy(ovl->strategy, strategy_list, sizeof(strategy_list));

    rec_dbg(lev, "TCA_CBQ_OVL_STRATEGY(%hu):", RTA_ALIGN(cbq->rta_len));
    rec_dbg(lev, "    [ tc_cbq_ovl(%d) ]", sizeof(*ovl));
    rec_dbg(lev, "        strategy(%d): %d(%s)",
        sizeof(ovl->strategy), ovl->strategy, strategy_list);
    rec_dbg(lev, "        priority2(%d): %d", sizeof(ovl->priority2), ovl->priority2);
    rec_dbg(lev, "        pad(%d): %d", sizeof(ovl->pad), ovl->pad);
    rec_dbg(lev, "        penalty(%d): %u", sizeof(ovl->penalty), ovl->penalty);
}

/*
 * debug attribute TCA_CBQ_RATE
 */
void debug_tca_cbq_rate(int lev, struct rtattr *cbq)
{
    struct tc_ratespec *spec;

    if(RTA_PAYLOAD(cbq) < sizeof(*spec)) {
        rec_dbg(lev, "TCA_CBQ_RATE(%hu): -- payload too short --",
            RTA_ALIGN(cbq->rta_len));
        return;
    }
    spec = (struct tc_ratespec *)RTA_DATA(cbq);

    rec_dbg(lev, "TCA_CBQ_RATE(%hu):", RTA_ALIGN(cbq->rta_len));

    debug_tc_ratespec(lev+1, spec, "");
}

/*
 * debug attribute TCA_CBQ_RTAB
 */
void debug_tca_cbq_rtab(int lev, struct rtattr *cbq)
{
    rec_dbg(lev, "TCA_CBQ_RTAB(%hu): -- ignored --", RTA_ALIGN(cbq->rta_len));
}

/*
 * debug attribute TCA_CBQ_POLICE
 */
void debug_tca_cbq_police(int lev, struct rtattr *cbq)
{
    struct tc_cbq_police *qopt;

    if(RTA_PAYLOAD(cbq) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_CBQ_POLICE(%hu): -- payload too short --",
            RTA_ALIGN(cbq->rta_len));
        return;
    }
    qopt = (struct tc_cbq_police *)RTA_DATA(cbq);

    rec_dbg(lev, "TCA_CBQ_POLICE(%hu):", RTA_ALIGN(cbq->rta_len));
    rec_dbg(lev, "    [ tc_cbq_police(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        police(%d): %d", sizeof(qopt->police), qopt->police);
    rec_dbg(lev, "        __res1(%d): %d", sizeof(qopt->__res1), qopt->__res1);
    rec_dbg(lev, "        __res2(%d): %d", sizeof(qopt->__res2), qopt->__res2);
}

/*
 * debug tc_cbq_xstats
 */
void debug_tc_cbq_xstats(int lev, struct rtattr *tca)
{
    struct tc_cbq_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_cbq_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_cbq_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        borrows(%d): %u", sizeof(xstats->borrows), xstats->borrows);
    rec_dbg(lev, "        overactions(%d): %u", sizeof(xstats->overactions), xstats->overactions);
    rec_dbg(lev, "        avgidle(%d): %d", sizeof(xstats->avgidle), xstats->avgidle);
    rec_dbg(lev, "        undertime(%d): %d", sizeof(xstats->undertime), xstats->undertime);
}

/*
 * debug dsmark options
 */
void debug_tca_options_dsmark(int lev, struct rtattr *tca)
{
    struct rtattr *dsmark[__TCA_DSMARK_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_dsmark(dsmark, tca);

    if(dsmark[TCA_DSMARK_INDICES])
        debug_tca_dsmark_indices(lev+1, dsmark[TCA_DSMARK_INDICES]);

    if(dsmark[TCA_DSMARK_DEFAULT_INDEX])
        debug_tca_dsmark_default_index(lev+1, dsmark[TCA_DSMARK_DEFAULT_INDEX]);

    if(dsmark[TCA_DSMARK_SET_TC_INDEX])
        debug_tca_dsmark_set_tc_index(lev+1, dsmark[TCA_DSMARK_SET_TC_INDEX]);

    if(dsmark[TCA_DSMARK_MASK])
        debug_tca_dsmark_mask(lev+1, dsmark[TCA_DSMARK_MASK]);

    if(dsmark[TCA_DSMARK_VALUE])
        debug_tca_dsmark_value(lev+1, dsmark[TCA_DSMARK_VALUE]);
}

/*
 * debug attribute TCA_DSMARK_INDICES
 */
void debug_tca_dsmark_indices(int lev, struct rtattr *dsmark)
{
    if(RTA_PAYLOAD(dsmark) < sizeof(unsigned short)) {
        rec_dbg(lev, "TCA_DSMARK_INDICES(%hu): -- payload too short --",
            RTA_ALIGN(dsmark->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_DSMARK_INDICES(%hu): 0x%04x",
        RTA_ALIGN(dsmark->rta_len), *(unsigned short *)RTA_DATA(dsmark));
}

/*
 * debug attribute TCA_DSMARK_DEFAULT_INDEX
 */
void debug_tca_dsmark_default_index(int lev, struct rtattr *dsmark)
{
    if(RTA_PAYLOAD(dsmark) < sizeof(unsigned short)) {
        rec_dbg(lev, "TCA_DSMARK_DEFAULT_INDEX(%hu): -- payload too short --",
            RTA_ALIGN(dsmark->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_DSMARK_DEFAULT_INDEX(%hu): 0x%04x",
        RTA_ALIGN(dsmark->rta_len), *(unsigned short *)RTA_DATA(dsmark));
}

/*
 * debug attribute TCA_DSMARK_SET_TC_INDEX
 */
void debug_tca_dsmark_set_tc_index(int lev, struct rtattr *dsmark)
{
    rec_dbg(lev, "TCA_DSMARK_SET_TC_INDEX(%hu): none",
        RTA_ALIGN(dsmark->rta_len));
}

/*
 * debug attribute TCA_DSMARK_MASK
 */
void debug_tca_dsmark_mask(int lev, struct rtattr *dsmark)
{
    if(RTA_PAYLOAD(dsmark) < sizeof(unsigned char)) {
        rec_dbg(lev, "TCA_DSMARK_MASK(%hu): -- payload too short --",
            RTA_ALIGN(dsmark->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_DSMARK_MASK(%hu): 0x%02x",
        RTA_ALIGN(dsmark->rta_len), *(unsigned char *)RTA_DATA(dsmark));
}

/*
 * debug attribute TCA_DSMARK_VALUE
 */
void debug_tca_dsmark_value(int lev, struct rtattr *dsmark)
{
    if(RTA_PAYLOAD(dsmark) < sizeof(unsigned char)) {
        rec_dbg(lev, "TCA_DSMARK_VALUE(%hu): -- payload too short --",
            RTA_ALIGN(dsmark->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_DSMARK_VALUE(%hu): 0x%02x",
        RTA_ALIGN(dsmark->rta_len), *(unsigned char *)RTA_DATA(dsmark));
}

/*
 * debug netem options
 */
void debug_tca_options_netem(int lev, struct rtattr *tca)
{
    struct rtattr *netem[__TCA_NETEM_MAX];
    struct tc_netem_qopt *qopt;

    if(RTA_PAYLOAD(tca) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_OPTIONS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    qopt = (struct tc_netem_qopt *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_netem_qopt(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        latency(%d): %u", sizeof(qopt->latency), qopt->latency);
    rec_dbg(lev, "        limit(%d): %u", sizeof(qopt->limit), qopt->limit);
    rec_dbg(lev, "        loss(%d): %u", sizeof(qopt->loss), qopt->loss);
    rec_dbg(lev, "        gap(%d): %u", sizeof(qopt->gap), qopt->gap);
    rec_dbg(lev, "        duplicate(%d): %u", sizeof(qopt->duplicate), qopt->duplicate);
    rec_dbg(lev, "        jitter(%d): %u", sizeof(qopt->jitter), qopt->jitter);

    parse_netem(netem, tca);

    if(netem[TCA_NETEM_CORR])
        debug_tca_netem_corr(lev+1, netem[TCA_NETEM_CORR]);

    if(netem[TCA_NETEM_DELAY_DIST])
        debug_tca_netem_delay_dist(lev+1, netem[TCA_NETEM_DELAY_DIST]);

    if(netem[TCA_NETEM_REORDER])
        debug_tca_netem_reorder(lev+1, netem[TCA_NETEM_REORDER]);

    if(netem[TCA_NETEM_CORRUPT])
        debug_tca_netem_corrupt(lev+1, netem[TCA_NETEM_CORRUPT]);

#if HAVE_DECL_TCA_NETEM_LOSS
    if(netem[TCA_NETEM_LOSS])
        debug_tca_netem_loss(lev+1, netem[TCA_NETEM_LOSS]);
#endif

#if HAVE_DECL_TCA_NETEM_RATE
    if(netem[TCA_NETEM_RATE])
        debug_tca_netem_rate(lev+1, netem[TCA_NETEM_RATE]);
#endif

#if HAVE_DECL_TCA_NETEM_ECN
    if(netem[TCA_NETEM_ECN])
        debug_tca_netem_ecn(lev+1, netem[TCA_NETEM_ECN]);
#endif
}

/*
 * debug attribute TCA_NETEM_CORR
 */
void debug_tca_netem_corr(int lev, struct rtattr *netem)
{
    struct tc_netem_corr *corr;

    if(RTA_PAYLOAD(netem) < sizeof(*corr)) {
        rec_dbg(lev, "TCA_NETEM_CORR(%hu): -- payload too short --",
            RTA_ALIGN(netem->rta_len));
        return;
    }
    corr = (struct tc_netem_corr *)RTA_DATA(netem);

    rec_dbg(lev, "TCA_NETEM_CORR(%hu):", RTA_ALIGN(netem->rta_len));
    rec_dbg(lev, "    [ tc_netem_corr(%d) ]", sizeof(*corr));
    rec_dbg(lev, "        delay_corr(%d): %u", sizeof(corr->delay_corr), corr->delay_corr);
    rec_dbg(lev, "        loss_corr(%d): %u", sizeof(corr->loss_corr), corr->loss_corr);
    rec_dbg(lev, "        dup_corr(%d): %u", sizeof(corr->dup_corr), corr->dup_corr);
}

/*
 * debug attribute TCA_NETEM_DELAY_DIST
 */
void debug_tca_netem_delay_dist(int lev, struct rtattr *netem)
{
    rec_dbg(lev, "TCA_NETEM_DELAY_DIST(%hu): -- ignored --", RTA_ALIGN(netem->rta_len));
}

/*
 * debug attribute TCA_NETEM_REORDER
 */
void debug_tca_netem_reorder(int lev, struct rtattr *netem)
{
    struct tc_netem_reorder *reorder;

    if(RTA_PAYLOAD(netem) < sizeof(*reorder)) {
        rec_dbg(lev, "TCA_NETEM_REORDER(%hu): -- payload too short --",
            RTA_ALIGN(netem->rta_len));
        return;
    }
    reorder = (struct tc_netem_reorder *)RTA_DATA(netem);

    rec_dbg(lev, "TCA_NETEM_REORDER(%hu):", RTA_ALIGN(netem->rta_len));
    rec_dbg(lev, "    [ tc_netem_reorder(%d) ]", sizeof(*reorder));
    rec_dbg(lev, "        probability(%d): %u",
        sizeof(reorder->probability), reorder->probability);
    rec_dbg(lev, "        correlation(%d): %u",
        sizeof(reorder->correlation), reorder->correlation);
}

/*
 * debug attribute TCA_NETEM_CORRUPT
 */
void debug_tca_netem_corrupt(int lev, struct rtattr *netem)
{
    struct tc_netem_corrupt *corrupt;

    if(RTA_PAYLOAD(netem) < sizeof(*corrupt)) {
        rec_dbg(lev, "TCA_NETEM_CORRUPT(%hu): -- payload too short --",
            RTA_ALIGN(netem->rta_len));
        return;
    }

    corrupt = (struct tc_netem_corrupt *)RTA_DATA(netem);

    rec_dbg(lev, "TCA_NETEM_CORRUPT(%hu):", RTA_ALIGN(netem->rta_len));
    rec_dbg(lev, "    [ tc_netem_corrupt(%d) ]", sizeof(*corrupt));
    rec_dbg(lev, "        probability(%d): %u",
        sizeof(corrupt->probability), corrupt->probability);
    rec_dbg(lev, "        correlation(%d): %u",
        sizeof(corrupt->correlation), corrupt->correlation);
}

#if HAVE_DECL_TCA_NETEM_LOSS
/*
 * debug attribute TCA_NETEM_LOSS
 */
void debug_tca_netem_loss(int lev, struct rtattr *netem)
{
    struct rtattr *loss[__NETEM_LOSS_MAX];

    rec_dbg(lev, "TCA_NETEM_LOSS(%hu):", RTA_ALIGN(netem->rta_len));

    parse_netem_loss(loss, netem);

    if(loss[NETEM_LOSS_GI])
        debug_netem_loss_gi(lev+1, loss[NETEM_LOSS_GI]);

    if(loss[NETEM_LOSS_GE])
        debug_netem_loss_ge(lev+1, loss[NETEM_LOSS_GE]);
}

/*
 * debug attribute NETEM_LOSS_GI
 */
void debug_netem_loss_gi(int lev, struct rtattr *loss)
{
    struct tc_netem_gimodel *gi;

    if(RTA_PAYLOAD(loss) < sizeof(*gi)) {
        rec_dbg(lev, "NETEM_LOSS_GI(%hu): -- payload too short --",
            RTA_ALIGN(loss->rta_len));
        return;
    }
    gi = (struct tc_netem_gimodel *)RTA_DATA(loss);

    rec_dbg(lev, "NETEM_LOSS_GI(%hu):", RTA_ALIGN(loss->rta_len));
    rec_dbg(lev, "    [ tc_netem_gimodel(%d) ]", sizeof(*gi));
    rec_dbg(lev, "        p13(%d): %u", sizeof(gi->p13), gi->p13);
    rec_dbg(lev, "        p31(%d): %u", sizeof(gi->p31), gi->p31);
    rec_dbg(lev, "        p32(%d): %u", sizeof(gi->p32), gi->p32);
    rec_dbg(lev, "        p14(%d): %u", sizeof(gi->p14), gi->p14);
    rec_dbg(lev, "        p23(%d): %u", sizeof(gi->p23), gi->p23);
}

/*
 * debug attribute NETEM_LOSS_GE
 */
void debug_netem_loss_ge(int lev, struct rtattr *loss)
{
    struct tc_netem_gemodel *ge;

    if(RTA_PAYLOAD(loss) < sizeof(*ge)) {
        rec_dbg(lev, "NETEM_LOSS_GE(%hu): -- payload too short --",
            RTA_ALIGN(loss->rta_len));
        return;
    }
    ge = (struct tc_netem_gemodel *)RTA_DATA(loss);

    rec_dbg(lev, "NETEM_LOSS_GE(%hu):", RTA_ALIGN(loss->rta_len));
    rec_dbg(lev, "    [ tc_netem_gemodel(%d) ]", sizeof(*ge));
    rec_dbg(lev, "        p(%d): %u", sizeof(ge->p), ge->p);
    rec_dbg(lev, "        r(%d): %u", sizeof(ge->r), ge->r);
    rec_dbg(lev, "        h(%d): %u", sizeof(ge->h), ge->h);
    rec_dbg(lev, "        k1(%d): %u", sizeof(ge->k1), ge->k1);
}
#endif

#if HAVE_DECL_TCA_NETEM_RATE
/*
 * debug attribute TCA_NETEM_RATE
 */
void debug_tca_netem_rate(int lev, struct rtattr *netem)
{
    struct tc_netem_rate *rate;

    if(RTA_PAYLOAD(netem) < sizeof(*netem)) {
        rec_dbg(lev, "TCA_NETEM_RATE(%hu): -- payload too short --",
            RTA_ALIGN(netem->rta_len));
        return;
    }
    rate = (struct tc_netem_rate *)RTA_DATA(netem);

    rec_dbg(lev, "TCA_NETEM_RATE(%hu):", RTA_ALIGN(netem->rta_len));
    rec_dbg(lev, "    [ tc_netem_rate(%d) ]", sizeof(*rate));
    rec_dbg(lev, "        rate(%d): %u", sizeof(rate->rate), rate->rate);
    rec_dbg(lev, "        packet_overhead(%d): %u",
        sizeof(rate->packet_overhead), rate->packet_overhead);
    rec_dbg(lev, "        cell_size(%d): %u", sizeof(rate->cell_size), rate->cell_size);
    rec_dbg(lev, "        cell_overhead(%d): %u",
        sizeof(rate->cell_overhead), rate->cell_overhead);
}
#endif

#if HAVE_DECL_TCA_NETEM_ECN
/*
 * debug attribute TCA_NETEM_ECN
 */
void debug_tca_netem_ecn(int lev, struct rtattr *netem)
{
    if(RTA_PAYLOAD(netem) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_NETEM_ECN(%hu): -- payload too short --",
            RTA_ALIGN(netem->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_NETEM_ECN(%hu): %u",
        RTA_ALIGN(netem->rta_len), *(unsigned *)RTA_DATA(netem));
}
#endif

#if HAVE_DECL_TCA_DRR_UNSPEC
/*
 * debug drr options
 */
void debug_tca_options_drr(int lev, struct rtattr *tca)
{
    struct rtattr *drr[__TCA_DRR_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_drr(drr, tca);

    if(drr[TCA_DRR_QUANTUM])
        debug_tca_drr_quantum(lev+1, drr[TCA_DRR_QUANTUM]);
}

/*
 * debug attribute TCA_DRR_QUANTUM
 */
void debug_tca_drr_quantum(int lev, struct rtattr *drr)
{
    if(RTA_PAYLOAD(drr) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_DRR_QUANTUM(%hu): -- payload too short --",
            RTA_ALIGN(drr->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_DRR_QUANTUM(%hu): %u",
        RTA_ALIGN(drr->rta_len), *(unsigned *)RTA_DATA(drr));
}

/*
 * debug tc_drr_xstats
 */
void debug_tc_drr_xstats(int lev, struct rtattr *tca)
{
    struct tc_drr_stats *stats;

    if(RTA_PAYLOAD(tca) < sizeof(*stats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    stats = (struct tc_drr_stats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_drr_stats(%d) ]", sizeof(*stats));
    rec_dbg(lev, "        deficit(%d): %u", sizeof(stats->deficit), stats->deficit);
}
#endif

#if HAVE_DECL_TCA_SFB_UNSPEC
/*
 * debug sfb options
 */
void debug_tca_options_sfb(int lev, struct rtattr *tca)
{
    struct rtattr *sfb[__TCA_SFB_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_sfb(sfb, tca);

    if(sfb[TCA_SFB_PARMS])
        debug_tca_sfb_parms(lev+1, sfb[TCA_SFB_PARMS]);
}

/*
 * debug attribute TCA_SFB_PARMS
 */
void debug_tca_sfb_parms(int lev, struct rtattr *sfb)
{
    struct tc_sfb_qopt *qopt;

    if(RTA_PAYLOAD(sfb) < sizeof(*qopt)) {
        rec_dbg(lev, "TCA_SFB_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(sfb->rta_len));
        return;
    }
    qopt = (struct tc_sfb_qopt *)RTA_DATA(sfb);

    rec_dbg(lev, "TCA_SFB_PARMS(%hu):", RTA_ALIGN(sfb->rta_len));
    rec_dbg(lev, "    [ tc_stats(%d) ]", sizeof(*qopt));
    rec_dbg(lev, "        rehash_interval(%d): %u",
        sizeof(qopt->rehash_interval), qopt->rehash_interval);
    rec_dbg(lev, "        warmup_time(%d): %u",
        sizeof(qopt->warmup_time), qopt->warmup_time);
    rec_dbg(lev, "        max(%d): %u", sizeof(qopt->max), qopt->max);
    rec_dbg(lev, "        bin_size(%d): %u", sizeof(qopt->bin_size), qopt->bin_size);
    rec_dbg(lev, "        increment(%d): %u", sizeof(qopt->increment), qopt->increment);
    rec_dbg(lev, "        decrement(%d): %u", sizeof(qopt->decrement), qopt->decrement);
    rec_dbg(lev, "        limit(%d): %u", sizeof(qopt->limit), qopt->limit);
    rec_dbg(lev, "        penalty_rate(%d): %u",
        sizeof(qopt->penalty_rate), qopt->penalty_rate);
    rec_dbg(lev, "        penalty_burst(%d): %u",
        sizeof(qopt->penalty_burst), qopt->penalty_burst);
}

/*
 * debug tc_sfb_xstats
 */
void debug_tc_sfb_xstats(int lev, struct rtattr *tca)
{
    struct tc_sfb_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_sfb_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_sfb_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        earlydrop(%d): %u", sizeof(xstats->earlydrop), xstats->earlydrop);
    rec_dbg(lev, "        penaltydrop(%d): %u", sizeof(xstats->penaltydrop), xstats->penaltydrop);
    rec_dbg(lev, "        bucketdrop(%d): %u", sizeof(xstats->bucketdrop), xstats->bucketdrop);
    rec_dbg(lev, "        queuedrop(%d): %u", sizeof(xstats->queuedrop), xstats->queuedrop);
    rec_dbg(lev, "        childdrop(%d): %u", sizeof(xstats->childdrop), xstats->childdrop);
    rec_dbg(lev, "        marked(%d): %u", sizeof(xstats->marked), xstats->marked);
    rec_dbg(lev, "        maxqlen(%d): %u", sizeof(xstats->maxqlen), xstats->maxqlen);
    rec_dbg(lev, "        maxprob(%d): %u", sizeof(xstats->maxprob), xstats->maxprob);
    rec_dbg(lev, "        avgprob(%d): %u", sizeof(xstats->avgprob), xstats->avgprob);
}
#endif

#if HAVE_DECL_TCA_QFQ_UNSPEC
/*
 * debug qfq options
 */
void debug_tca_options_qfq(int lev, struct rtattr *tca)
{
    struct rtattr *qfq[__TCA_QFQ_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_qfq(qfq, tca);

    if(qfq[TCA_QFQ_WEIGHT])
        debug_tca_qfq_weight(lev+1, qfq[TCA_QFQ_WEIGHT]);

    if(qfq[TCA_QFQ_LMAX])
        debug_tca_qfq_lmax(lev+1, qfq[TCA_QFQ_LMAX]);
}

/*
 * debug attribute TCA_QFQ_WEIGHT
 */
void debug_tca_qfq_weight(int lev, struct rtattr *qfq)
{
    if(RTA_PAYLOAD(qfq) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_QFQ_WEIGHT(%hu): -- payload too short --",
            RTA_ALIGN(qfq->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_QFQ_WEIGHT(%hu): %u",
        RTA_ALIGN(qfq->rta_len), *(unsigned *)RTA_DATA(qfq));
}

/*
 * debug attribute TCA_QFQ_LMAX
 */
void debug_tca_qfq_lmax(int lev, struct rtattr *qfq)
{
    if(RTA_PAYLOAD(qfq) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_QFQ_LMAX(%hu): -- payload too short --",
            RTA_ALIGN(qfq->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_QFQ_LMAX(%hu): %u",
        RTA_ALIGN(qfq->rta_len), *(unsigned *)RTA_DATA(qfq));
}

/*
 * debug tc_qfq_xstats
 */
void debug_tc_qfq_xstats(int lev, struct rtattr *tca)
{
    struct tc_qfq_stats *stats;

    if(RTA_PAYLOAD(tca) < sizeof(*stats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    stats = (struct tc_qfq_stats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_qfq_stats(%d) ]", sizeof(*stats));
    rec_dbg(lev, "        weight(%d): %u", sizeof(stats->weight), stats->weight);
    rec_dbg(lev, "        lmax(%d): %u", sizeof(stats->lmax), stats->lmax);
}
#endif

#if HAVE_DECL_TCA_CODEL_UNSPEC
/*
 * debug codel options
 */
void debug_tca_options_codel(int lev, struct rtattr *tca)
{
    struct rtattr *codel[__TCA_CODEL_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_codel(codel, tca);

    if(codel[TCA_CODEL_TARGET])
        debug_tca_codel_target(lev+1, codel[TCA_CODEL_TARGET]);

    if(codel[TCA_CODEL_LIMIT])
        debug_tca_codel_limit(lev+1, codel[TCA_CODEL_LIMIT]);

    if(codel[TCA_CODEL_INTERVAL])
        debug_tca_codel_interval(lev+1, codel[TCA_CODEL_INTERVAL]);

    if(codel[TCA_CODEL_ECN])
        debug_tca_codel_ecn(lev+1, codel[TCA_CODEL_ECN]);
}

/*
 * debug attribute TCA_CODEL_TARGET
 */
void debug_tca_codel_target(int lev, struct rtattr *codel)
{
    if(RTA_PAYLOAD(codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_CODEL_TARGET(%hu): -- payload too short --",
            RTA_ALIGN(codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_CODEL_TARGET(%hu): %u",
        RTA_ALIGN(codel->rta_len), *(unsigned *)RTA_DATA(codel));
}

/*
 * debug attribute TCA_CODEL_LIMIT
 */
void debug_tca_codel_limit(int lev, struct rtattr *codel)
{
    if(RTA_PAYLOAD(codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_CODEL_LIMIT(%hu): -- payload too short --",
            RTA_ALIGN(codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_CODEL_LIMIT(%hu): %u",
        RTA_ALIGN(codel->rta_len), *(unsigned *)RTA_DATA(codel));
}

/*
 * debug attribute TCA_CODEL_INTERVAL
 */
void debug_tca_codel_interval(int lev, struct rtattr *codel)
{
    if(RTA_PAYLOAD(codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_CODEL_INTERVAL(%hu): -- payload too short --",
            RTA_ALIGN(codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_CODEL_INTERVAL(%hu): %u",
        RTA_ALIGN(codel->rta_len), *(unsigned *)RTA_DATA(codel));
}

/*
 * debug attribute TCA_CODEL_ECN
 */
void debug_tca_codel_ecn(int lev, struct rtattr *codel)
{
    if(RTA_PAYLOAD(codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_CODEL_ECN(%hu): -- payload too short --",
            RTA_ALIGN(codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_CODEL_ECN(%hu): %u",
        RTA_ALIGN(codel->rta_len), *(unsigned *)RTA_DATA(codel));
}

/*
 * debug tc_codel_xstats
 */
void debug_tc_codel_xstats(int lev, struct rtattr *tca)
{
    struct tc_codel_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_codel_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_codel_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        maxpacket(%d): %u", sizeof(xstats->maxpacket), xstats->maxpacket);
    rec_dbg(lev, "        count(%d): %u", sizeof(xstats->count), xstats->count);
    rec_dbg(lev, "        lastcount(%d): %u", sizeof(xstats->lastcount), xstats->lastcount);
    rec_dbg(lev, "        ldelay(%d): %u", sizeof(xstats->ldelay), xstats->ldelay);
    rec_dbg(lev, "        drop_next(%d): %d", sizeof(xstats->drop_next), xstats->drop_next);
    rec_dbg(lev, "        drop_overlimit(%d): %u",
        sizeof(xstats->drop_overlimit), xstats->drop_overlimit);
    rec_dbg(lev, "        ecn_mark(%d): %u", sizeof(xstats->ecn_mark), xstats->ecn_mark);
    rec_dbg(lev, "        dropping(%d): %u", sizeof(xstats->dropping), xstats->dropping);
}
#endif

#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
/*
 * debug fq_codel options
 */
void debug_tca_options_fq_codel(int lev, struct rtattr *tca)
{
    struct rtattr *fq_codel[__TCA_FQ_CODEL_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_fq_codel(fq_codel, tca);

    if(fq_codel[TCA_FQ_CODEL_TARGET])
        debug_tca_fq_codel_target(lev+1, fq_codel[TCA_FQ_CODEL_TARGET]);

    if(fq_codel[TCA_FQ_CODEL_LIMIT])
        debug_tca_fq_codel_limit(lev+1, fq_codel[TCA_FQ_CODEL_LIMIT]);

    if(fq_codel[TCA_FQ_CODEL_INTERVAL])
        debug_tca_fq_codel_interval(lev+1, fq_codel[TCA_FQ_CODEL_INTERVAL]);

    if(fq_codel[TCA_FQ_CODEL_ECN])
        debug_tca_fq_codel_interval(lev+1, fq_codel[TCA_FQ_CODEL_ECN]);

    if(fq_codel[TCA_FQ_CODEL_FLOWS])
        debug_tca_fq_codel_flows(lev+1, fq_codel[TCA_FQ_CODEL_FLOWS]);

    if(fq_codel[TCA_FQ_CODEL_QUANTUM])
        debug_tca_fq_codel_quantum(lev+1, fq_codel[TCA_FQ_CODEL_QUANTUM]);
}

/*
 * debug attribute TCA_FQ_CODEL_TARGET
 */
void debug_tca_fq_codel_target(int lev, struct rtattr *fq_codel)
{
    if(RTA_PAYLOAD(fq_codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FQ_CODEL_TARGET(%hu): -- payload too short --",
            RTA_ALIGN(fq_codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FQ_CODEL_TARGET(%hu): %u",
        RTA_ALIGN(fq_codel->rta_len), *(unsigned *)RTA_DATA(fq_codel));
}

/*
 * debug attribute TCA_FQ_CODEL_LIMIT
 */
void debug_tca_fq_codel_limit(int lev, struct rtattr *fq_codel)
{
    if(RTA_PAYLOAD(fq_codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FQ_CODEL_LIMIT(%hu): -- payload too short --",
            RTA_ALIGN(fq_codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FQ_CODEL_LIMIT(%hu): %u",
        RTA_ALIGN(fq_codel->rta_len), *(unsigned *)RTA_DATA(fq_codel));
}

/*
 * debug attribute TCA_FQ_CODEL_INTERVAL
 */
void debug_tca_fq_codel_interval(int lev, struct rtattr *fq_codel)
{
    if(RTA_PAYLOAD(fq_codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FQ_CODEL_INTERVAL(%hu): -- payload too short --",
            RTA_ALIGN(fq_codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FQ_CODEL_INTERVAL(%hu): %u",
        RTA_ALIGN(fq_codel->rta_len), *(unsigned *)RTA_DATA(fq_codel));
}

/*
 * debug attribute TCA_FQ_CODEL_ECN
 */
void debug_tca_fq_codel_ecn(int lev, struct rtattr *fq_codel)
{
    if(RTA_PAYLOAD(fq_codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FQ_CODEL_ECN(%hu): -- payload too short --",
            RTA_ALIGN(fq_codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FQ_CODEL_ECN(%hu): %u",
        RTA_ALIGN(fq_codel->rta_len), *(unsigned *)RTA_DATA(fq_codel));
}

/*
 * debug attribute TCA_FQ_CODEL_FLOWS
 */
void debug_tca_fq_codel_flows(int lev, struct rtattr *fq_codel)
{
    if(RTA_PAYLOAD(fq_codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FQ_CODEL_FLOWS(%hu): -- payload too short --",
            RTA_ALIGN(fq_codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FQ_CODEL_FLOWS(%hu): %u",
        RTA_ALIGN(fq_codel->rta_len), *(unsigned *)RTA_DATA(fq_codel));
}

/*
 * debug attribute TCA_FQ_CODEL_QUANTUM
 */
void debug_tca_fq_codel_quantum(int lev, struct rtattr *fq_codel)
{
    if(RTA_PAYLOAD(fq_codel) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FQ_CODEL_QUANTUM(%hu): -- payload too short --",
            RTA_ALIGN(fq_codel->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FQ_CODEL_QUANTUM(%hu): %u",
        RTA_ALIGN(fq_codel->rta_len), *(unsigned *)RTA_DATA(fq_codel));
}

/*
 * debug tc_fq_codel_xstats
 */
void debug_tc_fq_codel_xstats(int lev, struct rtattr *tca)
{
    struct tc_fq_codel_xstats *xstats;

    if(RTA_PAYLOAD(tca) < sizeof(*xstats)) {
        rec_dbg(lev, "TCA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(tca->rta_len));
        return;
    }
    xstats = (struct tc_fq_codel_xstats *)RTA_DATA(tca);

    rec_dbg(lev, "TCA_XSTATS(%hu):", RTA_ALIGN(tca->rta_len));
    rec_dbg(lev, "    [ tc_fq_codel_xstats(%d) ]", sizeof(*xstats));
    rec_dbg(lev, "        type(%d): %u", sizeof(xstats->type), xstats->type);

    if(xstats->type == TCA_FQ_CODEL_XSTATS_QDISC) {
        struct tc_fq_codel_qd_stats *qd_stats = &(xstats->qdisc_stats);

        rec_dbg(lev, "        [ tc_fq_codel_qd_stats qdisc_stats(%d) ]",
            sizeof(*qd_stats));
        rec_dbg(lev, "            maxpacket(%d): %u",
            sizeof(qd_stats->maxpacket), qd_stats->maxpacket);
        rec_dbg(lev, "            drop_overlimit(%d): %u",
            sizeof(qd_stats->drop_overlimit), qd_stats->drop_overlimit);
        rec_dbg(lev, "            ecn_mark(%d): %u",
            sizeof(qd_stats->ecn_mark), qd_stats->ecn_mark);
        rec_dbg(lev, "            new_flow_count(%d): %u",
            sizeof(qd_stats->new_flow_count), qd_stats->new_flow_count);
        rec_dbg(lev, "            new_flows_len(%d): %u",
            sizeof(qd_stats->new_flows_len), qd_stats->new_flows_len);
        rec_dbg(lev, "            old_flows_len(%d): %u",
            sizeof(qd_stats->old_flows_len), qd_stats->old_flows_len);
    } else if(xstats->type == TCA_FQ_CODEL_XSTATS_QDISC) {
        struct tc_fq_codel_cl_stats *cl_stats = &(xstats->class_stats);

        rec_dbg(lev, "        [ tc_fq_codel_cl_stats class_stats(%d) ]",
            sizeof(*cl_stats));
        rec_dbg(lev, "            deficit(%d): %d",
            sizeof(cl_stats->deficit), cl_stats->deficit);
        rec_dbg(lev, "            ldelay(%d): %u",
            sizeof(cl_stats->ldelay), cl_stats->ldelay);
        rec_dbg(lev, "            count(%d): %u",
            sizeof(cl_stats->count), cl_stats->count);
        rec_dbg(lev, "            lastcount(%d): %u",
            sizeof(cl_stats->lastcount), cl_stats->lastcount);
        rec_dbg(lev, "            dropping(%d): %u",
            sizeof(cl_stats->dropping), cl_stats->dropping);
        rec_dbg(lev, "            drop_next(%d): %d",
            sizeof(cl_stats->drop_next), cl_stats->drop_next);
    }
}
#endif

/*
 * debug struct tc_ratespec
 */
void debug_tc_ratespec(int lev, struct tc_ratespec *rate, char *name)
{
    rec_dbg(lev, "[ tc_ratespec %s(%d) ]", name, sizeof(*rate));
    rec_dbg(lev, "    cell_log(%d): %d", sizeof(rate->cell_log), rate->cell_log);
#ifdef HAVE_STRUCT_TC_RATESPEC___RESERVED
    rec_dbg(lev, "    __reserved(%d): %d", sizeof(rate->__reserved), rate->__reserved);
#endif
#ifdef HAVE_STRUCT_TC_RATESPEC_LINKLAYER
    rec_dbg(lev, "    linklayer(%d): %d", sizeof(rate->linklayer), rate->linklayer);
#endif
#ifdef HAVE_STRUCT_TC_RATESPEC_FEATURE
    rec_dbg(lev, "    feature(%d): %hu", sizeof(rate->feature), rate->feature);
#endif
#ifdef HAVE_STRUCT_TC_RATESPEC_OVERHEAD
    rec_dbg(lev, "    overhead(%d): %hu", sizeof(rate->overhead), rate->overhead);
#endif
#ifdef HAVE_STRUCT_TC_RATESPEC_ADDEND
    rec_dbg(lev, "    addend(%d): %hd", sizeof(rate->addend), rate->addend);
#endif
#ifdef HAVE_STRUCT_TC_RATESPEC_CELL_ALIGN
    rec_dbg(lev, "    cell_align(%d): %hd", sizeof(rate->cell_align), rate->cell_align);
#endif
    rec_dbg(lev, "    mpu(%d): %hu", sizeof(rate->mpu), rate->mpu);
    rec_dbg(lev, "    rate(%d): %u", sizeof(rate->rate), rate->rate);
}

#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
/*
 *  convert TCQ_PLUG flags from number to string
 */
const char *convert_tcq_plug_action(int action)
{
#define _TCQ_PLUG_ACTION(s) \
    if(action == TCQ_PLUG_##s) \
        return(#s);
    _TCQ_PLUG_ACTION(BUFFER);
    _TCQ_PLUG_ACTION(RELEASE_ONE);
    _TCQ_PLUG_ACTION(RELEASE_INDEFINITE);
#undef _TCQ_PLUG_ACTION
    return("UNKNOWN");
}
#endif

/*
 * convert red flags from number to string
 */
void convert_tc_red_flags(int flags, char *flags_list, int len, int debug)
{
    if(!flags) {
        strncpy(flags_list, debug ? "NONE" : "none", len);
        return;
    }
#define _TC_RED_FLAGS(s1, s2) \
    if((flags & TC_RED_##s1) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~TC_RED_##s1) ? \
            strncat(flags_list, debug ? #s1 "," : #s2 ",", \
                len - strlen(flags_list) - 1) : \
            strncat(flags_list, debug ? #s1 : #s2, \
                len - strlen(flags_list) - 1);
    _TC_RED_FLAGS(ECN, ecn)
    _TC_RED_FLAGS(HARDDROP, harddrop)
#ifdef TC_RED_ADAPTATIVE
    _TC_RED_FLAGS(ADAPTATIVE, adaptative)
#endif
#undef _TC_RED_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, debug ? "UNKNOWN" : "unknown", len);
}

/*
 *  convert TCF_CBQ_LSS change from number to string
 */
void convert_tcf_cbq_lss_change(int change, char *change_list, int len)
{
    if(!change) {
        strncpy(change_list, "NONE", len);
        return;
    }
#define _TCF_CBQ_LSS_LEVEL(s) \
    if((change & TCF_CBQ_LSS_##s) && (len - strlen(change_list) - 1 > 0)) \
        (change &= ~TCF_CBQ_LSS_##s) ? \
            strncat(change_list, #s ",", len - strlen(change_list) -1) : \
            strncat(change_list, #s, len - strlen(change_list) - 1);
    _TCF_CBQ_LSS_LEVEL(FLAGS);
    _TCF_CBQ_LSS_LEVEL(EWMA);
    _TCF_CBQ_LSS_LEVEL(MAXIDLE);
    _TCF_CBQ_LSS_LEVEL(MINIDLE);
    _TCF_CBQ_LSS_LEVEL(OFFTIME);
    _TCF_CBQ_LSS_LEVEL(AVPKT);
#undef _TCF_CBQ_LSS_LEVEL
    if(!strlen(change_list))
        strncpy(change_list, "UNKNOWN", len);
}

/*
 *  convert TCF_CBQ_LSS flags from number to string
 */
void convert_tcf_cbq_lss_flags(int flags, char *flags_list, int len)
{
    if(!flags) {
        strncpy(flags_list, "NONE", len);
        return;
    }
#define _TCF_CBQ_LSS_FLAGS(s) \
    if((flags & TCF_CBQ_LSS_##s) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~TCF_CBQ_LSS_##s) ? \
            strncat(flags_list, #s ",", len - strlen(flags_list) - 1) : \
            strncat(flags_list, #s, len - strlen(flags_list) - 1);
    _TCF_CBQ_LSS_FLAGS(BOUNDED);
    _TCF_CBQ_LSS_FLAGS(ISOLATED);
#undef _TCF_CBQ_LSS_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, "UNKNOWN", len);
}

/*
 *  convert TCF_CBQ_OVL strategy from number to string
 */
void convert_tc_cbq_ovl_strategy(int strategy, char *strategy_list, int len)
{
    if(!strategy) {
        strncpy(strategy_list, "NONE", len);
        return;
    }
#define _TC_CBQ_OVL_STRATEGY(s) \
    if((strategy & TC_CBQ_OVL_##s) && (len - strlen(strategy_list) - 1 > 0)) \
        (strategy &= ~TC_CBQ_OVL_##s) ? \
            strncat(strategy_list, #s ",", len - strlen(strategy_list) - 1) : \
            strncat(strategy_list, #s, len - strlen(strategy_list) - 1);
    _TC_CBQ_OVL_STRATEGY(CLASSIC);
    _TC_CBQ_OVL_STRATEGY(DELAY);
    _TC_CBQ_OVL_STRATEGY(LOWPRIO);
    _TC_CBQ_OVL_STRATEGY(DROP);
    _TC_CBQ_OVL_STRATEGY(RCLASSIC);
#undef _TC_CBQ_OVL_STRATEGY
    if(!strlen(strategy_list))
        strncpy(strategy_list, "UNKNOWN", len);
}
