/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
  * You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

var MODULE_NAME = "mock-object-helpers";

var Cm = Components.manager;

function installInto(module) {
  module.MockObjectReplacer = MockObjectReplacer;
  module.MockObjectRegisterer = MockObjectRegisterer;
}

function MockObjectRegisterer(aContractID, aCID, aComponent) {
  this._contractID = aContractID;
  this._cid = Components.ID("{" + aCID + "}");
  this._component = aComponent;
}

MockObjectRegisterer.prototype = {
  register: function MOR_register() {
    let providedConstructor = this._component;
    this._mockFactory = {
      createInstance: function MF_createInstance(aOuter, aIid) {
        if (aOuter != null)
          throw Cr.NS_ERROR_NO_AGGREGATION;
        return new providedConstructor().QueryInterface(aIid);
      }
    };

   let componentRegistrar = Cm.QueryInterface(Ci.nsIComponentRegistrar);

   componentRegistrar.registerFactory(this._cid,
        "",
        this._contractID,
        this._mockFactory);
  },

  unregister: function MOR_unregister() {
    let componentRegistrar = Cm.QueryInterface(Ci.nsIComponentRegistrar);

    componentRegistrar.unregisterFactory(this._cid,
        this._mockFactory);
  },
}

/**
 * Allows registering a mock XPCOM component, that temporarily replaces the
 *  original one when an object implementing a given ContractID is requested
 *  using createInstance.
 *
 * @param aContractID
 *        The ContractID of the component to replace, for example
 *        "@mozilla.org/filepicker;1".
 *
 * @param aReplacementCtor
 *        The constructor function for the JavaScript object that will be
 *        created every time createInstance is called. This object must
 *        implement QueryInterface and provide the XPCOM interfaces required by
 *        the specified ContractID (for example
 *        Ci.nsIFilePicker).
 */

function MockObjectReplacer(aContractID, aReplacementCtor) {
  this._contractID = aContractID;
  this._replacementCtor = aReplacementCtor;
}

MockObjectReplacer.prototype = {
  /**
   * Replaces the current factory with one that returns a new mock object.
   *
   * After register() has been called, it is mandatory to call unregister() to
   * restore the original component. Usually, you should use a try-catch block
   * to ensure that unregister() is called.
   */
  register: function MORe_register() {
    if (this._originalFactory)
      throw Error("Invalid object state when calling register()");

    // Define a factory that creates a new object using the given constructor.
    var providedConstructor = this._replacementCtor;
    this._mockFactory = {
      createInstance: function MF_createInstance(aOuter, aIid) {
        if (aOuter != null)
          throw Cr.NS_ERROR_NO_AGGREGATION;
        return new providedConstructor().QueryInterface(aIid);
      }
    };

    var retVal = swapFactoryRegistration(this._cid, this._contractID, this._mockFactory, this._originalFactory);
    if ('error' in retVal) {
      throw new Exception("ERROR: " + retVal.error);
    } else {
      this._cid = retVal.cid;
      this._originalFactory = retVal.originalFactory;
    }
  },

  /**
   * Restores the original factory.
   */
  unregister: function MORe_unregister() {
    if (!this._originalFactory)
      throw Error("Invalid object state when calling unregister()");

    // Free references to the mock factory.
    swapFactoryRegistration(this._cid, this._contractID, this._mockFactory, this._originalFactory);

    // Allow registering a mock factory again later.
    this._cid = null;
    this._originalFactory = null;
    this._mockFactory = null;
  },

  // --- Private methods and properties ---

  /**
   * The factory of the component being replaced.
   */
  _originalFactory: null,

  /**
   * The CID under which the mock contractID was registered.
   */
  _cid: null,

  /**
   * The nsIFactory that was automatically generated by this object.
   */
  _mockFactory: null
}

/**
 * Swiped from mozilla/testing/mochitest/tests/SimpleTest/specialpowersAPI.js
 */
function swapFactoryRegistration(cid, contractID, newFactory, oldFactory) {
  let componentRegistrar = Cm.QueryInterface(Ci.nsIComponentRegistrar);

  var unregisterFactory = newFactory;
  var registerFactory = oldFactory;

  if (cid == null) {
    if (contractID != null) {
      cid = componentRegistrar.contractIDToCID(contractID);
      oldFactory = Cm.getClassObject(Cc[contractID], Ci.nsIFactory);
    } else {
      return {'error': "trying to register a new contract ID: Missing contractID"};
    }

    unregisterFactory = oldFactory;
    registerFactory = newFactory;
  }

  componentRegistrar.unregisterFactory(cid,
      unregisterFactory);

  // Restore the original factory.
  componentRegistrar.registerFactory(cid,
      "",
      contractID,
      registerFactory);
  return {'cid':cid, 'originalFactory':oldFactory};
}

