/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// @flow

import type {
  SourcePacket,
  ResumedPacket,
  PausedPacket,
  ThreadClient,
  Actions
} from "./types";

import { createPause, createSource } from "./create";
import sourceQueue from "../../utils/source-queue";

const CALL_STACK_PAGE_SIZE = 1000;

type Dependencies = {
  threadClient: ThreadClient,
  actions: Actions,
  supportsWasm: boolean
};

let actions: Actions;
let supportsWasm: boolean;
let isInterrupted: boolean;

function addThreadEventListeners(client: ThreadClient) {
  Object.keys(clientEvents).forEach(eventName => {
    client.addListener(eventName, clientEvents[eventName].bind(null, client));
  });
}

function setupEvents(dependencies: Dependencies) {
  const threadClient = dependencies.threadClient;
  actions = dependencies.actions;
  supportsWasm = dependencies.supportsWasm;
  sourceQueue.initialize(actions);

  if (threadClient) {
    addThreadEventListeners(threadClient);

    if (threadClient._parent) {
      // Parent may be BrowsingContextTargetFront/WorkerTargetFront and
      // be protocol.js.  Or DebuggerClient and still be old fashion actor.
      if (threadClient._parent.on) {
        threadClient._parent.on("workerListChanged", workerListChanged);
      } else {
        threadClient._parent.addListener(
          "workerListChanged",
          workerListChanged
        );
      }
    }
  }
}

async function paused(
  threadClient: ThreadClient,
  _: "paused",
  packet: PausedPacket
) {
  // If paused by an explicit interrupt, which are generated by the
  // slow script dialog and internal events such as setting
  // breakpoints, ignore the event.
  const { why } = packet;
  if (why.type === "interrupted" && !packet.why.onNext) {
    isInterrupted = true;
    return;
  }

  let response;
  try {
    // Eagerly fetch the frames
    response = await threadClient.getFrames(0, CALL_STACK_PAGE_SIZE);
  } catch (e) {
    console.log(e);
    return;
  }

  // NOTE: this happens if we fetch frames and then immediately navigate
  if (!response.hasOwnProperty("frames")) {
    return;
  }

  if (why.type != "alreadyPaused") {
    const pause = createPause(threadClient.actor, packet, response);
    await sourceQueue.flush();
    actions.paused(pause);
  }
}

function resumed(
  threadClient: ThreadClient,
  _: "resumed",
  packet: ResumedPacket
) {
  // NOTE: the client suppresses resumed events while interrupted
  // to prevent unintentional behavior.
  // see [client docs](../README.md#interrupted) for more information.
  if (isInterrupted) {
    isInterrupted = false;
    return;
  }

  actions.resumed(packet);
}

function newSource(
  threadClient: ThreadClient,
  _: "newSource",
  { source }: SourcePacket
) {
  sourceQueue.queue(createSource(threadClient.actor, source, { supportsWasm }));
}

function workerListChanged() {
  actions.updateWorkers();
}

const clientEvents = {
  paused,
  resumed,
  newSource
};

export { setupEvents, clientEvents, addThreadEventListeners };
