/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_IMAGE_IMAGERESOURCE_HPP
#define INDII_IMAGE_IMAGERESOURCE_HPP

#include "ImageCache.hpp"
#include "Image.hpp"

namespace indii {
/**
 * Abstract image resource manager.
 */
class ImageResource {
public:
  /**
   * Constructor.
   */
  ImageResource();

  /**
   * Destructor.
   */
  virtual ~ImageResource();
  
  /**
   * Fit scale within upper bounds.
   *
   * @param widthBound Width bound.
   * @param heightBound Height bound.
   *
   * @return Scale.
   *
   * Calculates the largest scale of the image such that its width does not
   * exceed @p widthBound and its height does not exceed @p heightBound,
   * while maintaining the aspect ratio.
   */
  int fitScale(const int widthBound, const int heightBound);

  /**
   * Get width of the image.
   *
   * @return Width of image.
   */
  int getWidth() const;
  
  /**
   * Get height of the image.
   *
   * @return Height of the image.
   */
  int getHeight() const;
  
  /**
   * Get image.
   *
   * @param scale
   *
   * @return A version of the image of the specified size.
   */
  Image* get(const float scale = 1.0f);

  /**
   * Get low quality image.
   *
   * @param scale
   *
   * @return A version of the image of the specified size.
   */
  Image* getLow(const float scale = 1.0f);

  /**
   * Get image for display.
   *
   * @param width Width of image in pixels.
   */
  wxImage* getDisplay(const int width);

  /**
   * Original image.
   */
  Image* original;
  
  /**
   * Resized image cache.
   */
  ImageCache* cache;
};
}

inline int indii::ImageResource::fitScale(const int widthBound,
    const int heightBound) {
  /* pre-condition */
  assert(widthBound >= 0);
  assert(heightBound >= 0);
  
  int widthScale = (getWidth() + widthBound - 1) / widthBound;
  int heightScale = (getHeight() + heightBound - 1) / heightBound;
  
  return std::max(widthScale, heightScale);
}

inline int indii::ImageResource::getWidth() const {
  return original->getWidth();
}

inline int indii::ImageResource::getHeight() const {
  return original->getHeight();
}

inline indii::Image* indii::ImageResource::get(const float scale) {
  return cache->get(scale);
}

inline indii::Image* indii::ImageResource::getLow(const float scale) {
  return cache->getLow(scale);
}

inline wxImage* indii::ImageResource::getDisplay(const int width) {
  return cache->getDisplay(width);
}

#endif
