#! /bin/sh
#
# $Id: nwclust.pl.AutoStart,v 1.1.2.5 2011/05/16 17:26:48 laim Exp $ Copyright (c) 2002-2011 EMC Corporation
#

#
# Copyright (c) 2002-2011 EMC Corporation.
#
# All rights reserved.  This is an UNPUBLISHED work, and
# comprises proprietary and confidential information of EMC.
# Unauthorized use, disclosure, and distribution are strictly
# prohibited.  Use, duplication, or disclosure of the software
# and documentation by the U.S. Government is subject to
# restrictions set forth in a license agreement between the
# Government and EMC or other written agreement specifying
# the Government's rights to use the software and any applicable
# FAR provisions, such as FAR 52.227-19.
#

# EMC AutoStart/NetWorker interface module.
#
#
# For:         Networker
# Platform:    Solaris, NT, AIX, HP, Linux
# LC Version:  5.0+
#
#  This script will provide domain information to STDOUT
#  for Networker backups.  The output will be in RAP format
#  like:
#
#	type: NSR_CLU_TYPE;
#	clu_type: NSR_LC_TYPE;
#	interface version: 1.0;
#
#	type: NSR_CLU_VIRTHOST;
#	hostname: <host1>;
#	owned paths: <path1>, <path2>;
#
#	type: NSR_CLU_VIRTHOST;
#	hostname: <host2>;
#	owned paths: <path3>, <path4>
#
#	.
#	.
#	.
#
#
#
#
# ************************************************************************
# Audit Trail
#
# Version	Date		Description
# -------	----		-----------
# V01.000	12-14-99	Original
# V01.001	02-18-00	Added $OnlineState var to detect RGs never run
# V01.002	05-28-00	Changed parse for data sources so that an IP
#				address does not need to be defined in the startup
#				sequence before a data source.
# V01.003	03-14-01	Added support for spaces in path names as well as
#				lack of "." in the path on UNIX variants.
#               10-01-02        Added code to cache and reuse domain information
#                               instead of connecting to agent every single time.
#                               The following environment variables control this
#                               behavior:
#                               - NW_LCMAP_STALE_THRESHOLD: seconds before cache
#                                                   is considered stale
#                                                   Default: 600
#                               - NW_LCMAP_LOCK_TIMEOUT: seconds to wait before
#                                                   printing out the stale info
#                                                   Default: 120
# V01.004	07-22-10	Added support for multiple IPs in one resource group.
# ************************************************************************
#
#
#   DEPENDENCIES
#	1.  FT_DIR and FT_DOMAIN must be set in order to run this script.
#	2.  The user id running the proc must be a valid user in the Legato
#		AAM domain.
#	3.  This script must also be able to write out the definition
#		file to the Current Working Directory.
#	4.  This script must be able to run the hostname command.
#
#  NOTE:  This script will determine owned file systems by linking the
#  IP Address in a Resource Group with ALL data sources that are not
#  network mounted in the Resource Group (dataSourceTypes NT_SHARED_DISK
#  and UX_File_System).  If a Resource Group contains multiple IPs,
#  all data of this group will be backed up using the IP corresponding to
#  NW Client resource defined in the NW Server.  Also, raw devices are not
#  currently supported in this script.
#
#  IPOverride:  specifying the RG attribute IPOverride enables the user to specify
#  the IP to mount point mapping on a per Resource Group basis.
#  IPOverride Attribute value format:
#	Attribute Name: IPOverride
#	Atrtribute Value: <IP_ADDRESS>=<path1>[,<path2>[...]]
#  EXAMPLE:
#    Attribute Name: IPOverride
#    Attribute Value: 192.168.2.201=/mount1,/mount2
#
#
#  Enhancements:
#  Command line args:
#     perl nwclust.pl alt_host <HOST_NAME> specifies an alternate host
#     perl nwclust.pl -version returns the version string.
#  Environment variables:
#     FT_NO_NFS will cause nfs data sources to NOT be returned in the rap
#               output.
#



package LC_NW_INTERFACE;
$aam_version = "5.0";
$version = "1.0";
if ($ARGV[0] =~ m/-version/i)
{
  printf("nwclust.pl version %s\n", $aam_version);
  exit 0;
}


#&initpwd;

#$cwd = $ENV{PWD};

# PARAMETERS
$TRUE = 1;
$FALSE = 0;
# NEVER set debug on in a working NWA environment. Debug msgs
# are sent to stdout. This will cause the nsrindexd process that
# calls this script (via lcmap) to be throroughly confused and
# result in NetWorker not functioning correctly
$Debug = $FALSE;

# First, take the defaults
# Changed the path of the variable cache file and deffile
#	to use the variable "TEMP" in windows, else use the path /tmp for other
#	operating systems. However, provision for user defined path would still
#	be made available by using - NW_LCMAP_CACHEFILE & NW_LCMAP_DEFFILE
#$CacheFile = "$ENV{FT_DIR}/log/agent/lcmap_cache";# must be absolute path
if ($ENV{OS} eq "Windows_NT")
{
  $CacheFile = "$ENV{TEMP}/lcmap_cache";
  $defFile = "$ENV{TEMP}/deffile.$$";
  $LockFile = "$ENV{TEMP}/lcmap_lock";
}
else
{
  $CacheFile = "/tmp/lcmap_cache";
  $defFile = "/tmp/deffile.$$";
  $LockFile = "/tmp/lcmap_lock";
}

$LockTimeout = 120;		#seconds
$CacheStaleThreshold = 6;	#seconds

#Let anything in the environment override these
if ($ENV{NW_LCMAP_STALE_THRESHOLD}) {
  $CacheStaleThreshold = $ENV{NW_LCMAP_STALE_THRESHOLD};
}

if ($ENV{NW_LCMAP_LOCK_TIMEOUT}) {
  $LockTimeout = $ENV{NW_LCMAP_LOCK_TIMEOUT};
}

if ($ENV{NW_LCMAP_LOCKFILE}) {
  $LockFile = $ENV{NW_LCMAP_LOCKFILE};
}

if ($ENV{NW_LCMAP_CACHEFILE}) {
  $CacheFile = $ENV{NW_LCMAP_CACHEFILE};
}

if ($ENV{NW_LCMAP_DEFFILE}) {
  $defFile = $ENV{NW_LCMAP_DEFFILE};
}

#  Set all the useful vars:
$FT_DIR = $ENV{FT_DIR};
$FT_DOMAIN = $ENV{FT_DOMAIN};
# If we can't contact the domain, then this becomes "";
$NSR_LC_TYPE = "NSR_LC_TYPE";

if ($ENV{OS} eq "Windows_NT")
{
  $ftcli = "ftcli.exe";
  # Update the path so that we can always find hostname.exe
  $ENV{PATH} .= ";$ENV{windir}\\system32";
  $MV = "";
  $GREP = "find";
}
else
{
  $ftcli = "./ftcli";
  # Update the path so that we can always find hostname
  $ENV{PATH} .= ":/usr/bin:/usr/sbin:/etc";
  $MV = "mv";
  $GREP = "grep";
}

$exportCommand = "$ftcli -d $FT_DOMAIN -c \"export $defFile\"";

$hostName = `hostname`;
chop $hostName;
($hostName, @rest) = split(/\./, $hostName);  #Chop the domain name off

if ($ARGV[0] eq "alt_host")
{
  $hostName=$ARGV[1];
}
$hostName =~ tr/A-Z/a-z/;

# $doNFS determines if this script returns NFS mounted data sources.
# If this behavior
# is not desired, set $doNFS=0.
$doNFS = 1;
if (defined($ENV{FT_NO_NFS}))
{
  $doNFS = 0;
}


# Global Utility Variables:

# For parsing data sources:
$OSType = "";
$dataSourceName = "";

# For parsing the RGs:
$rgName = "";
@IPAddress = ();
$IPcount = 0;
$DataSourceString = "";

#  Assume the RG is Offline until we find out  differently
$OnlineState = "OFFLINE";

&debug("Determining if cache is stale");
# If the cache isn't stale, then we don't need to create a
# ftcli process to query the AAM domain for cluster information
if (-f $CacheFile && !-z $CacheFile && ! &isCacheStale())  {
  &debug("Printing out cache");
  &printCache();
  exit(0);
}

&debug("Cache is stale, grabbing lock before attempting update");
# Grab the lock before updating the cache.
&grabLock();
&debug("Grabbed Lock");

#  All of the data required for the final output can be stored in
#  2 associative arrays:  %IP_Data_Map where the key is IP Address and
#  the value is a space separated list of UNIX mount point or NT
#  drive letters.  %DataSourcePaths will be created during the reading
#  of the export def file with
#  the key being the data source name and the value being the single
#  mount point or drive letter of the data source.

chdir("$FT_DIR/bin");
#  All the data we need will be available from an export file
#  First, unlink any previous version of the definition file.
unlink($defFile);
# if ftcli is executable in this environment, run the export
# command.
&debug("Running: $exportCommand");
if (-X "./$ftcli")
{
  `$exportCommand`;
}

# Obtain a list of resource groups that are online locally
$lrgCommand = "$ftcli -d ${FT_DOMAIN} -c \"listresourcegroups\" | $GREP \" Online\" | $GREP \"$hostName\"";
@lrgOutput = `$lrgCommand`;
foreach $lrgLine(@lrgOutput)
{
  # Extract everything starting from the State field to get the resource group
  # names. If the group name contains " Online" then the wrong name will be
  # returned if the group is not Online/OnlinePending. This is not a problem
  # since we only care about groups that are Online/OnlinePending.
  $pos = rindex $lrgLine, " Online";
  $lrgLine = substr $lrgLine, 0, $pos;
  $lrgLine =~ s/ *$//g;
}
chomp(@lrgOutput);

&debug("Parsing .def file");

# Now grab a handle to the defFile.  If, for whatever reason, the file
# does not exist or can't be opened, the NSR_LC_TYPE is set to null
# causing only the rap header to be output.
open(DEFFILE,"<$defFile") or $NSR_LC_TYPE = "";


#  Cycle through the def file looking first for data sources definitions
#  and then for IP/Data Source objects in the Resource Group.  If the
#  previous open command failed, then NSR_LC_TYPE is set to NULL and we
#  will skip to the end.

# PARSING KEYS:
$keyDataSource = "datasource";
$keyDataSourceType = "volumeType.name";
if ($NSR_LC_TYPE)
{
  while ($LINE = <DEFFILE>)
  {
    ###################################################################
    #  PART 1: get the data source definitions
    ###################################################################
    #  CASE: NEW DATA SOURCE DEFINITION
    #  If the line begins DataSource, then we have a new data source
    #  definition coming.
    if ($LINE =~ /^${keyDataSource}\s+(\S+)\s+{/)
    {
      $dataSourceName = $1;
      next;
    }

    #  CASE:  LINE SPECIFIES A DATASOURCE TYPE
    #  If the Data Source Name has a value and the line indicates the
    #  data source type:
    if (($dataSourceName) && ($LINE =~ /^\s*${keyDataSourceType}\s*=\s*(\S+)/))
    {
      #  If the data source type is not appropriate (not NT_Shared_Disk or
      #  UX_File_System), then reset $dataSourceName and OSType to "".
      #  &useDataSource will
      #  not only check the type, but set the OSType var to NT or UNIX.
      if (!(&useDataSource($LINE)))
      {
        &resetDataSourceVars();
      }
      next;
    }

    #  CASE:  DATA SOURCE DEFINITION ATTACHPOINT
    #  if  $dataSourceName has a value and the line indicates this is
    #  an attachpoint, then we want to get the attach point path and
    #  add it to %dataSourcePaths where the key is the datasource name.
    #
    #  Note that the pattern match is for the DEFAULT datasource
    #  config.  This is an unfortunate requirement.
    if (($dataSourceName) && ($LINE =~ /name = <default>#FT_ATTACHPOINT/))
    {
      #  Get the attach point from the next line
      $LINE = <DEFFILE>;
      if ($LINE =~ /value\s*=\s*(\S+)/)
      {
        $AttachPoint = $1;
      }

      #  The following is to distinguish between NFS shares and
      #  locally attached mounts on UNIX.  If $doNFS is set to 1
      #  (non-zero true), then add the data source to
      #  %DataSourcePaths regardless.  If $doNFS is set to 0, then
      #  exclude the data source if it contains a ":" and is a UX type
      #  (indicating NFS).
      if ($doNFS)
      {
        $DataSourcePaths{$dataSourceName} = $AttachPoint;
        # If NT, append the ":\"
        if ($OSType eq "NT")
        {
          $DataSourcePaths{$dataSourceName} .= ":\\";
        }
       }
      else
      {
        #  If the attach point is local (UNIX) add the DataSource
        #  and the attach point to the assoc array
        if (!($AttachPoint =~ /:/) || ($OSType eq "NT"))
        {
          $DataSourcePaths{$dataSourceName} = $AttachPoint;
          # If NT, append the ":\"
          if ($OSType eq "NT")
          {
            $DataSourcePaths{$dataSourceName} .= ":\\";
          }
         }
      }
      #  Reset the vars for the next go around.  Save a couple cycles.
      &resetDataSourceVars();
      next;
    }
    # If the Data Source is a Mirror then find here the attach point
    if (($dataSourceName) && ($LINE =~ /name = FT_SRC_DRIVE/))
    {
      #  Get the attach point from the next line
      $LINE = <DEFFILE>;
      if ($LINE =~ /value\s*=\s*(\S+)\s*\S*/)
      {
        $AttachPoint = $1;
      }
      $DataSourcePaths{$dataSourceName} = $AttachPoint;
      # If NT, append the ":\"
      if ($OSType eq "NT")
      {
        $DataSourcePaths{$dataSourceName} .= ":\\";
      }

      #  Reset the vars for the next go around.  Save a couple cycles.
      &resetDataSourceVars();
      next;
    }

    #  CASE:  end of DS def: if $dataSourceName is defined and we see a "}"
    #  then we've come to the end of the definition.  Reset the vars.
    if (($dataSourceName) && ($LINE =~ m/^# End Object/))
    {
      &resetDataSourceVars();
      next;
    }


    ###################################################################
    #  PART 2: parse the RG defs getting the first IP address and the
    #          DS Names
    ###################################################################

    #  CASE:  New Resource Group defined
    if ($LINE =~ /^resourcegroup (.*) {$/)
    {
      # This is a resource group.  In AAM 5.0, the RG run time info
      # is not available in the definition file.  We need to verify
      # the current state and location of the RG.
      $rgName = $1;
      $rgCommand = "$ftcli -d ${FT_DOMAIN} -c \"getresourcegroup \\\"$rgName\\\"\"";
      @rgOutput = `$rgCommand`;
      undef %rgAttributeHash;
      foreach $rgLine(@rgOutput)
      {
        if ($rgLine =~ /name=(\S+)\s+value=(\S+)/)
        {
          $rgAttributeHash{$1} = $2;
        }
      }

      # Check that the RG is ONLINE locally.  If the
      # AAM for Networker module is configured for this RG, then
      # skip this check.
      if (defined($rgAttributeHash{ft_AAMNetworkerModule}) ||
          (grep "$_" eq "$rgName", @lrgOutput))
      {
        $OnlineState = "ONLINE";
      }
      else
      {
        $rgName = "";
        next;
      }

      # CASE: IPOverride attribute found.  Add it immediately to the
      # IPDataMap and and reset the vars.
      # The IPOvveride attribute value must be in
      # the format "<IPADDRESS>=<path>[,<path>[...]]".
      if ( defined($rgAttributeHash{IPOverride}) )
      {
        if ($rgAttributeHash{IPOverride} =~ /\s*(\S+)\s*=\s*(.+)/)
        {
          push(@IPAddress, $1);
          $dataSourceString = $2;
          $IP_Data_Map{$IPAddress[0]} = $dataSourceString;
          @IPAddress = ();
          $dataSourceString = "";
          $rgName = "";
        }
      }
      next;
    }
    if ($rgName)
    {
      # CASE:  Found an IP Address. Add it to @IPAddress array.
      if ($LINE =~ /^\s*ip.ipAddress.name\s*=\s*(\S+)/)
      {
        push(@IPAddress, $1);
      }

      if ($LINE =~ /^\s*ds.dataSourceID.name\s*=\s*(\S+)/)
      {
        $dataSourceEntry = $1;
        # if the path string is not empty, then add it to the
		# $dataSourceString
		if ($DataSourcePaths{$dataSourceEntry})
        {
			# if this isn't the first entry, then prepend a comma
			if ($dataSourceString)
			{
			  $dataSourceString .= ", $DataSourcePaths{$dataSourceEntry}";
			}
			else
			{
			  $dataSourceString = "$DataSourcePaths{$dataSourceEntry}";
			}
        }
	next;
      }

      # CASE:  LINE = END_STARTUP.  end of rule; commit the info reset the vars
      if ($LINE =~ /takeOfflineSequence/)
      {
        # If there exists at least one IP address and one data source, and the
        # state is ONLINE, save the data for rap printout.
        # Note that $OnlineState will be ONLINE if either the RG is online
        # OR the attribute ft_AAMNetworkerModule is set.
        $IPcount = @IPAddress;
        if (($IPcount > 0) && (uc($OnlineState) eq "ONLINE") &&
            $dataSourceString)
        {
            for($i = 0; $i < $IPcount ; $i++) {
                  $IP_Data_Map{$IPAddress[$i]} = $dataSourceString;
            }
        }
        @IPAddress = ();
        $IPcount = 0;
        $dataSourceString = "";
        $rgName = "";
        next;
      }
    } # end if($rgName)
  }   # end while(<LINE>)
}     # end if (NSR_LC_TYPE)


# Close the file handle
close(DEFFILE);
&debug("Creating Cache File");
&createCacheFile();

# Clean up the trash.
unlink($defFile);


&debug("Releasing Lock");
&releaseLock();
&debug("Here is the output:");
&printCache();
&debug("Done");
#chdir($cwd);

#######################################################################
#  Subroutines
#######################################################################

##########
# SUBROUTINE:	resetDataSourceVars()
#
# DESCRIPTION:set the data source vars to "".  These vars are keys in the script
# so it knows when it cares about the the contents of a particular line, i.e,
# if dataSoureName is "", then we are not looking at a line of the def file
# that contains relevant data source info.
##########

sub resetDataSourceVars
{
	$dataSourceName = "";
	$OSType = "";
}

#######################################################################
# SUBROUTINE:	useDataSource(String $CurrentLine)
#
# INPUT:	$currentLine -> current line read from the LC def file
# RETURN:	1(True), 0(False)
#
# DESCRIPTION:returns true if the Current line contains a data source type that
#	should be used.  Excluded data source types are volume managers
#	and NT_Network_Shares.
#######################################################################
sub useDataSource
{
	local ($currentLine) = @_;
	if ($currentLine =~ /winfilesystem/)
	{
		$OSType = "NT";
		return 1;
	}
	elsif ($currentLine =~ /mirror/)
	{
		$OSType = "NT";
		return 1;
	}
	elsif ($currentLine =~ /unixfilesystem/)
	{
		$OSType = "UNIX";
		return 1;
	}
	elsif ($currentLine =~ /NT_Shared_Disk/i)
	{
		$OSType = "NT";
		return 1;
	}
	elsif ($currentLine =~ /AAM_Mirror/i)
	{
		$OSType = "NT";
		return 1;
	}
	elsif ($currentLine =~ /solaris_fs/) #FK check and fix
	{
		$OSType = "UNIX";
		return 1;
	}
	elsif ($currentLine =~ /hpux_fs/) #FK check and fix
	{
		$OSType = "UNIX";
		return 1;
	}
	elsif ($currentLine =~ /linux_fs/) #FK check and fix
	{
		$OSType = "UNIX";
		return 1;
	}
	elsif ($currentLine =~ /aix_fs/) #FK check and fix
	{
		$OSType = "UNIX";
		return 1;
	}
	elsif ($currentLine =~ /nfs/) #FK check and fix
	{
		$OSType = "UNIX";
		return 1;
	}
	else
	{
		return 0;
	}
}

#######################################################################
# SUBROUTINE:  initpwd and chdir
#
# DESCRIPTION: utils to manage working directory changes.
#######################################################################
sub initpwd {
    if ($ENV{'PWD'}) {
	local($dd,$di) = stat('.');
	local($pd,$pi) = stat($ENV{'PWD'});
	if ($di != $pi || $dd != $pd) {
	    chop($ENV{'PWD'} = `cd`);
	}
    }
    else {
	chop($ENV{'PWD'} = `cd`);
    }
    if ($ENV{'PWD'} =~ m|(/[^/]+(/[^/]+/[^/]+))(.*)|) {
	local($pd,$pi) = stat($2);
	local($dd,$di) = stat($1);
	if ($di == $pi && $dd == $pd) {
	    $ENV{'PWD'}="$2$3";
	}
    }
}

sub chdir {
    local($newdir) = shift;
    if (chdir $newdir) {
	if ($newdir =~ m#^/#) {
	    $ENV{'PWD'} = $newdir;
	}
	else {
	    local(@curdir) = split(m#/#,$ENV{'PWD'});
	    @curdir = '' unless @curdir;
	    foreach $component (split(m#/#, $newdir)) {
		next if $component eq '.';
		pop(@curdir),next if $component eq '..';
		push(@curdir,$component);
	    }
	    $ENV{'PWD'} = join('/',@curdir) || '/';
	}
    }
    else {
	0;
    }
}

##########
#	SUBROUTINE:	grabLock
#
#	INPUT:
#	RETURN:
#
#	DESCRIPTION: Wait for $LockFile to be created. The max wait time is
#	controlled by the variable $LockTimeout.  After $LockTimeout seconds,
#	just read the stale cache if it exists.
##########
sub grabLock {
  local ($i) = $LockTimeout;
  local ($str);

  # if the LockFile exists, then somebody else has grabbed the
  # lock.  Spin for $LockTimeout seconds waiting for the lock to
  # be released Once $LockTimeout seconds have elapsed, just go
  # ahead and print out the stale cluster information
  if (-f $LockFile) {
    &debug("Lock exists, waiting for it to be released");
    while ($i-- > 0) {
      sleep(1);
      # Lock file has disappeared, this means somebody must've
      # updated the cache for us to use
      if (! -f $LockFile){
	$str = "$i seconds left in timeout, but lockfile disappeared";
	&debug($str);
	&printCache();
	exit(0);
      }
    }
  }

  # By virtue of somebody else grabbing the lock, it is possible
  # that the CacheFile has been updated. In that case, just print
  # out the contents of the cache file and exit

  # If the CacheFile hasn't been updated, then grab the lock and
  # return

  # If the lockfile still exists, and the timeout period has expired,
  # we need to break the lock
  if (! -f $LockFile){

    if (! &isCacheStale()) {
      &debug ("Ready to grab lock, but noticed that the cache has been refreshed");
      &printCache();
      exit(0);
    }

    open(LOCKFILE, ">$LockFile");
    close(LOCKFILE);
    &debug("Created lock file");
    return $TRUE;
  }
  else {
    &debug("Timeout has expired, but lockfile hasn't been removed, printing out stale info");
    &printCache();
  }
}

##########
#	SUBROUTINE:	releaseLock
#
#	INPUT:
#	RETURN:
#
#	DESCRIPTION:   Delete the $LockFile
##########
sub releaseLock {
  # Delete the lockfile
  # Make sure that the lock hasn't been broken
  unlink($LockFile);
}

##########
#	SUBROUTINE:	isCacheStale
#
#	INPUT:
#	RETURN:
#
#	DESCRIPTION:   Return TRUE if file is > $CacheStaleThreshold seconds old
##########
sub isCacheStale {
  local ($timeStamp);
  local ($now);
  local ($diff);

  if (!-f $CacheFile) {
    &debug("$CacheFile does not exist, asssuming stale cache");
    return $TRUE;
  }

  $now = time;
  $timeStamp = 0;
  $timeStamp = (stat $CacheFile)[9];

  &debug("Timestamp on $CacheFile: $timeStamp; Currently: $now");

  $diff = int($now) - int($timeStamp);
  &debug("The difference is $diff");

  if  (int($diff) > int($CacheStaleThreshold)) {
    return $TRUE;
  }

  return $FALSE;
}


##########
#	SUBROUTINE:	printCache
#
#	INPUT:
#	RETURN:
#
#	DESCRIPTION:   cat the contents of $CacheFile
##########
sub printCache {
  open(CACHEFILE, "<$CacheFile") || die "Cannot Open $CacheFile. Please check if $LockFile exists -";

  while (<CACHEFILE>) {
    print $_;
  }
}

##########
#	SUBROUTINE:	createCacheFile
#
#	INPUT:
#	RETURN:
#
#	DESCRIPTION:   Use globals $NSR_LC_TYPE and %IP_Data_Map to
#                      create the cache file. Note that we write data to
#                      a temporary file and mv it into the real file
##########
sub createCacheFile {

  local ($tempCache) = $CacheFile . ".tmp" . $$;

  open(CACHEFILE, ">$tempCache") || die "Cannot open $tempCache";
  &debug ("Opened temporary file: $tempCache");

  print CACHEFILE "type: NSR_CLU_TYPE;\n";
  print CACHEFILE "clu_type: $NSR_LC_TYPE;\n";
  print CACHEFILE "interface version: $version;\n\n";

  if ($NSR_LC_TYPE) {
    # for each key\value pair in IP_Data_Map, stdout an NSR_CLU_VIRTHOST definition
    while (($ip, $mounts) = each(%IP_Data_Map)) {
      print CACHEFILE "type: NSR_CLU_VIRTHOST;\n";
      print CACHEFILE "hostname: $ip;\n";
      print CACHEFILE "owned paths: $mounts;\n\n";
    }
  }

  close CACHEFILE;
  &debug("closed cachefile");


  &debug ("Closed temporary file: $tempCache");

  # If something strange happened and the ftcli output parsing didn't work correctly,
  # don't override the cache file
  if (-f $tempCache && !-z $tempCache)
  {
    &debug("Going to move temporary cache file to its final place ($CacheFile)");
    rename $tempCache,$CacheFile || &releaseLock() && die "Unable to create cache file. Error: $!\n";
    &debug ("Moved temporary file ($tempCache) to the real cache file ($CacheFile)");
  }
}


#################################################################################
# Subroutine:   debug()
# Arguments:    message
# Returns:      nothing
#################################################################################
sub debug {
  local ($msg) = @_;
    if ($Debug == $TRUE)
    {
	print "$$: $msg \n";
    }
    return;
}

