#!/bin/bash
# Copyright (C) 2016-2020 McAfee, LLC. All Rights Reserved.
# Set the environment variable where commands will be commonly present

# Use only bash for executing the script
ps -o args= -p "$$" | grep bash > /dev/null
if [ $? -ne 0 ]
then
     echo "Error: This script can be run by bash shell only"
     echo "Control script for Trellix Endpoint Security for Linux Threat Prevention"
     echo "$0 [start | stop | restart | status | enable | disable]"
     exit -1
fi

PATH=$PATH:/sbin:/usr/sbin:/bin:/usr/bin

function usage()
{
    echo "Control script for Trellix Endpoint Security for Linux Threat Prevention"
    echo "$0 [start | stop | restart | status | enable | disable]"
    exit -1
}

function checkUser()
{
    # Check control script is executed as root
    ID="$(id -u)"
    if [ $? -ne 0 -o "$ID" -ne 0 ]
    then
        echo "Must be root / sudo user to $initCmd Trellix Endpoint Security for Linux Threat Prevention"
        exit -1
    fi
}

if [ $# -ne 1 ]
then
    usage
fi

# Default return value
retVal=0
initCmd=$1
# Check the type of distribution
if [ -f "/etc/redhat-release" ]
then
    distribRel=$(cat /etc/redhat-release)
elif [ -f "/etc/system-release" ]
then
    distribRel=$(cat /etc/system-release)
elif [ -f "/etc/SuSE-release" ]
then
    distribRel=$(cat /etc/SuSE-release)
elif [ -f /etc/os-release ]
then
    distribRel=$(awk -F '=' '/^VERSION_ID=/ {print $2}' /etc/os-release | sed -e 's/"//g')
elif [ -f "/etc/lsb-release" ]
then
    distribRel=$(grep DISTRIB_RELEASE /etc/lsb-release | awk -F "=" '{print $2}')
fi
# Replace any space or () with _
distribRelString=$(echo ${distribRel//[ ()]/_})
# Enable case insensitive match
shopt -s nocasematch
# Detect the type of init supported and set the correct init commands for use
case $distribRelString in
    # RHEL 6, CentOS 6, Amazon Linux, Ubuntu 1204, 1404 and 1410  is using upstart
    12.04|\
    14.04|\
    14.10|\
    Red_Hat_Enterprise_Linux_Server_release_6*|\
    Red_Hat_Enterprise_Linux_Workstation_release_6*|\
    Amazon_Linux_AMI_release*|\
    CentOS_Linux_release_6*|\
    CentOS_release_6*)
        INIT_CMD="initctl $initCmd mfetpd"
        ENABLE_CMD="cp -f /opt/McAfee/ens/tp/init/mfetpd.conf /etc/init/mfetpd.conf"
        DISABLE_CMD="rm -f /etc/init/mfetpd.conf"
        ;;
    # Rest is systemd (RHEL and CentOS 7, Suse 12 etc.)
    *)
        INIT_CMD="systemctl --no-ask-password $initCmd mfetpd.service"
        ENABLE_CMD="systemctl --no-ask-password -q $initCmd mfetpd.service"
        DISABLE_CMD="systemctl --no-ask-password -q $initCmd mfetpd.service"
        ;;
esac
case $initCmd in
    start|\
    stop|\
    restart)
        checkUser
        eval ${INIT_CMD}
        retVal=$?
        ;;
    status)
        eval ${INIT_CMD}
        retVal=$?
        ;;
    enable)
        checkUser
        eval ${ENABLE_CMD}
        retVal=$?
        ;;
    disable)
        checkUser
        eval ${DISABLE_CMD}
        retVal=$?
        ;;
    *)
        usage
        ;;
esac

exit $retVal
