#!/bin/bash
# Copyright (C) 2016-2021 McAfee, LLC. All Rights Reserved.
# This is wrapper script which will call fileaccess-control.sh and depending on the
# return value, it will update the UseKernelModule tag of pref.xml
# This script is usually called by the init script and should always return 0

FILEACCESS_KMOD_SCRIPTS_DIR=/opt/McAfee/ens/esp/modules/fileaccess/scripts
MFE_TP_VAR_DIR=/var/McAfee/ens/tp
MFE_TP_ROOT_DIR=/opt/McAfee/ens/tp

USE_KERNEL_MODULE=0

getSyscallTableAddress64Bit()
{
    if [ -f /proc/kallsyms ]
    then
        addressFile=/proc/kallsyms
    else
        echo "File /proc/kallsyms is not available in system. FileAccess kernel module will not load."
        return 1
    fi

    sys_call_table_str=`cat ${addressFile} | grep -w "sys_call_table$" | grep -v grep | awk '{print $1}'`
    sed -e "s/<SyscallTableAddress>.*<\/SyscallTableAddress>/<SyscallTableAddress>${sys_call_table_str}<\/SyscallTableAddress>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
    mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
    sed -e "s/<SyscallTableAddress>.*<\/SyscallTableAddress>/<SyscallTableAddress>${sys_call_table_str}<\/SyscallTableAddress>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
    mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
}

getKallsymsLookUpNameAddress()
{
    kallsyms_lookup_name_addr=`cat /proc/kallsyms | grep -w "kallsyms_lookup_name$" | awk '{print $1}'`
    sed -e "s/<KallsymsLookUpNameAddress>.*<\/KallsymsLookUpNameAddress>/<KallsymsLookUpNameAddress>${kallsyms_lookup_name_addr}<\/KallsymsLookUpNameAddress>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
    mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
    sed -e "s/<KallsymsLookUpNameAddress>.*<\/KallsymsLookUpNameAddress>/<KallsymsLookUpNameAddress>${kallsyms_lookup_name_addr}<\/KallsymsLookUpNameAddress>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
    mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
}

getSyscallTableAddress32Bit()
{
    if [ -f /proc/kallsyms ]
    then
        addressFile=/proc/kallsyms
    else
        echo "File /proc/kallsyms is not available in system. FileAccess kernel module will not load."
        return 1
    fi

    sys_call_table_32_bit_str=`cat ${addressFile} | grep -w "ia32_sys_call_table$" | grep -v grep | awk '{print $1}'`
    sed -e "s/<SyscallTableAddress32Bit>.*<\/SyscallTableAddress32Bit>/<SyscallTableAddress32Bit>${sys_call_table_32_bit_str}<\/SyscallTableAddress32Bit>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
    mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
    sed -e "s/<SyscallTableAddress32Bit>.*<\/SyscallTableAddress32Bit>/<SyscallTableAddress32Bit>${sys_call_table_32_bit_str}<\/SyscallTableAddress32Bit>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
    mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
}

checkKernelModuleSupportAndUpdatePrefXml()
{
    # If UseFanotify is enabled, pass "usefanotify" as an argument to fileaccess-control.sh
    USE_FANOTIFY=$(sed -n 's:.*<UseFanotify>\(.*\)</UseFanotify>.*:\1:p' ${MFE_TP_VAR_DIR}/prefs.xml)
    if [ $USE_FANOTIFY -eq 0 ]
    then
        ${FILEACCESS_KMOD_SCRIPTS_DIR}/fileaccess-control.sh
    else
        ${FILEACCESS_KMOD_SCRIPTS_DIR}/fileaccess-control.sh usefanotify
    fi

    if [ $? -eq 0 ]
    then
        USE_KERNEL_MODULE=1
    fi
    #get syscall table address for 64 bit system calls and update to prefs.xml
    getSyscallTableAddress64Bit
    #get kallsyms_lookup_name address and update to prefs.xml
    getKallsymsLookUpNameAddress
    #get syscall table address for 32 bit system calls and update to prefs.xml
    getSyscallTableAddress32Bit
    # If USE_KERNEL_MODULE is 1 i.e. kernel module support is enabled , then disable UseFanotify.
    if [ $USE_KERNEL_MODULE -eq 1 ]
    then
        sed -e "s/<UseFanotify>.<\/UseFanotify>/<UseFanotify>0<\/UseFanotify>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
        mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
        sed -e "s/<UseFanotify>.<\/UseFanotify>/<UseFanotify>0<\/UseFanotify>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
        mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
    else
        # If kernel module support is disabled, then enable UseFanotify
        sed -e "s/<UseFanotify>.<\/UseFanotify>/<UseFanotify>1<\/UseFanotify>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
        mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
        sed -e "s/<UseFanotify>.<\/UseFanotify>/<UseFanotify>1<\/UseFanotify>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
        mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
    fi
    #Replace any UseKernelModule tag value (i.e. 0 or 1) with the latest value
    sed -e "s/<UseKernelModule>.<\/UseKernelModule>/<UseKernelModule>${USE_KERNEL_MODULE}<\/UseKernelModule>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
    mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
    sed -e "s/<UseKernelModule>.<\/UseKernelModule>/<UseKernelModule>${USE_KERNEL_MODULE}<\/UseKernelModule>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
    mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml

}

checkIsOASSupportedAndUpdatePrefXml()
{
    minDebianSupport=8
    isOASSupported=1
    #For Kernel module supported distro , OAS is always supported
    if [ $USE_KERNEL_MODULE -eq 0 ]
    then
        if [ -f /etc/os-release ]
        then
            distribRelString=$(awk -F '=' '/^ID=/ {print $2}' /etc/os-release)
            distribRelMajNum=$(awk -F '=' '/^VERSION_ID=/ {print $2}' /etc/os-release | sed -e 's/\.//g' -e 's/"//g')
            distribRel=$(awk -F '=' '/^PRETTY_NAME=/ {print $2}' /etc/os-release)
        fi
        if [ -n "${distribRelString}" ]
        then
            if [ "${distribRelString}" = "debian" ]
            then
                if [ "${distribRelMajNum}" -le "${minDebianSupport}" ]
                then
                    isOASSupported=0
            	fi
            fi
        fi
    fi

    #Update var/prefs.xml
    #Check IsOASSupported xml tag is present , if not add a new one else update the tag
    count=`grep -i "IsOASSupported" ${MFE_TP_VAR_DIR}/prefs.xml | wc -l`
    if [ $count -eq 0 ]
    then
        isOASSupportTag="<IsOASSupported>$isOASSupported</IsOASSupported>"
        #Adding space to match proper xml alignment
        sed -e "/IsOASEnabled/ a \ \ \ \ \ \ \ \ ${isOASSupportTag}" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
        mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
    else
        sed -e "s/<IsOASSupported>.<\/IsOASSupported>/<IsOASSupported>${isOASSupported}<\/IsOASSupported>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
        mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
    fi

    #Update etc/prefs.xml
    #Check IsOASSupported xml tag is present , if not add a new one else update the tag
    count=`grep -i "IsOASSupported" ${MFE_TP_ROOT_DIR}/etc/prefs.xml | wc -l`
    if [ $count -eq 0 ]
    then
        isOASSupportTag="<IsOASSupported>$isOASSupported</IsOASSupported>"
        #Adding space to match proper xml alignment
        sed -e "/IsOASEnabled/ a \ \ \ \ \ \ \ \ ${isOASSupportTag}" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
        mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
    else
        sed -e "s/<IsOASSupported>.<\/IsOASSupported>/<IsOASSupported>${isOASSupported}<\/IsOASSupported>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
        mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
    fi

}
# If usefanotify is passed as a parameter it will check whether fanotify is supported on the machine or not
# If fanotify is supported , fileaccess-control.sh will return 1 else 0
# However the output of fileaccess-control.sh is ignored as fileaccess-control-wrapper.sh should always return 0
if [ ! -z $1 ]
then
    if [ $1 == "usefanotify" ]
    then
        ${FILEACCESS_KMOD_SCRIPTS_DIR}/fileaccess-control.sh usefanotify
        exit 0
    fi
fi

checkIsFanotifySupportedAndUpdatePrefXml()
{
    unameValue=$(uname -r)
    bootConfigFile='/boot/config-'$unameValue
    if [ -f $bootConfigFile ]
    then
        # If CONFIG_FANOTIFY or CONFIG_FANOTIFY_ACCESS_PERMISSIONS is not true in /boot/config-`uname -r`
        # Then FANotify is not supported by the kernel
        # Check these two flags and update prefs.xml accordingly
        configFanotify=$(awk -F '=' '/^CONFIG_FANOTIFY=/ {print $2}' $bootConfigFile)
        fanotifyAccessPermission=$(awk -F '=' '/^CONFIG_FANOTIFY_ACCESS_PERMISSIONS=/ {print $2}' $bootConfigFile)
        if [ "$configFanotify" == "y" ] && [ "$fanotifyAccessPermission" == "y" ]
        then
            sed -e "s/<IsFanotifySupported>.<\/IsFanotifySupported>/<IsFanotifySupported>1<\/IsFanotifySupported>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
            mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
            sed -e "s/<IsFanotifySupported>.<\/IsFanotifySupported>/<IsFanotifySupported>1<\/IsFanotifySupported>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
            mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
        else
            sed -e "s/<IsFanotifySupported>.<\/IsFanotifySupported>/<IsFanotifySupported>0<\/IsFanotifySupported>/g" ${MFE_TP_VAR_DIR}/prefs.xml > ${MFE_TP_VAR_DIR}/prefs.xml-updated
            mv ${MFE_TP_VAR_DIR}/prefs.xml-updated ${MFE_TP_VAR_DIR}/prefs.xml
            sed -e "s/<IsFanotifySupported>.<\/IsFanotifySupported>/<IsFanotifySupported>0<\/IsFanotifySupported>/g" ${MFE_TP_ROOT_DIR}/etc/prefs.xml > ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated
            mv ${MFE_TP_ROOT_DIR}/etc/prefs.xml-updated ${MFE_TP_ROOT_DIR}/etc/prefs.xml
        fi
    fi
}

checkKernelModuleSupportAndUpdatePrefXml
checkIsOASSupportedAndUpdatePrefXml
checkIsFanotifySupportedAndUpdatePrefXml

# prefs.xml permission gets changed when we copy prefs to updated prefs and  vice-versa 
chmod 600 ${MFE_TP_ROOT_DIR}/etc/prefs.xml
chmod 600 ${MFE_TP_VAR_DIR}/prefs.xml

exit 0
