#!/bin/bash
# Copyright (C) 2019-2020 McAfee, LLC. All Rights Reserved.
# Script to validate if ESP installation / upgrade can be done.
# It also works for HF packages

# Exit codes on failure in this script are
# 1 - This script can be run by bash shell only.
# 2 - Invalid command line option passed during installation.
# 5 - Must be a root user to run this script.
# 6 - 64bit MFEcma 5.6.4-110 or above is required for installation to continue.
# 7 - Installation file is missing.
# 8 - Installation RPM or DEB file is missing.
# 9 - Installation failed.
# 10 - Failed to extract downloaded installation file.
# 16 - Installation conflicts with existing TP installation.
# 17 - Installation conflicts with existing FW installation.
# 18 - Installation conflicts with existing TP and FW installation.
# 0 - Installation was successful
# NOTE: Exit codes are kept uniform for all installation and uninstallation scripts

# Unset all the language locale and set LANG to C to disable localization
unset LC_ALL
unset LANG
export LANG=C
export LC_ALL=C

usage()
{
    echo "Usage: $0 install-type log-file-path installation-directory esp-basic-tarball [esp-kernel-module-tarball]"
    echo "install-type can be 'silent', 'prompt' or 'epo'"
    echo "log-file-path is the absolute path of the log file where log messages will also go"
    echo "installation-directory is the directory from where installation scripts are executed and packages are extracted"
    echo "esp-basic-tarball is the absolute path of the ESP tar.gz which contains ESP and Runtime Debian and RPM packages"
    echo "esp-kernel-module-tarball is optional and is the absolute path of the ESP Kernel Module tar.gz"
    echo "  This contains ESP File Access and ESP AAC Debian and RPM packages"
    exit 2
}

# Function to log messages to stdout and syslog
logMessage()
{
    message=$1
    echo "${message}"
    logger -t ${me} "${message}"
    # If running in prompt mode, save the output to the log file as well
    if [ ${installType} = "prompt" ]
    then
        echo "${message}" >> $logFile
    fi
}

# Compares two product versions a and b and return its value. Both should be in 10.2.2.1105 format
# Returns 1 for more (a > b), 2 for less (a < b) and 0 for equal version (a == b)
vercomp () {
    if [[ $1 == $2 ]]
    then
        return 0
    fi
    a=$1
    b=$2
    a=$(echo ${a}| sed 's/-/./')
    b=$(echo ${b} | sed 's/-/./')
    local IFS=.
    local i ver1=($a) ver2=($b)
    # fill empty fields in ver1 with zeros
    for ((i=${#ver1[@]}; i<${#ver2[@]}; i++))
    do
        ver1[i]=0
    done
    for ((i=0; i<${#ver1[@]}; i++))
    do
        if [[ -z ${ver2[i]} ]]
        then
            # fill empty fields in ver2 with zeros
            ver2[i]=0
        fi
        if ((10#${ver1[i]} > 10#${ver2[i]}))
        then
            return 1
        fi
        if ((10#${ver1[i]} < 10#${ver2[i]}))
        then
            return 2
        fi
    done
    return 0
}

# Check minimum MA version which supports 64 bit Message Bus. Ref KB KB51573 and KB82105
# First parameter is "yes" if this is a debian system
checkMAVersion()
{
    debian=$1
    if [ ${debian} = "yes" ]
    then
        installedMAArch=""
        dpkg -s mfecma 2>/dev/null | grep "Status: install ok installed" >/dev/null 2>&1
        if [ $? -ne 0 ]
        then
            logMessage "MFEcma(x86_64) 5.6.4-110 or above is required for installation to continue."
            cleanup 6
        fi
        installedMAVersion=$(dpkg -s mfecma | grep ^Version | awk -F ': ' '{print $2}')
        installedMARelease=$(dpkg -s mfecma | grep ^Release | awk -F ': ' '{print $2}')
        installedMAVersion="${installedMAVersion}.${installedMARelease}"
        if [ -f "/opt/McAfee/agent/lib/lib64/rsdk/libma_msgbus.so" ]
        then
            installedMAArch="x86_64"
        fi
    else
        installedMAVersion=$(rpm -q --queryformat "%{VERSION}.%{RELEASE}\n" MFEcma)
        if [ $? -ne 0 ]
        then
            logMessage "MFEcma(x86_64) 5.6.4-110 or above is required for installation to continue."
            cleanup 6
        fi
        installedMAArch=$(rpm -q --queryformat "%{ARCH}\n" MFEcma)
    fi
    vercomp "5.6.4.110" ${installedMAVersion}
    if [ $? -eq 1 ]
    then
        logMessage "MFEcma ${installedMAVersion} is unsupported."
        logMessage "MFEcma(x86_64) 5.6.4-110 or above is required for installation to continue."
        cleanup 6
    fi
    if [ "${installedMAArch}" = "x86_64" ]
    then
        logMessage "MFEcma(x86_64) ${installedMAVersion} is installed."
    else
        logMessage "MFEcma(x86) ${installedMAVersion} is unsupported."
        logMessage "MFEcma(x86_64) 5.6.4-110 or above is required for installation to continue."
        cleanup 6
    fi
}

# Common method to get the version of a product installed
# First argument is the name of the product whose version is required. The letters case of the product name
# should be that of rpm as rpm is case sensitive whereas dpkg command is case insensitive.
# Uses echo to return the installed product version in the format: 10.6.6.102 which caller should save
getInstalledProductVersion()
{
    productName=$1
    if [ ${DEB_VARIANT} = "yes" ]
    then
        installedProductVersion=$(dpkg -s ${productName} | grep ^Version | awk -F ': ' '{print $2}')
    else
        installedProductVersion=$(rpm -q --queryformat "%{VERSION}.%{RELEASE}\n" ${productName})
    fi
    echo ${installedProductVersion}
}

# This method accepts
# $1 - Name of the product
# $2 - Version of the product that is currently available
# This method returns-
# 0- when the version installed is the same and we need not upgrade
# 1- when the product is not installed and we need to do a fresh installation
# 2- when version installed in this system is older than available version
getInstallTypeRPM()
{
    productName=$1
    availableProductVersionString=$2
    installedProductVersionString=$(rpm -q --queryformat "%{VERSION}.%{RELEASE}\n" ${productName})
    if [ $? -ne 0 ]
    then
        logMessage "${productName} will be installed"
        return 1
    fi
    logMessage "${productName} installed in this system is - ${installedProductVersionString}"
    availableProductVersionString=$(echo ${availableProductVersionString} | sed 's/\(.*\)-/\1./')
    vercomp ${availableProductVersionString} ${installedProductVersionString}
    upgradeType=$?
    if [ ${upgradeType} -eq 1 ]
    then
        logMessage "${productName} is an upgrade candidate, since version installed in this system is ${installedProductVersionString}, which is older than available version ${availableProductVersionString}"
        return 2
    elif [ ${upgradeType} -eq 2 ]
    then
        logMessage "Not upgrading ${productName} since version installed in this system is ${installedProductVersionString}, which is newer than available version ${availableProductVersionString}"
        return 0
    elif [ ${upgradeType} -eq 0 ]
    then
        logMessage "Not upgrading ${productName} since version installed in this system is ${installedProductVersionString}, is same as available version ${availableProductVersionString}"
        return 0
    fi
}

# This method accepts
# $1 - Name of the product
# $2 - Version of the product that is currently available
# This method returns-
# 0- when no upgrade is required.
# 1- when the product is not installed and we need to do a fresh installation
# 2- when version installed in this system is older than available version
getInstallTypeDEB()
{
    productName=$1
    availableProductVersionString=$2
    minSupportedProductVersionString=$3
    installedProductVersionString=$(dpkg -s ${productName} 2>/dev/null | grep ^Version | awk -F ': ' '{print $2}')
    if [ -z ${installedProductVersionString} ]
    then
        logMessage "Installing ${productName} as it is currently not installed"
        return 1
    else
        logMessage "${productName} installed in this system is - ${installedProductVersionString}"
        dpkg --compare-versions "${availableProductVersionString}" gt "${installedProductVersionString}"
        if [ $? -eq 0 ]
        then
            logMessage "${productName} is an upgrade candidate, since version installed in this system is ${installedProductVersionString}, which is older than available version ${availableProductVersionString}"
            return 2
        else
            dpkg --compare-versions "${availableProductVersionString}" lt "${installedProductVersionString}"
            if [ $? -eq 0 ]
            then
                logMessage "Not upgrading ${productName} since version installed in this system is ${installedProductVersionString}, which is newer than available version ${availableProductVersionString}"
            fi
            return 0
        fi
    fi
}

# Function to obsolete a package, if present. Else install the other provided package via getInstallTypeRPM
# 1st param - Package or an array of packages(if multiple pcakages need to be checked against) which are going to be obsoleted. (package 1)
# 2nd param - New package which is obsoleting 1st package. (package 2)
# 3rd param - Version of package 2 bundled in this pack.
# This function returns -
# 2 : If package 1 is present and is getting updated to package 2
# Return value of getInstallTypeRPM() : In case if package 1 is not present.
upgradeObsoletedOrInstallTypeRPM()
{
    packagesListToObsolete=( $(echo "$1") )
    packageToInstall=$2
    packageToInstallVersionString=$3
    for packageToObsolete in "${packagesListToObsolete[@]}"
    do
        # Check whether the package to be obsoleted is present or not.
        rpm -qa | grep -q ${packageToObsolete}
        if [ $? -eq 0 ]
        then
            obsoletedPackageVersion=$(rpm -q --queryformat "%{VERSION}.%{RELEASE}\n" ${packageToObsolete})
            logMessage "${packageToObsolete} installed in this system is - ${obsoletedPackageVersion}"
            logMessage "${packageToObsolete} will be upgraded to ${packageToInstall}"
            return 2
        fi
    done
    # The package that we attempted to obsolete is not present, proceed with normal course of installation.
    getInstallTypeRPM ${packageToInstall} ${packageToInstallVersionString}
    return $?
}

# Function to obsolete a package, if present. Else install the other provided package via getInstallTypeDEB
# 1st param - Package which is going to be obsoleted. (package 1)
# 2nd param - New package which is obsoleting 1st package. (package 2)
# 3rd param - Version of package 2 bundled in this pack.
# This function returns -
# 2 : If package 1 is present and is getting updated to package 2
# Return value of getInstallTypeDEB() : In case if package 1 is not present.
upgradeObsoletedOrInstallTypeDEB()
{
    packageToObsolete=$1
    packageToInstall=$2
    packageToInstallVersionString=$3

    # Check whether the package to be obsoleted is present or not.
    obsoletedPackageVersion=$(dpkg -s ${packageToObsolete} 2>/dev/null | grep ^Version | awk -F ': ' '{print $2}')
    if [ ! -z "${obsoletedPackageVersion}" ]
    then
        logMessage "${packageToObsolete} installed in this system is - ${obsoletedPackageVersion}"
        logMessage "${packageToObsolete} will be upgraded to ${packageToInstall}"
        return 2
    else
        # The package that we attempted to obsolete is not present, proceed with normal course of installation.
        getInstallTypeDEB ${packageToInstall} ${packageToInstallVersionString}
        return $?
    fi
}

# Test if RPM package installation generates a conflict with McAfeeTP or McAfeeFW
# TBD: Should check for VSEL be added?
# Update global variables for any conflicts
checkForRPMConflicts()
{
    fileToTest=$1
    productNameToTest=$2
    logMessage "Testing for any conflict in installing ${productNameToTest}"
    resultOfTest=$(rpm --test -i ${fileToTest} 2>&1)
    checkForISecTPConflict=$(echo ${resultOfTest} | grep -q -E 'ISecTP .* conflicts')
    # Returns 0 if grep was successful in getting a match
    if [ $? -eq 0 ]
    then
        isTPConflicted=1
        installedISecTPVersion=$(getInstalledProductVersion ISecTP 2>/dev/null)
        logMessage "New version of ${productNameToTest} conflicts with installed ISecTP ${installedISecTPVersion}. ISecTP should also be upgraded."
    else
        checkForMFETPConflict=$(echo ${resultOfTest} | grep -q -E 'McAfeeTP .* conflicts')
        if [ $? -eq 0 ]
        then
            isTPConflicted=1
            installedMFETPVersion=$(getInstalledProductVersion McAfeeTP 2>/dev/null)
            logMessage "New version of ${productNameToTest} conflicts with installed McAfeeTP ${installedMFETPVersion}. McAfeeTP should also be upgraded."
        fi
    fi
    checkForFWConflict=$(echo ${resultOfTest} | grep -q -E 'McAfeeFW .* conflicts')
    # Returns 0 if grep was successful in getting a match
    if [ $? -eq 0 ]
    then
        isFWConflicted=1
        installedFWVersion=$(getInstalledProductVersion McAfeeFW 2>/dev/null)
        logMessage "New version of ${productNameToTest} conflicts with installed McAfeeFW ${installedFWVersion}. McAfeeFW should also be upgraded."
    fi
}

# Test if DEB package installation generates a conflict with mcafeetp or mcafeefw
# TBD: Should check for VSEL be added?
# Update global variables for any conflicts
checkForDEBConflicts()
{
    fileToTest=$1
    productNameToTest=$2
    logMessage "Testing for any conflict in installing ${productNameToTest}"
    resultOfTest=$(apt-get --no-remove -s -o Dir::Etc::Sourcelist=${tmpDir}/sources.list -o Dir::Etc::SourceParts=${tmpDir} -o Dir::Cache::Archives=${tmpDir}/cache/apt/archives -o Dir::State::Lists=${tmpDir}/lib/apt/lists -o DPkg::Options::=--admindir=${tmpDir}/lib/dpkg install ${productNameToTest} 2>&1)
    checkForISecTPConflict=$(echo ${resultOfTest} | grep -q -iE isectp)
    # Returns 0 if grep was successful in getting a match
    if [ $? -eq 0 ]
    then
        isTPConflicted=1
        # Using the product name used for rpm as dpkg command is case insensitive
        installedISecTPVersion=$(getInstalledProductVersion ISecTP 2>/dev/null)
        logMessage "New version of ${productNameToTest} conflicts with installed isectp ${installedISecTPVersion}. isectp should also be upgraded."
    else
        checkForMFETPConflict=$(echo ${resultOfTest} | grep -q -iE mcafeetp)
        if [ $? -eq 0 ]
        then
            isTPConflicted=1
            installedMFETPVersion=$(getInstalledProductVersion McAfeeTP 2>/dev/null)
            logMessage "New version of ${productNameToTest} conflicts with installed mcafeetp ${installedMFETPVersion}. mcafeetp should also be upgraded."
        fi
    fi
    checkForFWConflict=$(echo ${resultOfTest} | grep -q -iE "mcafeefw")
    # Returns 0 if grep was successful in getting a match
    if [ $? -eq 0 ]
    then
        isFWConflicted=1
        installedFWVersion=$(getInstalledProductVersion McAfeeFW 2>/dev/null)
        logMessage "New version of ${productNameToTest} conflicts with installed mcafeefw ${installedFWVersion}. mcafeefw should also be upgraded."
    fi
}

# Function to cleanup temporary files
cleanup()
{
    # Save exit code
    exitCode=$1
    # Delete the temporary directory
    rm -rf "${tmpDir}"
    exit ${exitCode}
}

# This will be used to automatically determine the name of the output log file
declare -r me=${0##*/}
# Set default permissions of files to allow read / write only for owner
umask 077
# Used to track if this is a rpm or deb based system
# yes - Debian based system, no for RPM based system
DEB_VARIANT="no"

# In prompt mode, it will also save output to a log file
installType="prompt"

# Check installer is executed as root
ID="$(id -u)"
if [ $? -ne 0 -o "$ID" -ne 0 ]
then
    logMessage "Must be root to install this product"
    exit 5
fi

numargs=$#
if [ "${numargs}" -lt 4 ]
then
    echo "Invalid option"
    usage
else
    # tmpDir is a mandatory option and is a directory where installation will be done from.
    # Caller PP script should be using the same directory to install the PP from.
    # But PP should not copy any file here before calling our script, it may get deleted.
    installType=$1
    logFile=$2
    tmpDir=$3
    installerFile=$4
    installKernelModule=0
    kernelModuleFile=""
    # Install Kernel Module if available
    if [ "${numargs}" -gt 4 ]
    then
        installKernelModule=1
        kernelModuleFile=$5
    fi
fi

# Ensure tmpDir exists
if [ ! -d ${tmpDir} ]
then
    usage
fi

if [ ! -f "${installerFile}" ]
then
    logMessage "Unable to locate the main installation file ${installerFile}"
    exit 7
fi

# If expected version of McAfeeESP kernel is not present then we look for any available McAfeeESP-KernelModule package
# and try to install available KM package.
if [ ${installKernelModule} -eq 1 ]
then
    if [ ! -f "${kernelModuleFile}" ]
    then
        availableKernelModuleFile=$(ls McAfeeESP-KernelModule-*-Full.linux.tar.gz | tail -1)
        if [ ! -z "${availableKernelModuleFile}" ] && [ -f "${availableKernelModuleFile}" ]
        then
            kernelModuleFile=${availableKernelModuleFile}
            logMessage "Proceeding with installation with available kernel module ${availableKernelModuleFile}"
        else
            logMessage "Unable to locate the kernel module installation file ${kernelModuleFile}"
            exit 7
        fi
    fi
fi

logMessage "${me} execution time: $(date)"

# Flags to set installation type for each module
# 0 - Do not install, 1 - Install, 2 - Upgrade module
isecRTInstall=0
mfeESPInstall=0
mfeESPFileAccessInstall=0
mfeESPAACInstall=0
# This is the number of packages that needs to be installed / upgraded
noOfPkgsToInstall=0

# Flags to track if any of the ESP packages have a conflict with a PP
# 0 - No conflict with PP, 1 - new ESP conflicts with PP
isTPConflicted=0
isFWConflicted=0

# All known rpm based systems either have /etc/redhat-release (Redhat, Fedora, CentOS) or /etc/SuSE-release (SuSE, openSuSE)
if [ -f /etc/redhat-release -o -f /etc/SuSE-release ]
then
    DEB_VARIANT="no"
elif [ -f /etc/system-release ]
then
    # Amazon Linux AMI is rpm based, has /etc/system-release and its content starts with Amazon
    distribRel=$(cat /etc/system-release)
    amazonSearchPattern="^Amazon"
    if [[ $distribRel =~ $amazonSearchPattern ]]
    then
        DEB_VARIANT="no"
    fi
elif [ -f /etc/os-release ]
then
    # SuSE 15 and above does not ship with /etc/SuSE-release; check /etc/os-release instead
    distribId=$(cat /etc/os-release | grep ^ID=)
    slesSearchPattern="sles|sled"
    opensuseSearchPattern="opensuse|openSUSE"
    if [[ $distribId =~ $slesSearchPattern ]]
    then
        DEB_VARIANT="no"
    elif [[ $distribId =~ $opensuseSearchPattern ]]
    then
        DEB_VARIANT="no"
    else
        DEB_VARIANT="yes"
    fi
else
    DEB_VARIANT="yes"
fi

tar -C "${tmpDir}" -xzf ${installerFile}
if [ $? -ne 0 ]
then
    logMessage "Failed to extract the installation file ${installerFile}"
    cleanup 10
fi

if [ ${installKernelModule} -eq 1 ]
then
    tar -C "${tmpDir}" -xzf ${kernelModuleFile}
    if [ $? -ne 0 ]
    then
        logMessage "Failed to extract the installation file ${kernelModuleFile}"
        cleanup 10
    fi
fi

if [ ${DEB_VARIANT} = "yes" ]
then
    logMessage "Detected deb based distribution"
    checkMAVersion ${DEB_VARIANT}
    # Since this is a RPM based distribution, delete any DEB packages.
    # Delete any tar.gz which will also not be required.
    /bin/rm -f ${tmpDir}/*.rpm ${tmpDir}/*.tar.gz
else
    logMessage "Detected rpm based distribution"
    checkMAVersion ${DEB_VARIANT}
    # Since this is a RPM based distribution, delete any DEB packages.
    # Delete any tar.gz which will also not be required.
    /bin/rm -f ${tmpDir}/*.deb ${tmpDir}/*.tar.gz
fi

# For every known rpm / deb in the directory; check if it can be installed or upgraded
# If it cannot be installed or upgraded, then delete it from the directory
# Caller script should install the left over rpm / deb in the directory using yum / zypper / apt-get
for file in ${tmpDir}/*
do
    case "$file" in
        *McAfeeRt-*)
            MCAFEE_RT_PACKAGE_FILE="${file}"
            if [ ${DEB_VARIANT} = "yes" ]
            then
                AVAILABLE_MCAFEE_RT_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeRt-\(.*\).deb/\1/')
                # If Runtime is shipped, Check if Runtime for Linux is installed
                getInstallTypeDEB McAfeeRt ${AVAILABLE_MCAFEE_RT_VERSION_INT}
                isecRTInstall=$?
            else
                AVAILABLE_MCAFEE_RT_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeRt-\(.*\).x86_64.rpm/\1/')
                # If Runtime is shipped, Check if Runtime for Linux is installed
                getInstallTypeRPM McAfeeRt ${AVAILABLE_MCAFEE_RT_VERSION_INT}
                isecRTInstall=$?
            fi
            # Delete the package, if it is not being installed or upgraded
            if [ ${isecRTInstall} -eq 0 ]
            then
                rm -f ${file}
            else
                let noOfPkgsToInstall="${noOfPkgsToInstall}+1"
            fi
            ;;
        *McAfeeESPAac-*)
            MCAFEE_ESP_AAC_PACKAGE_FILE="${file}"
            if [ ${DEB_VARIANT} = "yes" ]
            then
                AVAILABLE_MCAFEE_ESP_AAC_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeESPAac-\(.*\).deb/\1/')
                # If McAfeeESPAAC is shipped, Check if ISecESPAAC for Linux is installed
                # If yes, we need to upgrade it to McAfeeESPAAC
                # Else we need to go with normal McAfeeESPAAC installation.
                upgradeObsoletedOrInstallTypeDEB isecespaac mcafeeespaac ${AVAILABLE_MCAFEE_ESP_AAC_VERSION_INT}
                mfeESPAACInstall=$?
            else
                AVAILABLE_MCAFEE_ESP_AAC_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeESPAac-\(.*\).x86_64.rpm/\1/')
                # If McAfeeESPAAC is shipped, Check if ISecESPAAC for Linux or HIPS(HIPS can be present only on RPM based distributions)
                # If yes, we need to upgrade it to McAfeeESPAAC
                # Else we need to go with normal McAfeeESPAAC installation.
                # Pass the packages to be obsoletes as array variable
                obsoletePackageArray=("ISecESPAac" "MFEhiplsm")
                obsoletePacakes="${obsoletePackageArray[@]}"
                upgradeObsoletedOrInstallTypeRPM "$obsoletePacakes" McAfeeESPAac ${AVAILABLE_MCAFEE_ESP_AAC_VERSION_INT}
                mfeESPAACInstall=$?
            fi
            # Delete the package, if it is not being installed or upgraded
            if [ ${mfeESPAACInstall} -eq 0 ]
            then
                rm -f ${file}
            else
                let noOfPkgsToInstall="${noOfPkgsToInstall}+1"
            fi
            ;;
        *McAfeeESPFileAccess-*)
            MFE_ESP_FILEACCESS_PACKAGE_FILE="${file}"
            if [ ${DEB_VARIANT} = "yes" ]
            then
                AVAILABLE_MFE_ESP_FILEACCESS_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeESPFileAccess-\(.*\).deb/\1/')
                # If McAfeeESPESPFileAccess is shipped, Check if ISecESPFileAccess for Linux is installed
                # If yes, we need to upgrade it to McAfeeESPFileAccess
                # Else we need to go with normal McAfeeESPFileAccess installation.
                upgradeObsoletedOrInstallTypeDEB isecespfileaccess mcafeeespfileaccess ${AVAILABLE_MFE_ESP_FILEACCESS_VERSION_INT}
                mfeESPFileAccessInstall=$?
            else
                AVAILABLE_MFE_ESP_FILEACCESS_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeESPFileAccess-\(.*\).x86_64.rpm/\1/')
                # If McAfeeESPESPFileAccess is shipped, Check if ISecESPFileAccess for Linux is installed
                # If yes, we need to upgrade it to McAfeeESPFileAccess
                # Else we need to go with normal McAfeeESPFileAccess installation.
                upgradeObsoletedOrInstallTypeRPM ISecESPFileAccess McAfeeESPFileAccess ${AVAILABLE_MFE_ESP_FILEACCESS_VERSION_INT}
                mfeESPFileAccessInstall=$?
            fi
            # Delete the package, if it is not being installed or upgraded
            if [ ${mfeESPFileAccessInstall} -eq 0 ]
            then
                rm -f ${file}
            else
                let noOfPkgsToInstall="${noOfPkgsToInstall}+1"
            fi
            ;;
        *McAfeeESP-*)
            MCAFEE_ESP_PACKAGE_FILE="${file}"
            if [ ${DEB_VARIANT} = "yes" ]
            then
                AVAILABLE_MCAFEE_ESP_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeESP-\(.*\).deb/\1/')
                # If ESP is shipped, Check if ISecESP for Linux is installed
                # If yes, we need to upgrade it to McAfeeESP
                # Else we need to go with normal McAfeeESP installation.
                upgradeObsoletedOrInstallTypeDEB isecesp mcafeeesp ${AVAILABLE_MCAFEE_ESP_VERSION_INT}
                mfeESPInstall=$?
            else
                AVAILABLE_MCAFEE_ESP_VERSION_INT=$(echo "${file}" | sed 's/.*McAfeeESP-\(.*\).x86_64.rpm/\1/')
                # If ESP is shipped, Check if ISecESP for Linux is installed
                # If yes, we need to upgrade it to McAfeeESP
                # Else we need to go with normal McAfeeESP installation.
                upgradeObsoletedOrInstallTypeRPM ISecESP McAfeeESP ${AVAILABLE_MCAFEE_ESP_VERSION_INT}
                mfeESPInstall=$?
            fi
            # Delete the package, if it is not being installed or upgraded
            if [ ${mfeESPInstall} -eq 0 ]
            then
                rm -f ${file}
            else
                let noOfPkgsToInstall="${noOfPkgsToInstall}+1"
            fi
            ;;
    esac
done

# Test if package installation generates a conflict with a known product
if [ ${noOfPkgsToInstall} -gt 0 ]
then
    # For Debian based distribution, create a temporary file repository
    if [ ${DEB_VARIANT} = "yes" ]
    then
        logMessage "Creating a temporary file based apt repository in ${tmpDir}"
        echo "deb file:${tmpDir} ./" > ${tmpDir}/sources.list
        # Create the cache and lib directory
        mkdir -p ${tmpDir}/cache/apt/archives
        mkdir -p ${tmpDir}/lib/apt/lists
        # Reuse the existing system dpkg directory
        ln -s /var/lib/dpkg ${tmpDir}/lib/dpkg
        if [ $? -ne 0 ]
        then
            logMessage "Error in creating a symlink to existing dpkg administrative directory"
            cleanup 9
        fi
        logMessage "Created a symlink to existing dpkg administrative directory"
        pushd ${tmpDir} > /dev/null 2>&1
        apt-ftparchive packages . > Packages
        gzip -f Packages
        popd > /dev/null 2>&1
        cp -f ${tmpDir}/Packages.gz ${tmpDir}/lib/apt/lists/
        if [ $? -ne 0 ]
        then
            logMessage "Error in creating Packages.gz containing our packages"
            cleanup 9
        fi
        apt-get -o Dir::Etc::Sourcelist=${tmpDir}/sources.list -o Dir::Etc::SourceParts=${tmpDir} -o Dir::Cache::Archives=${tmpDir}/cache/apt/archives -o Dir::State::Lists=${tmpDir}/lib/apt/lists -o DPkg::Options::=--admindir=${tmpDir}/lib/dpkg -o Acquire::AllowInsecureRepositories=true update > /dev/null 2>&1
        if [ $? -ne 0 ]
        then
            logMessage "Error in running apt-get update from temporary apt repository"
            cleanup 9
        fi
        if [ ${isecRTInstall} -gt 0 ]
        then
            checkForDEBConflicts ${MCAFEE_RT_PACKAGE_FILE} "mcafeert"
        fi
        if [ ${mfeESPAACInstall} -gt 0 ]
        then
            checkForDEBConflicts ${MCAFEE_ESP_AAC_PACKAGE_FILE} "mcafeeespaac"
        fi
        if [ ${mfeESPFileAccessInstall} -gt 0 ]
        then
            checkForDEBConflicts ${MFE_ESP_FILEACCESS_PACKAGE_FILE} "mcafeeespfileaccess"
        fi
        if [ ${mfeESPInstall} -gt 0 ]
        then
            checkForDEBConflicts ${MCAFEE_ESP_PACKAGE_FILE} "mcafeeesp"
        fi
    else
        if [ ${isecRTInstall} -gt 0 ]
        then
            checkForRPMConflicts ${MCAFEE_RT_PACKAGE_FILE} "McAfeeRt"
        fi
        if [ ${mfeESPAACInstall} -gt 0 ]
        then
            checkForRPMConflicts ${MCAFEE_ESP_AAC_PACKAGE_FILE} "McAfeeESPAac"
        fi
        if [ ${mfeESPFileAccessInstall} -gt 0 ]
        then
            checkForRPMConflicts ${MFE_ESP_FILEACCESS_PACKAGE_FILE} "McAfeeESPFileAccess"
        fi
        if [ ${mfeESPInstall} -gt 0 ]
        then
            checkForRPMConflicts ${MCAFEE_ESP_PACKAGE_FILE} "McAfeeESP"
        fi
    fi
fi

if [ ${isTPConflicted} -eq 1 -a ${isFWConflicted} -eq 1 ]
then
    retVal=18
elif [ ${isTPConflicted} -eq 1 ]
then
    retVal=16
elif [ ${isFWConflicted} -eq 1 ]
then
    retVal=17
else
    retVal=0
fi
exit ${retVal}
