#!/bin/bash
# Copyright (C) 2016-2021 McAfee, LLC. All Rights Reserved.

# Use only bash for executing the script
ps -o args= -p "$$" | grep bash > /dev/null
if [ $? -ne 0 ]
then
     echo "Error: This script can be run by bash shell only"
     exit -1
fi

# Set the environment variable where commands will be commonly present
PATH=$PATH:/sbin:/usr/sbin:/bin:/usr/bin

# Unset environment variables
unset LD_LIBRARY_PATH

#Check if Run By root or not
if [ $EUID -ne 0 ]
then
    echo "This script can be run by root or a user with sudo privileges"
    exit -1
fi

echo "Running Trellix Endpoint Security for Linux Threat Prevention MER tool - 10.7.13-20"

exec <&-        # close stdin
exec 3>&1       # duplicate stdout

declare -r logDir=$HOME
declare -r mfeoasmgrLogDir=/var/McAfee/ens/log/tp/mfeoasmgr
declare -r mfetpcliLogDir=/var/McAfee/ens/log/tp/mfetpcli
declare -r mfescanactionmgrLogDir=/var/McAfee/ens/log/tp/mfescanactionmgr
declare -r mfescanfactoryLogDir=/var/McAfee/ens/log/tp/mfescanfactory
declare -r mfetpdLogDir=/var/McAfee/ens/log/tp/mfetpd
declare -r installLogDir=/tmp
declare -r odscollectorDir=/var/McAfee/ens/log/tp/mfeodscollector
declare -r odsreportDir=/var/McAfee/ens/log/tp/odsreport
declare -r prefXml=/var/McAfee/ens/tp/prefs.xml

# Mertool output prefix file name
log="mfetp-mer.txt"
section="0"
subsection="0"
subsubsection="0"
timestamp=$(date +%d_%m_%Y_%H-%M-%S)
targzFileName=mfetp-mer_${timestamp}.tar.gz
mfe_tp_dir_install_path=/opt/McAfee/ens/tp/bin
mfeesp_dir_install_path=/opt/McAfee/ens/esp
mfetp_init_path=/opt/McAfee/ens/tp/init
#Tells if Debian or other distro
isDebianTypeDistro=0
# Assume timeout binary is not present
TIMEOUT_PRESENT=0
trap cleanup EXIT
cleanup()
{
    return 0
}
setLog()
{
    if [ -s "$1" ]
    then
        rm -f $1 || return -1
    fi
    exec 4>"$1"
    return 0
}

sendLog()
{
    exec >&4
}

sendUser()
{
    exec >&3
}

# Sets isDebianTypeDistro  as 0 if Debian based and 1 if non-Debian
getLinuxDistro()
{
    if [ -x /usr/bin/dpkg ]
    then
        isDebianTypeDistro=0
    else
        isDebianTypeDistro=1
    fi
}

newSection()
{
    local name="$1"

    section=$(($section + 1))
    subsection="0"
    subsubsection="0"
    subsectionEnabled="true"
    subsubsectionEnabled="true"

    sectionEnabled="true"
    [[ -n "$onlySection" ]] &&
    [[ "$onlySection" -ne "$section" ]] &&
    sectionEnabled=""

    [[ -n "$sectionEnabled" ]] &&
    printf "\n%s%d\t%s\n" \
           "${outline:+*}" "$section" "$name"
}

newSubSection()
{
    local name="$1"

    subsection=$(($subsection + 1))
    subsubsection="0"
    subsubsectionEnabled="true"

    subsectionEnabled="true"
    [[ -n "$onlySubSection" ]] &&
    [[ "$onlySubSection" -ne "$subsection" ]] &&
        subsectionEnabled=""

    [[ -n "$sectionEnabled" ]] &&
    [[ -n "$subsectionEnabled" ]] &&
        printf "\n%s%d.%d\t%s\n" \
           "${outline:+**}" "$section" "$subsection" "$name"
}

newSubSubSection()
{
    local name="$1"

    subsubsection=$(($subsubsection + 1))

    subsubsectionEnabled="true"
    [[ -n "$onlySubSubSection" ]] &&
    [[ "$onlySubSubSection" -ne "$subsubsection" ]] &&
        subsubsectionEnabled=""

    [[ -n "$sectionEnabled" ]] &&
    [[ -n "$subsectionEnabled" ]] &&
    [[ -n "$subsubsectionEnabled" ]] &&
        printf "\n%s%d.%d.%d\t%s\n" \
           "${outline:+***}" "$section" "$subsection" "$subsubsection" "$name"
}

runPipeCommand()
{
    [[ -n "$sectionEnabled" ]] &&
    [[ -n "$subsectionEnabled" ]] &&
    [[ -n "$subsubsectionEnabled" ]] &&
        eval LANG=C "$@" 2>&1
}

# Run a command with a timeout of 5 seconds only if available
# Use this only for content which may take time to collect and is good to have
runPipeCommandWithTimeout()
{
    if [ ${TIMEOUT_PRESENT} -eq 1 ]
    then
        [[ -n "$sectionEnabled" ]] &&
        [[ -n "$subsectionEnabled" ]] &&
        [[ -n "$subsubsectionEnabled" ]] &&
            eval LANG=C "timeout 5 $@" 2>&1
    fi
}

runCommand()
{
    [[ -n "$sectionEnabled" ]] &&
    [[ -n "$subsectionEnabled" ]] &&
    [[ -n "$subsubsectionEnabled" ]] &&
    printf "\n[%s]\n" "$*" &&
        eval LANG=C "$@" 2>&1
}

runAlternativeTimeout()
{
    time=$1
    shift
    exec $@ &
    pid=$!
    interval=1
    while [ $time -gt 0 ]
    do
        sleep $interval
        kill -0 $pid 2>/dev/null || return
        time=$(( time-interval ))
    done
    kill -0 $pid 2>/dev/null  && { kill $pid && wait $pid; } 2>/dev/null
}


# Run a command with a timeout of 5 seconds only if available
# Use this only for content which may take time to collect and is good to have
runCommandWithTimeout()
{
    if [ ${TIMEOUT_PRESENT} -eq 1 ]
    then
        [[ -n "$sectionEnabled" ]] &&
        [[ -n "$subsectionEnabled" ]] &&
        [[ -n "$subsubsectionEnabled" ]] &&
        printf "\n[%s]\n" "$*" &&
            eval LANG=C "timeout 5 $@" 2>&1
    else
        [[ -n "$sectionEnabled" ]] &&
        [[ -n "$subsectionEnabled" ]] &&
        [[ -n "$subsubsectionEnabled" ]] &&
        printf "\n[%s]\n" "$*" &&
        runAlternativeTimeout 5 $@ 2>&1
    fi
}

alternativeLsof()
{
    pids=$( ls /proc )
    for pid in $pids
    do
        fdDir=/proc/$pid/fd
            if [ -d  $fdDir ]
            then
                fds=$( ls $fdDir )
                for fd in $fds
                do
                    link=$fdDir/$fd
                    if [ -s $link ]
                    then
                        isNum=$( echo $pid | tr -d "[0-9]" )
                        if [ -z $isNum ]
                        then
                            processInfo=$( ps -p $pid -o ppid= -o user= -o comm= )
                            ppid=$( echo $processInfo | awk '{print $1}' )
                            puser=$( echo $processInfo | awk '{print $2}' )
                            pname=$( echo $processInfo | awk '{print $3}' )
                            pfile=$( readlink -f $link )
                            printf "%-20s %-10s %-10s %-20s \n" $pname $pid $ppid $pfile
                       fi
                   fi
               done
            fi
   done
}

dumpAPRuleConfig()
{
    newSubSection "AP Rule Configuration Details"

    # extract the count of currently configured rules in AP
    apRuleCount=$( "$mfe_tp_dir_install_path/mfetpcli" --getallaprule  2>&1 | sed -n '/^|/p' | cut -d ' ' -f1 | cut -d '|' -f2 | tail -n 1 )
    if [ -z "$apRuleCount" ]
    then
        echo "\nNo Rules are Present\n"
        return
    fi
    # rules start from index 1 to the count we obtained above.
    ruleIndex=1
    while [[ "$ruleIndex" -le "$apRuleCount" ]]
    do
        runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getapruleconfig $ruleIndex
        ruleIndex=$[$ruleIndex+1]
    done
}

if [[ -n "$log" ]]
then
    setLog "${log}"
    sendLog
else
    sendUser
fi

# Check if timeout binary is present
which timeout > /dev/null 2>&1
if [ $? == 0 ]
then
    TIMEOUT_PRESENT=1
fi

#create temp dir for all file output
merToolTempDir=$(mktemp -d)

echo "Trellix Endpoint Security for Linux Threat Prevention diagnostic report"
echo "Created on: $(date)"
#Section Output Started
newSection "Trellix Endpoint Security for Linux Threat Prevention"

newSubSection "Running ESP mer script"

runCommand "$mfeesp_dir_install_path/scripts/mfeesp-mer.sh" > /dev/null
runPipeCommand cp ./mfeesp-mer*  $merToolTempDir/
runPipeCommand rm ./mfeesp-mer*

newSubSection "Trellix Endpoint Security for Linux Threat Prevention files"

runCommand ls -aRl /opt/McAfee/ens/tp
runCommand ls -aRl /var/McAfee/ens/log/tp
runCommand ls -aRl /var/McAfee/ens/tp

# Loader config file
if [ -s "/etc/ld-mfeensrt-3.0.so.conf" ]
then
    runPipeCommand cp /etc/ld-mfeensrt-3.0.so.conf  $merToolTempDir/
fi

# Loader cache file
if [ -s "/etc/ld-mfeensrt-3.0.so.cache" ]
then
    runPipeCommand cp /etc/ld-mfeensrt-3.0.so.cache  $merToolTempDir/
fi

initConfigurationCopied=false
# Init configuration file for upstart based distributions
if [ -s "/etc/init/mfetpd.conf" ]
then
    runPipeCommand cp /etc/init/mfetpd.conf $merToolTempDir/
    initConfigurationCopied=true
fi

# Init configuration file for sysVinit based distributions
if [ -s "/etc/init.d/mfetpd" ]
then
    runPipeCommand cp /etc/init.d/mfetpd $merToolTempDir/
    initConfigurationCopied=true
fi

# Init configuration file for systemd based distributions
if [ -s "/usr/lib/systemd/system/mfetpd.service" ]
then
    runPipeCommand cp /usr/lib/systemd/system/mfetpd.service $merToolTempDir/
    initConfigurationCopied=true
fi

if [ "${initConfigurationCopied}" = false ]
then
    echo "No init configuration file found for Trellix Endpoint Security for Linux Threat Prevention."
fi

# VSEL to ENSL migration log
if [ -s "/var/McAfee/ens/log/tp/migrate-vsel.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/migrate-vsel.log  $merToolTempDir/
fi

# ENSL upgrade log
if [ -s "/var/McAfee/ens/log/tp/upgrade.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/upgrade.log  $merToolTempDir/
fi

# ENSL migration log
if [ -s "/var/McAfee/ens/log/tp/migrate-mfetp.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/migrate-mfetp.log  $merToolTempDir/
fi

# Threat prevention Product log
if [ -s "/var/McAfee/ens/log/tp/mfetpd.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/mfetpd.log*  $merToolTempDir/
fi

# CLI Product log
if [ -s "/var/McAfee/ens/log/tp/mfetpcli.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/mfetpcli.log*   $merToolTempDir/
fi

# mfeoasmgr log
if [ -s "/var/McAfee/ens/log/tp/mfeoasmgr.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/mfeoasmgr.log*   $merToolTempDir/
fi

# mfescanactionmgr log
if [ -s "/var/McAfee/ens/log/tp/mfescanactionmgr.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/mfescanactionmgr.log*   $merToolTempDir/
fi

# mfescanfactory log
if [ -s "/var/McAfee/ens/log/tp/mfescanfactory.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/mfescanfactory.log*   $merToolTempDir/
fi

# mfeodscollector log
if [ -s "/var/McAfee/ens/log/tp/mfeodscollector.log" ]
then
    runPipeCommand cp /var/McAfee/ens/log/tp/mfeodscollector.log*   $merToolTempDir/
fi

# TP preference xml
if [ -s "/var/McAfee/ens/tp/prefs.xml" ]
then
    runPipeCommand cp /var/McAfee/ens/tp/prefs.xml   $merToolTempDir/TPPrefs.xml
fi

# TP default preference xml
if [ -s "/opt/McAfee/ens/tp/etc/prefs.xml" ]
then
    runPipeCommand cp /opt/McAfee/ens/tp/etc/prefs.xml   $merToolTempDir/Default_TPPrefs.xml
fi

# Cron tab file
if [ -s "/etc/crontab" ]
then
    runPipeCommand cp /etc/crontab   $merToolTempDir/
fi

# AP rule Cache file
if [ -s "/var/McAfee/ens/tp/aprulecache.db" ]
then
    runPipeCommand cp /var/McAfee/ens/tp/aprulecache.db $merToolTempDir/
fi

# Exploit Prevention content rule cache file
if [ -s "/var/McAfee/ens/tp/ep/epcontentrules.db" ]
then
    runPipeCommand cp /var/McAfee/ens/tp/ep/epcontentrules.db $merToolTempDir/
fi

# Exploit Prevention combined rule cache file
if [ -s "/var/McAfee/ens/tp/ep/epcombinedrules.db" ]
then
    runPipeCommand cp /var/McAfee/ens/tp/ep/epcombinedrules.db $merToolTempDir/
fi

# Exploit Prevention TCL rules format xml
if [ -s "/var/McAfee/ens/tp/ep/eprulestclformat.xml" ]
then
    runPipeCommand cp /var/McAfee/ens/tp/ep/eprulestclformat.xml $merToolTempDir/EpRulesTclFormat.xml
fi

newSection "System Info"

newSubSection "Loader Info"
runCommand ls -al /opt/McAfee/ens/runtime/3.0/lib/ld-linux-x86-64.so.2

newSubSection "OS Info"
# Check the type of distribution
if [ -f "/etc/redhat-release" ]
then
    runCommand cat /etc/redhat-release
elif [ -f "/etc/system-release" ]
then
    runCommand cat /etc/system-release
elif [ -f "/etc/SuSE-release" ]
then
    runCommand cat /etc/SuSE-release
elif [ -f "/etc/lsb-release" ]
then
    runCommand cat /etc/lsb-release
elif [ -f "/etc/debian_version" ]
then
    runCommand cat /etc/debian_version
elif [ -f "/etc/os-release" ]
then
    runCommand cat /etc/os-release
fi

newSubSection "Iptables Info"
runCommand iptables -L -n -v

newSubSection "System Information"
newSubSubSection "Operating System"
runCommand uname -a

newSubSubSection "System uptime"
runCommand uptime

newSubSubSection "Last logged in users"
runCommand last -n 20

newSubSubSection "Installed software"
getLinuxDistro
if [ $isDebianTypeDistro -eq 0 ]
then
    runCommandWithTimeout dpkg -l
else
    printf "\n[%s]\n" "rpm -qa|sort"
    runPipeCommandWithTimeout rpm -qa|sort
fi


newSubSection "Free resources"
newSubSubSection "File system"
runCommand df -h

newSubSubSection "Memory"
runCommand free

newSubSubSection "Virtual Memory"
runCommand vmstat

newSubSubSection "Kernel"

runCommand uname -r
runCommand cat /boot/config-'uname -r' | grep -i XEN
runCommand lsmod
runCommand cat /proc/net/netlink
runCommand cat /proc/sys/kernel/osrelease
runCommand cat /proc/sys/kernel/version
runCommand cat /proc/cmdline
runCommand cat /proc/devices
runCommand cat /proc/modules
runCommand cat /proc/slabinfo
runCommand cat /proc/filesystems
runCommand cat /proc/mounts
runCommand cat /proc/partitions
runCommand cat /proc/fs/nfs/exports
runCommand cat /proc/sys/fs/inode-state
runCommand cat /proc/sys/fs/dentry-state
runCommand cat /proc/sys/fs/file-nr

newSubSubSection "Limits"
runCommand ulimit -a

newSubSubSection "Exports"
runCommand export

newSubSection "Network"
newSubSubSection "Interfaces"
runCommand ip addr ls

newSubSubSection "Routes"
runCommand ip route

newSubSubSection "Connection information"
runCommand ss -nautp

newSubSubSection "Socket information"
runCommand ss -nx

# Closest equivalent to netstat -s
newSubSubSection "Protocol statistics"
runCommand nstat -asz

newSubSubSection "Runtime Kernel parameters"
runCommand sysctl -a

#Open files info
which lsof > /dev/null 2>&1
if [ $? == 0 ]
then
    runCommand lsof > ${merToolTempDir}/lsof.txt
else
    runCommand alternativeLsof | sort | sed '/^(deleted/d' > ${merToolTempDir}/lsof.txt
fi
#Check if journalctl is present or not
which journalctl > /dev/null 2>&1
if [ $? == 0 ]
then
    runCommand journalctl > ${merToolTempDir}/journalctl.txt
fi
newSubSection "Processes"
newSubSubSection "Process Tree"
runCommand pstree

newSubSubSection "Top Processes"
runCommand top -b -n 1

newSubSubSection "Top Processes (with Threads)"
runCommand top -H -b -n 1

newSubSubSection "Process Details"
runCommand ps -elwf

newSubSubSection "Process Details with Thread information and Thread name"
runCommand ps -elwT

newSubSubSection "Process Details with Process and Thread startup time"
runCommand ps -eT -wo user,pid,spid,%cpu,%mem,vsz,rss,tty,stat,lstart,cmd

newSubSection "Hardware"
newSubSubSection "CPU"
runCommand cat /proc/cpuinfo

newSubSubSection "Memory"
runCommand cat /proc/meminfo

newSubSection "IPC"
runCommand cat /proc/sysvipc/msg
runCommand cat /proc/sysvipc/sem
runCommand cat /proc/sysvipc/shm

# Add support to copy firewall log file that SLES 12 is configured to log to
if [ -f /var/log/firewall ]
then
    runPipeCommand cp /var/log/firewall* ${merToolTempDir}/
fi

if [ -f /usr/bin/sar ]
then
        newSubSection "SAR output"
        runCommand /usr/bin/sar -A
fi
if [ -f /usr/sbin/sestatus ]
then
        newSubSection "SELinux Status"
        runCommand /usr/sbin/sestatus
fi

newSubSection "Threat Prevention Status"
newSubSubSection "Summary"
runCommand "$mfetp_init_path/mfetpd-control.sh" status

newSubSubSection "Version"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --version

newSubSubSection "OAS Profile Standard"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasprofileconfig standard

newSubSubSection "OAS Profile Highrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasprofileconfig highrisk

newSubSubSection "OAS Profile Lowrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasprofileconfig lowrisk

newSubSubSection "OAS Summary"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --summary

newSubSubSection "Process list"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --processlist

newSubSubSection "Exclusion list standard"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --exclusionlist --profile standard

newSubSubSection "Exclusion list highrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --exclusionlist --profile highrisk

newSubSubSection "Exclusion list lowrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --exclusionlist --profile lowrisk

newSubSubSection "File types to scan Default and Specified Profile- standard"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --listfiletypestoscan defaultandspecified --profile standard

newSubSubSection "File types to scan Default and Specified Profile- highrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --listfiletypestoscan defaultandspecified --profile highrisk

newSubSubSection "File types to scan Default and Specified Profile- lowrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --listfiletypestoscan defaultandspecified --profile lowrisk

newSubSubSection "File types to scan OnlySpecified Profile- standard"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --listfiletypestoscan onlyspecified --profile standard

newSubSubSection "File types to scan OnlySpecified Profile- highrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --listfiletypestoscan onlyspecified --profile highrisk

newSubSubSection "File types to scan OnlySpecified Profile- lowrisk"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoasconfig --listfiletypestoscan onlyspecified --profile lowrisk

newSubSubSection "OAS Deferred scan status"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getdeferredscan

newSubSubSection "OAS CPU usage limit"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getoascpulimit

newSubSubSection "Tasks"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --listtasks

newSubSubSection "Scan cache status"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --showscancachestatus

newSubSubSection "Log Settings"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --showlogsettings

newSubSubSection "Repository Settings"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --showrepositories

newSubSubSection "AP Status"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getapstatus

newSubSubSection "AP Rules"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getallaprule

newSubSubSection "Exploit Prevention Status"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getepstatus

newSubSubSection "Exploit Prevention Signatures"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getallepsignatures

newSubSubSection "Exploit Prevention Exclusions"
runCommandWithTimeout "$mfe_tp_dir_install_path/mfetpcli" --getepexclusions

# To dump detailed configuration for AP rules
dumpAPRuleConfig

newSubSection "Resolv Info"
echo ""
runCommand cat /etc/resolv.conf
echo ""

newSubSection "Hosts Info"
echo ""
runCommand cat /etc/hosts
echo ""

#for mfeoasmgr log directory
if [ -d $mfeoasmgrLogDir ]
then
tar -czf ${merToolTempDir}/mfeoasmgrLogDir.tar.gz $mfeoasmgrLogDir/* >/dev/null 2>&1
fi

#for mfetpcli log directory
if [ -d $mfetpcliLogDir ]
then
tar -czf ${merToolTempDir}/mfetpcliLogDir.tar.gz $mfetpcliLogDir/*.log* >/dev/null 2>&1
fi

#for mfescanactionmgr log directory
if [ -d $mfescanactionmgrLogDir ]
then
tar -czf ${merToolTempDir}/mfescanactionmgrLogDir.tar.gz $mfescanactionmgrLogDir/*.log* >/dev/null 2>&1
fi

#for mfescanfactory log directory
if [ -d $mfescanfactoryLogDir ]
then
tar -czf ${merToolTempDir}/mfescanfactoryLogDir.tar.gz $mfescanfactoryLogDir/*.log* >/dev/null 2>&1
fi

#for mfetpd log directory
if [ -d $mfetpdLogDir ]
then
tar -czf ${merToolTempDir}/mfetpdLogDir.tar.gz $mfetpdLogDir/* >/dev/null 2>&1
fi

# For installation log files
if [ -d $installLogDir ]
then
tar -czf ${merToolTempDir}/installLogDir.tar.gz /tmp/ensltp-standalone-setup.log /tmp/ensltp-epo-setup.log $installLogDir/install-mfetp.sh.log $installLogDir/uninstall-mfetp.sh.log $installLogDir/isectp-setup.log $installLogDir/setupHF.log >/dev/null 2>&1
fi

#for ODS Report directory
if [ -d $odsreportDir ]
then
tar -czf ${merToolTempDir}/odsreportDir.tar.gz $odsreportDir/*.log* $odsreportDir/archive/* >/dev/null 2>&1
fi

# For ODS Collectory directory
if [ -d $odscollectorDir ]
then
tar -czf ${merToolTempDir}/odscollectorDir.tar.gz $odscollectorDir/* > /dev/null 2>&1
fi

#for Quarantine meta files
quarantineDir=$(grep "<QuarantineDirectory>" $prefXml |  cut -d'<' -f2 | cut -d'>' -f2)
if [ -d $quarantineDir ]
then
tar -czf ${merToolTempDir}/quarantineMeta.tar.gz $quarantineDir*.meta >/dev/null 2>&1
runCommand ls -alZR ${quarantineDir} > ${merToolTempDir}/quarantine_files_with_selinux_context.txt
fi

# TP default rules
if [ -d "/var/McAfee/ens/tp/apdefaultrules" ]
then
tar -czf $merToolTempDir/apdefaultrules.tar.gz /var/McAfee/ens/tp/apdefaultrules >/dev/null 2>&1
fi

#copy Mertool log file to temp folder for comprerssion
cp $log ${merToolTempDir}

curLogDir=$( pwd )
pushd $merToolTempDir  >> /dev/null
outputMerDir="mfetp-mer_${timestamp}"
mkdir -p ${outputMerDir}
mv * ${outputMerDir}/ > /dev/null 2>&1
tar -czf $curLogDir/$targzFileName ${outputMerDir} >/dev/null 2>&1
popd >> /dev/null

#Remove temp directory
rm -rf ${merToolTempDir}
#Remove Mertool log file
rm $curLogDir/$log
exec >&3
echo "Completed generating Trellix Endpoint Security for Linux Threat Prevention diagnostic report."
echo "It is available at ${curLogDir}/${targzFileName}."
exit 0
