#!/usr/bin/python
# Copyright (C) 2016-2019 McAfee, LLC. All Rights Reserved.
#  - Check if supported VSEL is installed
#  - Determine the actual product installation directory
#  - Extract the migration data from the following files and prepare relevant CLI commands
#    - VSEL Configuration File
#    - ODS Configuration File and Sqlite DB
#    - Crontab
# NOTE: This migration will be done only for Unmanaged systems
# NOTE: Any ePO Task or exclusions added from ePO will be ignored


import os
import sys
import re
import shutil
import subprocess
import logging
import platform
import time

CLI_BIN = '/opt/McAfee/ens/tp/bin/mfetpcli'
NAILS_OPTIONS_FILE = "/opt/NAI/package/McAfeeVSEForLinux/nails.options"
VSEL_PACKAGE_NAME = "McAfeeVSEForLinux"
LOG_FILE_PATH = "/var/McAfee/ens/log/tp/"
NAILS_BACKUP_DIR = "/var/McAfee/ens/log/tp/vsel"
NAILS_LIB_DIR = "/opt/NAI/LinuxShield/lib/"
Update_Task_Counter = 1

def usage():
    '''
    Usage/Help
    '''
    print(sys.argv[0] +" </absolute/path/to/backup-dir>")

def _isOSDebianVar():
    '''
    Checks the os release file
    RETURN: False for rpm based system
            True for debian based system
    '''
    debvar = False
    os_name = None

    if sys.version_info[:3] >= (2, 6, 0):
        os_name = platform.linux_distribution()[0]
    else:
        os_name = platform.dist()[0]
    if os_name.lower() in ('ubuntu', 'linuxmint', 'debian'):
        debvar = True
    return debvar

def executeCmd(args, shellFlag=False, waitForOutput=True, env=None):
    """
    Method to execute system command and return response.
    This method returns a tuple on execution.
    If the command execution is successful, returns a tuple (0,result, error)
    If the command execution is unsuccessful, returns a tuple (resultCode, result, error )
    """
    try:
        output = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shellFlag, env=env)
        if not waitForOutput:
            return (0, "", "")
        (res, err) = output.communicate()
        rescode = output.poll()
        if rescode != 0:
            logging.debug("Response code of execution of command is %d" % rescode)
            logging.debug("Response string is %s" % err.strip())
        return (rescode, res, err)
    except:
        _e = sys.exc_info()[1]
        logging.exception("Exception occured while executing %s" % _e)
        return (1, "Failed to execute", "Fail")

def isVSELInstalled():
    '''
    Checks for VSEL package in database
    RETURN : True if VSEL is installed
            False if VSEL is not installed
    '''
    pkg_name = VSEL_PACKAGE_NAME
    _cmd = ['rpm', '-q']
    debvar = _isOSDebianVar()

    if debvar:
        _cmd = ['dpkg', '-l']
        pkg_name = VSEL_PACKAGE_NAME.lower()

    _rescode, _res, _err = executeCmd(_cmd + [pkg_name])
    if _rescode:
        #Unable to get the list of packages"
        return False
    else:
        output = _res.strip()
        if output.endswith("not-installed") or output.endswith("(no description available)"):
            return False
    return True

def isSystemManaged():
    ''' Fn to verify that it is not ePO managed system
        RETURN : True , if system is running in managed mode
                 False , if system is running in unmanaged mode
    '''
    _cmd = '/opt/McAfee/agent/bin/cmdagent -i'
    _ret, _output, _err = executeCmd(_cmd, True)

    if _output and _ret == 0:
        regex = '\S+\s+AgentMode:\s+(\d)'
        mode = re.findall(regex, _output.strip())
        if mode[0] != '0':
            return True
    else:
        logging.error("Could not determine if system is managed or unmanaged")
    return False

def dumpODSTaskFromDB(sqlite_path, nails_db_path, ods_filepath):
    ''' Reads the On -demand task data of schedule table from nailsd.db
        and copies it to text file in backup dir
        RETURN: 0 , if successfully copies the data
                2 , if no ods task is present
                1 , otherwise
        '''
    #query for ods tasks
    _query_ods = "select * from schedule where taskType='On-Demand' and taskInfo not LIKE 'profileName=ePO_%';"
    ods_data = _runQuery(sqlite_path, nails_db_path, _query_ods)
    if ods_data is None:
        logging.error("Failure in dumping ods info from DB")
        return 1
    elif len(ods_data) == 0:
        logging.info("No ODS task was found")
        return 2
    else:
        logging.debug("writing ods task data  from nails db to text file %s"%ods_filepath)
        fh = open(ods_filepath, "w+")
        for line in ods_data:
            fh.write(line)
        fh.close()
    return 0

def dumpUpdateTaskFromDB(sqlite_path, nails_db_path, update_filepath):
    '''
    Reads the Update task data of schedule table from nailsd.db
    and copies it to text file in backup dir
    RETURN: 0 , if successfully copies the data
            2 , if no update task is present
            1 , otherwise
    '''
    #query for update task
    _query_update = "select * from schedule where taskType='Update';"
    update_data = _runQuery(sqlite_path, nails_db_path, _query_update)

    if update_data is None:
        logging.error("Failure in dumping info from DB")
        return 1
    if len(update_data) == 0:
        logging.info("No Update task was found")
        return 2
    else:
        logging.debug("writing update task data  from nails db to text file %s "%update_filepath)
        fh = open(update_filepath, "w+")
        for line in update_data:
            fh.write(line)
        fh.close()
    return 0

def _runQuery(sqlite_path, nails_db_path, _query):
    '''
    Fn to run SQL query and get the result
    RETURN: output of query if successful
            None , otherwise
    '''
    logging.debug("Running query : %s" % _query)
    environmentDict = dict(os.environ);
    environmentDict['LD_LIBRARY_PATH'] = NAILS_LIB_DIR
    _rescode, _res, _err = executeCmd([sqlite_path, nails_db_path, _query], False, True, environmentDict)
    if _rescode:
        logging.error("Failed to create the sqlite process")
        return None
    task_data = _res.strip()
    return task_data

def _readFile(filename):
    """
    Fn to read the given file
    Returns the content of file as list.
    """
    lines = []
    try:
        fh = open(filename, 'r')
        if fh is None:
            return None
        lines = fh.readlines()
    except:
        logging.exception("Exception occured while reading file : %s", filename)
        return None
    fh.close()
    return lines

def getVSELDirs(filepath):
    """
    Fn to read the nails_options file on given filepath
    Returns the list containing [ VSEL Install Directory , VSEL Runtime Directory ] on success
    Returns an empty list , otherwise
    """
    nails_options = _readFile(filepath)
    lsh_dirs = []
    if not nails_options:
        logging.error("Could not read file :%s", filepath)
    else:
        for line in nails_options:
            install_dir = re.search('SILENT_INSTALLDIR\s*=\s*"(\S+)"', line)
            runtime_dir = re.search('SILENT_RUNTIMEDIR\s*=\s*"(\S+)"', line)
            if install_dir:
                lsh_dirs.insert(0, install_dir.group(1))
            elif runtime_dir:
                lsh_dirs.insert(1, runtime_dir.group(1))
        if len(lsh_dirs) != 2:
            logging.error("Could not Read installation or runtime dir from nails.options")
    return lsh_dirs

def getVSELVersion(install_dir):
    '''
    Fn returns the currently installed version of VSEL in integer format
    If fails to get the version returns None
    '''
    regex = '\s+(\d+\.\d+\.\d+\.\d+)-(\d+)-(\S+)'
    nails_binary_path = install_dir + "/bin/nails"
    version = None
    _cmd = [nails_binary_path, '-v']
    _rescode, _res, _err = executeCmd(_cmd)
    if _rescode:
        logging.error("Failed to get VSEL version")
        return None
    data = _res.strip()
    version_info = re.findall(regex, data)
    if not version_info:
        return None
    else:
        version = version_info[0][0][0] + version_info[0][0][2] + version_info[0][0][4]
    return int(version)

def parseOasFile(config):
    '''
    Reads the VSEL On-access config file line by line and converts the settings to ENSL CLI commands
    appends CLI commands to clist
    '''
    try:
        sor = None
        sow = None
        scanfile_option = None
        scanfile_list = []
        paction = None
        saction = None
        pactionpup = None
        sactionpup = None
        _previous_param = None
        _previous_excl = None
        excl_with_subfolder = []
        excl_without_subfolder = []
        _fd = open(config, 'r')

        logging.debug("================= Migrated values for On-Access :")
        for _line in _fd:
            try:
                _val = _line.split(":")[1]
            except:
                continue

            # migrate syslog settings
            if _line.startswith("log.useSyslog:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --usesyslog enable")
                    logging.debug("Syslog logging will be enabled")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --usesyslog disable")
                    logging.debug("Syslog logging will be disabled")
            # OAS on/off
            if _line.startswith("nailsd.oasEnabled:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --setoasglobalconfig --oas on")
                    logging.debug("On-access will be enabled")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --setoasglobalconfig --oas off")
                    logging.debug("On-access will be disabled")

            # OAS Scan Timeout
            if _line.startswith("nailsd.profile.OAS.scanMaxTmo:"):
                clist.append(CLI_BIN +" --setoasglobalconfig --maxscantime "+_val.strip())
                logging.debug("Scan time out : %s " %_val.strip())

            # Scan on Read/Scan on Write
            if _line.startswith("nailsd.profile.OAS.scanOnRead:"):
                if _val.strip().lower() == "true":
                    sor = "true"
                if _val.strip().lower() == "false":
                    sor = "false"
            if _line.startswith("nailsd.profile.OAS.scanOnWrite:"):
                if _val.strip().lower() == "true":
                    sow = "true"
                if _val.strip().lower() == "false":
                    sow = "false"
            if sor and sow:
                if sor == "true" and sow == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --setmode mcafee")
                    logging.debug("Scan mode : Let McaAfee Decide ( sor + sow )")
                    sor = None
                    sow = None
                if sor == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --setmode sor")
                    logging.debug("Scan mode : Scan On Read ")
                if sow == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --setmode sow")
                    logging.debug("Scan mode : Scan On Write ")
                sow = None
                sor = None

            # Scan All Files
            if _line.startswith("nailsd.profile.OAS.filter.extensions.mode:"):
                if _val.strip().lower() == "all":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --filetypestoscan all")
                if _val.strip().lower() == "default":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --filetypestoscan defaultandspecified")
                if _val.strip().lower() == "add":
                    scanfile_option = "defaultandspecified"
                if _val.strip().lower() == "replace":
                    scanfile_option = "onlyspecified"
                logging.debug("Extension mode (file types to scan) : %s " %_val.strip().lower())

            # Get extension list
            if _line.startswith("nailsd.profile.OAS.filter.extensions.list:"):
                exts = (_val.strip().lower()).split("|")
                for ext in exts:
                    if _isFileTypeValid(ext.strip()):
                        scanfile_list.append(ext.strip())
                    else:
                        logging.info("%s file type will not be migrated" %ext.strip())
            # Check if we can add the --filetypetoscan command
            if scanfile_list and scanfile_option != None:
                # create extension string
                extstr = ''
                ext_counter = 0
                # appending extensionlist command for each 20 extensions
                for extval in scanfile_list:
                    if extstr:
                        extstr = extstr+","+extval
                    else:
                        extstr = extval
                    ext_counter = ext_counter + 1
                    if ext_counter == 20:
                        clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --filetypestoscan "+scanfile_option+" --addfiletype "+extstr)
                        logging.debug("Added File types : %s "  %extstr)
                        ext_counter = 0
                        extstr = ''
                if ext_counter != 0:
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --filetypestoscan "+scanfile_option+" --addfiletype "+extstr)
                    logging.debug("Added File types : %s "  %extstr)
                scanfile_list = []
                scanfile_option = None

            # Get Scan Compressed file setting
            if _line.startswith("nailsd.profile.OAS.decompArchive:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanarchive enable")
                    logging.debug("Scan Compressed Files : enable")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanarchive disable")
                    logging.debug("Scan Compressed Files : disable ")

            # Scan network Drives
            if _line.startswith("nailsd.profile.OAS.scanNWFiles:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --networkscan enable")
                    logging.debug("Scan Network Files : enable")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --networkscan disable")
                    logging.debug("Scan Network Files : disable")

            # Scan MIME
            if _line.startswith("nailsd.profile.OAS.mime:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanmime enable")
                    logging.debug("Scan Mime : enable")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanmime disable")
                    logging.debug("Scan Mime : disable")

            # Scan PUP
            if _line.startswith("nailsd.profile.OAS.program:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanpups enable")
                    logging.debug("Scan pups : enable")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanpups disable")
                    logging.debug("Scan pups : disable")

            # Scan Unknown Program Threats
            if _line.startswith("nailsd.profile.OAS.heuristicAnalysis:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanunknownprograms enable")
                    logging.debug("Scan Unknown Programs : enable")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanunknownprograms disable")
                    logging.debug("Scan Unknown Programs : disable")

            # Scan Unknown Macro Threats
            if _line.startswith("nailsd.profile.OAS.macroAnalysis:"):
                if _val.strip().lower() == "true":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanunknownmacros enable")
                    logging.debug("Scan Unknown Macros : enable")
                if _val.strip().lower() == "false":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --scanunknownmacros disable")
                    logging.debug("Scan Unknown Macros : disable")

            # Threat detection first response
            #if _line.startswith("nailsd.profile.OAS.action.Default.primary:"):
            #   if _val.strip().lower() == "clean":
            #       paction = "clean"
            #   if _val.strip().lower() == "delete":
            #       paction = "delete"
            #   if _val.strip().lower() == "block":
            #       paction = "deny"
            # Threat detection first response
            if _line.startswith("nailsd.profile.OAS.action.Default.primary:"):
                if _val.strip().lower() == "clean":
                    paction = "clean"
                elif _val.strip().lower() == "delete":
                    paction = "delete"
                    saction = "deny"
                elif _val.strip().lower() == "block":
                    paction = "deny"
                    saction = "nothing"
                else:
                    paction = "clean"
                logging.debug("Migrated primary action for %s action is : %s" % (_val.strip().lower(), paction))

            # Threat detection second response
            if _line.startswith("nailsd.profile.OAS.action.Default.secondary:"):
                if not saction:
                    if _val.strip().lower() == "delete":
                        saction = "delete"
                    elif _val.strip().lower() == "block":
                        saction = "deny"
                    else:
                        saction = "delete"
                logging.debug("Migrated secondary action for %s action is : %s" % (_val.strip().lower(), saction))

            # Check if we have both primary and secondary action, then add the command
            if saction and paction:
                if saction == "nothing":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --primaryaction " +paction)
                elif not (paction == "clean" and saction == "delete"):
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --primaryaction " \
                                +paction+" --secondaryaction "+saction)

                paction = None
                saction = None
            # PUP detection first response
            if _line.startswith("nailsd.profile.OAS.action.App.primary:"):
                if _val.strip().lower() == "clean":
                    pactionpup = "clean"
                elif _val.strip().lower() == "delete":
                    pactionpup = "delete"
                    sactionpup = "deny"
                elif _val.strip().lower() == "block":
                    pactionpup = "deny"
                    sactionpup = "nothing"
                else:
                    pactionpup = "clean"
                logging.debug("Primary action for pup detection is mapped from %s to : %s" %(_val.strip().lower(), pactionpup))

            # PUP detection second response
            if _line.startswith("nailsd.profile.OAS.action.App.secondary:"):
                if not sactionpup:
                    if _val.strip().lower() == "delete":
                        sactionpup = "delete"
                    elif _val.strip().lower() == "block":
                        sactionpup = "deny"
                    else:
                        sactionpup = "delete"
                logging.debug("Secondary action for pup detection is mapped from %s to : %s" %(_val.strip().lower(), sactionpup))

            # Check if we have both primary and secondary action, then add the command
            if sactionpup and pactionpup:
                if sactionpup == "nothing":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --primaryactionpup " +pactionpup)
                elif not (pactionpup == "clean" and sactionpup == "delete"):
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --primaryactionpup " \
                                +pactionpup+" --secondaryactionpup "+sactionpup)

                pactionpup = None
                sactionpup = None

            # Scan Action on timeout
            if _line.startswith("nailsd.profile.OAS.action.timeout:"):
                if _val.strip().lower() == "block":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --onscantimeout deny")
                    logging.debug("Action on timeout : deny ")
                if _val.strip().lower() == "pass":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --onscantimeout allow")
                    logging.debug("Action on timeout : allow")

            # Scan Action on error
            if _line.startswith("nailsd.profile.OAS.action.error:"):
                if _val.strip().lower() == "block":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --onscanerror deny")
                    logging.debug("Action on Error : deny")
                if _val.strip().lower() == "pass":
                    clist.append(CLI_BIN +" --setoasprofileconfig --profile standard --onscanerror allow")
                    logging.debug("Action on Error : allow")

            # Extract exclusions
            # Exclusion will not be migrated if:
            #       - it does not start with "/" will not be migrated
            #       - it contains regex character
            #       - length exceeds 4096 character
            #       - it is ePO exclusion
            # If subdir is also excluded then path will be treated as directory , so trailing "/" is appended to the path
            # If subdir is not excluded then path will be migrated as following:
            #       a) If path ends with "/" it will be considered as directory exclusion
            #       b) If path does not ends with "/" , it will be considered as file exclusion
            if _line.startswith("nailsd.profile.OAS.filter"):
                if _line.startswith("nailsd.profile.OAS.filter.ePOexclusion"):
                    if _line.split(":")[0].endswith(".path"):
                        logging.info("%s is an ePO exclusion. Hence not migrating this exclusion" % _line.split(":")[1].strip())
                    continue
                _param = _line.split(":")[0]
                if _previous_param and _previous_excl and _param.endswith("subdir") and (_param[:-7] == _previous_param):
                    if _val.strip().lower() == "true":
                        if not _previous_excl.endswith("/"):
                            _previous_excl = _previous_excl+"/"
                        excl_with_subfolder.append(_previous_excl)
                        logging.debug("Exclusion with subfolder : %s " %_previous_excl)
                    if _val.strip().lower() == "false":
                        excl_without_subfolder.append(_previous_excl)
                        logging.debug("Exclusion without subfolder : %s " %_previous_excl)
                    _previous_param = None
                    _previous_excl = None

                if _param.endswith(".path"):
                    if _isPathValid(_val.strip()):
                        _previous_param = _param[:-5]
                        _previous_excl = _val.strip()
                    else:
                        logging.debug("Hence Skipping the migration of this exclusion.")
                        _previous_param = None
                        _previous_excl = None
        _migrateExclusions(excl_with_subfolder, excl_without_subfolder)
    except:
        _e = sys.exc_info()[1]
        logging.exception("Exception: %s" %str(_e))
    if _fd:
        _fd.close()
    return True

def _migrateExclusions(excl_with_subfolder, excl_without_subfolder):
    '''
    Fn that accepts two list of exclusion paths
    Appends CLI commands for exclusions in clist
    '''
    excl_counter = 0
    excl_str = ''
    # add exclusions with subdir
    for excl in excl_with_subfolder:
        if excl_str:
            excl_str = excl_str+","+excl
        else:
            excl_str = excl
        excl_counter = excl_counter + 1
        if excl_counter == 20:
            clist.append(CLI_BIN + " --setoasprofileconfig --profile standard  --addexclusionrw --excludepaths '"+excl_str+"' --excludesubfolder")
            excl_counter = 0
            excl_str = ''
    if excl_counter != 0:
        clist.append(CLI_BIN + " --setoasprofileconfig --profile standard  --addexclusionrw --excludepaths '"+excl_str+"' --excludesubfolder")
    excl_counter = 0
    excl_str = ''
    # add exclusions without subdir
    for excl in excl_without_subfolder:
        if excl_str:
            excl_str = excl_str+","+excl
        else:
            excl_str = excl
        excl_counter = excl_counter + 1
        if excl_counter == 20:
            clist.append(CLI_BIN + " --setoasprofileconfig --profile standard  --addexclusionrw --excludepaths '"+excl_str+"'")
    if excl_counter != 0:
        clist.append(CLI_BIN + " --setoasprofileconfig --profile standard  --addexclusionrw --excludepaths '"+excl_str+"'")

def readODSProfiles(filename, db_file):
    """
    Fn to read the profile from the given config file.
    Returns the config key and values in form of dictionary
    """
    _lines = _readFile(filename)
    if _lines is None:
        return None

    task_profile = dict()
    try:
        for _line in _lines:
            if not re.match('nailsd\.profile\.', _line):
                continue
            (key, value) = _line.split(':')
            value = value.rstrip().lstrip().rstrip("\n")

            key_string_holder = key.split('.')[2:]
            profile_name = key_string_holder.pop(0)

            #ignore default profiles
            if profile_name == 'ODS' or profile_name == 'ODS_default':
                continue
            #ignore ePO task profiles
            elif profile_name.startswith("ePO"):
                continue
            else:
                if not task_profile.has_key(profile_name):
                    task_profile[profile_name] = dict()

                if len(key_string_holder) == 1:
                    key_lev1 = key_string_holder.pop(0)
                    task_profile[profile_name][key_lev1] = value
                elif len(key_string_holder) > 1:
                    key_lev1 = key_string_holder.pop(0)

                    if not task_profile[profile_name].has_key(key_lev1):
                        task_profile[profile_name][key_lev1] = dict()

                    if len(key_string_holder) == 1:
                        key_lev2 = key_string_holder.pop(0)
                        task_profile[profile_name][key_lev1][key_lev2] = value
                    elif len(key_string_holder) > 1:
                        key_lev2 = key_string_holder.pop(0)
                        if not task_profile[profile_name][key_lev1].has_key(key_lev2):
                            task_profile[profile_name][key_lev1][key_lev2] = dict()
                        if len(key_string_holder) == 1:
                            key_lev3 = key_string_holder.pop(0)
                            task_profile[profile_name][key_lev1][key_lev2][key_lev3] = value
                        elif len(key_string_holder) > 1:
                            key_lev3 = key_string_holder.pop(0)
                            if not task_profile[profile_name][key_lev1][key_lev2].has_key(key_lev3):
                                task_profile[profile_name][key_lev1][key_lev2][key_lev3] = dict()
                            key_lev4 = key_string_holder.pop(0)
                            task_profile[profile_name][key_lev1][key_lev2][key_lev3][key_lev4] = value
                        else:
                            task_profile[profile_name][key_lev1][key_lev2] = value
                    else:
                        task_profile[profile_name][key_lev1] = value

        # adding scanpath and ODS schedule into task_profile from nails database
        database_entries = _readFile(db_file)

        for line in database_entries:
            ods_info = line.split("|")
            profile_name = re.findall("^profileName=(\S+),", ods_info[4])[0]
            task_profile[profile_name]['task_name'] = ods_info[1]
            task_profile[profile_name]['scan_paths'] = dict()
            no_of_scan_paths = len(ods_info) - 13 + 1

            for i in range(no_of_scan_paths):
                index = 4 + i
                task_profile[profile_name]['scan_paths'][i] = dict()
                ods_scan_path_detail = re.findall("^\S+h:(.+);\S+:(\S+)", ods_info[index])[0]
                task_profile[profile_name]['scan_paths'][i]['path'] = ods_scan_path_detail[0]
                task_profile[profile_name]['scan_paths'][i]['exclude_subdir'] = ods_scan_path_detail[1]

            #implementation of VSEL task schedule migration
            task_profile[profile_name]['schedule'] = ods_info[2].strip()
    except:
        _e = sys.exc_info()[1]
        logging.exception("%s" %str(_e))
    return task_profile

def migrateODS(config_file, db_file, clist):
    '''
    Parse for ODS settings and creates ODS task commands for migration
    appends these commands into clist
    RETURN :True , if successful
            False , otherwise
    '''
    try:
        vsel_ods_dict = readODSProfiles(config_file, db_file)
        if not vsel_ods_dict:
            logging.error("Unable to migrate ODS task")
            return False

        for profile, ods_info in vsel_ods_dict.iteritems():
            ensl_ods_dict = dict()
            logging.info("======================== Migrating ODS profile %s " %profile)
            logging.debug("Preparing a new attribute dictionary for task : %s" %ods_info['task_name'])

            if _isNameValid(ods_info['task_name']):
                ensl_ods_dict['name'] = ods_info['task_name']
            else:
                ensl_ods_dict['name'] = "Migrated_Task_" + profile
                logging.debug("<%s> task name is not a valid task name for ENSL" % ods_info['task_name'])
                logging.debug("Hence, Changing the task name to : %s" %ensl_ods_dict['name'])

            logging.debug("options are : ")
            #scan archive option
            if ods_info['decompArchive'] == 'true':
                ensl_ods_dict['scanarchive'] = 'enable'
            else:
                ensl_ods_dict['scanarchive'] = 'disable'
            logging.debug("scanarchive : %s" %ensl_ods_dict['scanarchive'])
            
            #network scan option
            if ods_info['scanNWFiles'] == 'true':
                ensl_ods_dict['scannetworkdrives'] = 'enable'
            else:
                ensl_ods_dict['scannetworkdrives'] = 'disable'
            logging.debug("scannetworkdrives : %s" %ensl_ods_dict['scannetworkdrives'])
            
            #scan mime option
            if ods_info['mime'] == 'true':
                ensl_ods_dict['scanmime'] = 'enable'
            else:
                ensl_ods_dict['scanmime'] = 'disable'
            logging.debug("scanmime : %s" %ensl_ods_dict['scanmime'])

            #scan potentially unwanted programs
            if ods_info['program'] == 'true':
                ensl_ods_dict['scanpups'] = 'enable'
            else:
                ensl_ods_dict['scanpups'] = 'disable'
            logging.debug("scanpups : %s" %ensl_ods_dict['scanpups'])

            #scan macros
            if ods_info['macroAnalysis'] == 'true':
                ensl_ods_dict['scanunknownmacros'] = 'enable'
            else:
                ensl_ods_dict['scanunknownmacros'] = 'disable'
            logging.debug("scanunknownmacros : %s " %ensl_ods_dict['scanunknownmacros'])

            # Scan Unknown Program Threats
            if ods_info['heuristicAnalysis'] == 'true':
                ensl_ods_dict['scanunknownprograms'] = 'enable'
            else:
                ensl_ods_dict['scanunknownprograms'] = 'disable'
            logging.debug("scanunknownprograms : %s" %ensl_ods_dict['scanunknownprograms'])

            #set maximum scan Time
            ensl_ods_dict['maxscantime'] = ods_info['scanMaxTmo']

            #set actions
            if 'secondary' not in ods_info['action']['Default']:
                ods_info['action']['Default']['secondary'] = None
            if 'secondary' not in ods_info['action']['App']:
                ods_info['action']['App']['secondary'] = None
            paction, saction = _setODSActions(ods_info['action']['Default']['primary'], ods_info['action']['Default']['secondary'])
            pactionpup, sactionpup = _setODSActions(ods_info['action']['App']['primary'], ods_info['action']['App']['secondary'])

            #set Default Primary action
            ensl_ods_dict['primaryaction'] = paction
            logging.debug("Migrating VSEL primary action %s to ENSL primary action : %s"%(ods_info['action']['Default']['primary'], paction))

            #set Default secondary action
            if saction == "nothing":
                ensl_ods_dict['secondaryaction'] = None
            else:
                ensl_ods_dict['secondaryaction'] = saction
            logging.debug("Migrating VSEL secondary action %s to ENSL secondary action : %s"%(ods_info['action']['Default']['secondary'], saction))

            #set primary action for pup
            ensl_ods_dict['primaryactionpup'] = pactionpup
            logging.debug("Migrating VSEL pup primary action %s to ENSL pup primary action : %s"%(ods_info['action']['App']['primary'], pactionpup))

            #set secondary action for pup
            if sactionpup == "nothing":
                ensl_ods_dict['secondaryactionpup'] = None
            else:
                ensl_ods_dict['secondaryactionpup'] = sactionpup
            logging.debug("Migrating VSEL pup secondary action %s to ENSL pup secondary action : %s"%(ods_info['action']['App']['secondary'], sactionpup))

            #set file types to scan
            file_type = None
            if ods_info['filter']['extensions']['mode'] == 'all':
                ensl_ods_dict['filetypestoscan'] = 'all'

            elif ods_info['filter']['extensions']['mode'] == 'default':
                ensl_ods_dict['filetypestoscan'] = 'defaultandspecified'

            elif ods_info['filter']['extensions']['mode'] == 'add':
                file_type = 'defaultandspecified'

            elif ods_info['filter']['extensions']['mode'] == 'replace':
                file_type = 'onlyspecified'

            if file_type is not None:
                # Get extension list
                scan_file_list = None
                exts = (ods_info['filter']['extensions']['list'].lower()).split("|")
                for ext in exts:
                    if _isFileTypeValid(ext):
                        if not scan_file_list:
                            scan_file_list = ext.strip()
                        else:
                            scan_file_list = scan_file_list + "," + ext.strip()

                    else:
                        logging.info("%s filetype will not be migrated"%ext)
                ensl_ods_dict['filetypestoscan'] = file_type
                ensl_ods_dict['addfiletype'] = scan_file_list

            else:
                ensl_ods_dict['addfiletype'] = None
            logging.debug("filetypestoscan : %s" %ensl_ods_dict['filetypestoscan'])
            logging.debug("addfiletype : %s " %ensl_ods_dict['addfiletype'])

            #set exclusions with and without subfolder
            filters = ods_info['filter']
            # remove the extension from filter , keep only exclusion paths
            del filters['extensions']
            excl_not_subdir = ''
            excl_with_subdir = ''
            for i in range(len(filters)):
                excl_str = filters[str(i)]['path']
                if not _isPathValid(excl_str):
                    logging.info("Path is not valid. Hence Not migrating this exclusion")
                else:
                    #split into exclusion with sudir enabled and excusions without subdir
                    if filters[str(i)]['subdir'] == 'true':
                        if not excl_str.endswith("/"):
                            excl_str = excl_str + "/"
                        if excl_with_subdir:
                            excl_with_subdir = excl_with_subdir + "," + excl_str
                        else:
                            excl_with_subdir = excl_str
                    else:
                        if excl_not_subdir:
                            excl_not_subdir = excl_not_subdir + "," + excl_str
                        else:
                            excl_not_subdir = excl_str
            ensl_ods_dict['excludepaths'] = excl_not_subdir
            ensl_ods_dict['excludepathwithsubfolder'] = excl_with_subdir
            logging.debug("Exclusions with subdir: %s " %ensl_ods_dict['excludepaths'])
            logging.debug("Exclusions without subdir: %s " %ensl_ods_dict['excludepathwithsubfolder'])

            #migrate the schedule
            ensl_ods_dict['schedule'] = convertSchedule(ensl_ods_dict['name'], ods_info['schedule'])

            # migrate scan path for ODS task
            #1) VSEL ODS task may contain file or directory
            #       Any path that ends with "/" will be treated as directory in ENSL
            #       Any path that does not ends with "/" will be treated as file in ENSL
            # 2)if scan paths contains the combination of paths that include subdir and path that does not include subdir scan
            #   then split the task into two tasks:
            #                                   - task 1 : with subdir scan enabled
            #                                   - task 2 : subdir scan disabled
            #   Reason: in VSEL "subdir_scan" option is associated with each path
            #   but in ENSL "subdir scan" is a global option

            temp_sched = dict()

            if len(ods_info['scan_paths']) == 1:
                scan_path = ods_info['scan_paths'][0]['path']
                if ods_info['scan_paths'][0]['exclude_subdir'] == 'true':
                    ensl_ods_dict['scansubfolders'] = 'disable'
                else:
                    ensl_ods_dict['scansubfolders'] = 'enable'

                if _isPathValid(scan_path):
                    clist.append(CLI_BIN + createODSCmd(ensl_ods_dict, scan_path))
                    if not ensl_ods_dict['schedule'] == "unscheduled":
                        temp_sched['name'] = ensl_ods_dict['name']
                        temp_sched['schedule'] = ensl_ods_dict['schedule']
                    logging.info("Single ODS task with scan path '%s' will be migarted" %scan_path)
                else:
                    logging.info("Skipping the migration of this task because scan path %s is not a valid path" %scan_path)

            if len(ods_info['scan_paths']) > 1:
                num_path_subdir = 0
                path_subdir = ''
                num_path_not_subdir = 0
                path_not_subdir = ''
                for i in range(len(ods_info['scan_paths'])):
                    if _isPathValid(ods_info['scan_paths'][i]['path']):
                        if ods_info['scan_paths'][i]['exclude_subdir'] == 'false':
                            num_path_subdir = num_path_subdir + 1
                            if path_subdir:
                                path_subdir = path_subdir + ',' + ods_info['scan_paths'][i]['path']
                            else:
                                path_subdir = ods_info['scan_paths'][i]['path']

                        if ods_info['scan_paths'][i]['exclude_subdir'] == 'true':
                            num_path_not_subdir = num_path_not_subdir + 1
                            if path_not_subdir:
                                path_not_subdir = path_not_subdir + "," + ods_info['scan_paths'][i]['path']
                            else:
                                path_not_subdir = ods_info['scan_paths'][i]['path']
                    else:
                        logging.debug("%s is an invalid scan path. It will not be migrated" % ods_info['scan_paths'][i]['path'])

                if num_path_subdir == 0 and num_path_not_subdir == 0:
                    logging.info("Migration of this task will be skipped because all scan paths are invalid")
                elif num_path_subdir == 0:
                    logging.debug("Subdirectory option is false for all scan paths : %s" %path_not_subdir)
                    logging.debug("Therefore , Creating a Single ODS task where scansubfolders option is disabled")
                    ensl_ods_dict['scansubfolders'] = 'disable'
                    clist.append(CLI_BIN +createODSCmd(ensl_ods_dict, path_not_subdir))
                    if not ensl_ods_dict['schedule'] == "unscheduled":
                        temp_sched['name'] = ensl_ods_dict['name']
                        temp_sched['schedule'] = ensl_ods_dict['schedule']

                elif num_path_not_subdir == 0:
                    logging.debug("Subdirectory option is true for all scan paths : %s" %path_subdir)
                    logging.debug("Therefore , Creating a Single ODS task where scansubfolders option is enabled")
                    ensl_ods_dict['scansubfolders'] = 'enable'
                    clist.append(CLI_BIN +createODSCmd(ensl_ods_dict, path_subdir))
                    if not ensl_ods_dict['schedule'] == "unscheduled":
                        temp_sched['name'] = ensl_ods_dict['name']
                        temp_sched['schedule'] = ensl_ods_dict['schedule']
                else:
                    logging.debug("There are %s paths with subdir option false and %s with subdir option true" %(num_path_not_subdir, num_path_subdir))
                    logging.debug("Therefore , Splitting task <%s> into two tasks" %ensl_ods_dict['name'])
                    task1_name = ensl_ods_dict['name']+'_1'
                    task2_name = ensl_ods_dict['name']+'_2'
                    logging.debug("Creating first ODS task <%s> with subdir option enabled for scan_path %s" %(task1_name, path_subdir))
                    ensl_ods_dict['name'] = task1_name
                    ensl_ods_dict['scansubfolders'] = 'enable'
                    clist.append(CLI_BIN +createODSCmd(ensl_ods_dict, path_subdir))
                    if not ensl_ods_dict['schedule'] == "unscheduled":
                        temp_sched['name'] = ensl_ods_dict['name']
                        temp_sched['schedule'] = ensl_ods_dict['schedule']
                        logging.info("Migrating the schedule with first task . Second will be unscheduled")

                    logging.debug("Creating second ODS task <%s> with subdir option disabled  for scan_path %s" %(task2_name, path_not_subdir))
                    ensl_ods_dict['name'] = task2_name
                    ensl_ods_dict['scansubfolders'] = 'disable'
                    clist.append(CLI_BIN +createODSCmd(ensl_ods_dict, path_not_subdir))

            if temp_sched:
                schedule_info.append(temp_sched)
    except:
        _e = sys.exc_info()[1]
        logging.exception(" %s" %str(_e))
        return False
    return True

def _isPathValid(path):
    '''
    Fn checks whether the given path is a valid scan path
    RETURN : True ,if it is a valid path
             False, Otherwise
    '''
    reg_char = ['+', '(', ')', '<', '>', '$', '^', '[', ']', '\\', '|', '{', '}', '&', '@', '!', '#', '%', '"', ';', ',']
    # if a path that does not begin with "/" it is invalid path
    if not path.startswith("/"):
        logging.info("Path < %s > does not start with '/' " %path)
        return False
    # if a path contain regex character , it is invalid path
    for char in reg_char:
        if char in path:
            logging.info("Path < %s > contains regex character '%s' "  %(path, char))
            return False
    if len(path) > 4096:
        logging.info("Length of path < %s > is greater than the maximum allowed length 4096" %path)
        return False
    return True

def _isFileTypeValid(ext):
    '''
    Fn checks whether given extension is valid
    RETURN : true , if extension is valid
             fasle , if not valid
    '''
    reg_char = ['+', '(', ')', '<', '>', '$', '^', '[', ']', '\\', '|', '.', '&', '@', '!', '#', '%', '"', ';', ',']
    #extension is invalid if it contains special characters
    for char in reg_char:
        if char in ext:
            logging.info("%s is invalid extension because it contains special characters" %ext)
            return False
    #extension is invalid if contains more than 12 characters
    if len(ext) > 12:
        logging.info("%s is invalid extension because length of extension is more than maximum allowed length 12"%ext)
        return False
    return True

def convertSchedule(task_name, old_schedule):
    '''
    Fn accepts the VSEL schedule of a task and converts to corresponding ENSL schedule
    returns the new schedule string
    '''
    new_schedule = "unscheduled"
    if old_schedule.startswith("type=unscheduled"):
        logging.info("No schedule is associated with the task : %s", task_name)
        new_schedule = "unscheduled"

    elif old_schedule.startswith("type=daily"):
        if old_schedule.startswith("type=daily,recurrence=1"):
            time = re.findall("type\S+,\S+,hour=(\S+),minute=(\S+)", old_schedule)
            if time:
                hour = time[0][0]
                min = time[0][1]
                if len(min) < 2:
                    min = '0' + min
                if len(hour) < 2:
                    hour = '0' + hour
                new_schedule = "--daily --starttime "+hour+":"+min
            else:
                logging.error("Error while converting schedule %s for task %s:" %(old_schedule, task_name))
                new_schedule = "unscheduled"

        else:
            logging.info("There is no mapping for schedule %s in ENSL. Hence not migrating schedule for task %s" %(old_schedule, task_name))
            new_schedule = "unscheduled"

    elif old_schedule.startswith("type=weekly"):
        if old_schedule.startswith("type=weekly,recurrence=1"):
            time = re.findall("type\S+,\S+,hour=(\S+),minute=(\S+),days=(\S+)", old_schedule)
            if time:
                hour = time[0][0]
                min = time[0][1]
                if len(min) < 2:
                    min = '0' + min
                if len(hour) < 2:
                    hour = '0' + hour
                days = time[0][2].split(";")
                weekday = ['sun', 'mon', 'tue', 'wed', 'thu', 'fri', 'sat']
                day_str = ''

                for i in range(len(weekday)):
                    if str(i) in days:
                        if day_str:
                            day_str = day_str + "," + weekday[i]
                        else:
                            day_str = weekday[i]
                            day_str = weekday[i]
                new_schedule = "--weekly --dayofweek '"+ day_str + "' --starttime "+hour+":"+min
            else:
                logging.error("Error while converting schedule %s for task" %old_schedule)
                new_schedule = "unscheduled"

        else:
            logging.info("There is no mapping for schedule %s in ENSL. Hence not migrating schedule"%old_schedule)
            new_schedule = "unscheduled"

    else:
        logging.info("There is no mapping for schedule %s in ENSL. Hence not migrating schedule"%old_schedule)
        new_schedule = "unscheduled"

    logging.debug("Mapping for %s schedule is : %s " %(old_schedule, new_schedule))
    return new_schedule

def createODSCmd(ods_details, scan_path):
    '''
    Returns the command to create an ODS task
    based on provided settings
    '''
    _cmd = " --addodstask " \
           + "--name '"+ ods_details['name'] + "' "                                \
           + "--scanpaths '"+ scan_path + "' "                                     \
           + "--scanarchive " + ods_details['scanarchive'] + " "                   \
           + "--scanmime " + ods_details['scanmime'] + " "                         \
           + "--scanpups " + ods_details['scanpups'] + " "                         \
           + "--scanunknownprograms " + ods_details['scanunknownprograms'] + " "   \
           + "--scanunknownmacros " + ods_details['scanunknownmacros'] + " "       \
           + "--scansubfolders " + ods_details['scansubfolders'] + " "             \
           + " --scannetworkdrives " + ods_details['scannetworkdrives'] + " "            \
           + "--maxscantime " + ods_details['maxscantime'] + " "

    # add exclusion if exists
    if ods_details['excludepaths']:
        _cmd = _cmd + "--excludepaths '"+ ods_details['excludepaths'] + "' "
    if ods_details['excludepathwithsubfolder']:
        _cmd = _cmd + "--excludepathwithsubfolder '"+ ods_details['excludepathwithsubfolder'] + "' "

    # add file extensions if exists
    _cmd = _cmd + "--filetypestoscan " + ods_details['filetypestoscan'] + " "
    if  ods_details['addfiletype']:
        _cmd = _cmd + "--addfiletype "+ ods_details['addfiletype'] + " "

    #add primary and secondary actions
    if ods_details['secondaryaction'] is None:
        _cmd = _cmd + "--primaryaction " + ods_details['primaryaction'] + " "
    else:
        _cmd = _cmd + "--primaryaction " + ods_details['primaryaction'] + " "   \
                + "--secondaryaction " + ods_details['secondaryaction'] + " "

    if ods_details['secondaryactionpup'] is None:
        _cmd = _cmd + "--primaryactionpup " + ods_details['primaryactionpup'] + " "
    else:
        _cmd = _cmd + "--primaryactionpup " + ods_details['primaryactionpup'] + " "         \
                    + "--secondaryactionpup " + ods_details['secondaryactionpup'] + " "

    return _cmd

def migrateUpdateTask(update_db_file):
    '''
    Fn reads the update task info from given file
    appends the corresponding CLI commands for update task in CLI
    RETURN: 0, if successful
            1, otherwise
    '''
    data = _readFile(update_db_file)
    global Update_Task_Counter
    try:
        if not data:
            logging.error("unable to migrate update task")
            return False
        for line in data:
            temp_sched = dict()
            updated_name = dict()
            update_info = line.split('|')
            task_name = update_info[1]
            if task_name == "LinuxShield Update":
                if update_info[2].startswith("type=unscheduled"):
                    logging.info("VSEL default DAT update task is unscheduled, Hence after migration ENSL default DAT and Engine update task will be unscheduled")
                    continue
                else:
                    temp_sched['name'] = "Default Client Update task"
                    temp_sched['schedule'] = "--daily --starttime 00:15"
                    schedule_info.append(temp_sched)
                    continue
                
            to_update = update_info[4].split('=')[1]
            update_item_list = to_update.split(";")

            update_type = None
            for item in update_item_list:
                if  update_type:
                    update_type = update_type + "," + item
                else:
                    update_type = item

            #add update task command to clist
            logging.debug("==================== Migrating update task : %s" % task_name)
            logging.debug("Update type is : %s " %update_type)
            if not _isNameValid(task_name):
                # if comma is present in update_type, i.e update type is either "engine,dat" or "dat,engine"
                if update_type.find(',') != -1:
                    update_type_vector = update_type.split(',') 
                    updated_name['name'] = "Migrated_Update_Task_" + str(Update_Task_Counter) + "_" + update_type_vector[0].title() + "_" + update_type_vector[1].title()
                else:
                    updated_name['name'] = "Migrated_Update_Task_" + str(Update_Task_Counter) + "_" + update_type.title()
                Update_Task_Counter +=1
                logging.debug("<%s> update name is not a valid task name for ENSL" % task_name)
                logging.debug("Hence, Changing the task name to : %s" %updated_name['name'])
                task_name = updated_name['name']
            clist.append(CLI_BIN + " --addupdatetask --name '"+task_name+"' --updatetype '"+update_type+"'")
            #add update task schedule to schedule_info
            schedule = convertSchedule(task_name, update_info[2])

            if not schedule == "unscheduled":
                temp_sched['name'] = task_name
                temp_sched['schedule'] = schedule
            if temp_sched:
                schedule_info.append(temp_sched)
        return True
    except:
        _e = sys.exc_info()[1]
        logging.exception(" %s" %str(_e))
        return False

def _isNameValid(name_str):
    '''
    RETURN: True if given string is a valid name for an ODS task
            False , otherwise
    '''
    invalid_chars = ['/', '"', '[', ']', ':', '|', '<', '>', '+', '=', ';', ',', '?', '*', '@', '&', '.']
    for char in invalid_chars:
        if char in name_str:
            return False
    return True

def _setODSActions(paction_vsel, saction_vsel):
    '''
    Fn to accept primary and secondary action of VSEL ODS task
    Maps them into valid actions for ENSL ODS task
    Returns a tuple ( primary action for ENSL , secondary action for ENSL)
    '''
    paction_ensl = None
    saction_ensl = None

    if paction_vsel.lower() == "clean":
        paction_ensl = "clean"

    elif paction_vsel.lower() == "delete":
        paction_ensl = "delete"
        saction_ensl = "continue"

    elif paction_vsel.lower() == "pass":
        paction_ensl = "continue"
        saction_ensl = "nothing"

    else:
        paction_ensl = "clean"

    if not saction_ensl:
        if saction_vsel.lower() == "delete":
            saction_ensl = "delete"

        elif saction_vsel.lower() == "pass":
            saction_ensl = "continue"

        else:
            saction_ensl = "delete"
    return (paction_ensl, saction_ensl)

def main():
    '''
    Main Loop
    '''
    #Configure the logger for migration script
    log_format = "%(asctime)s  %(levelname)6s:  %(message)s"
    log_date_format = '%a, %d %b %Y %H:%M:%S'

    global clist, schedule_info
    clist = []
    schedule_info = []

    if len(sys.argv) < 2:
        usage()
        sys.exit(0)
    if not isVSELInstalled():
        sys.exit(0)
    else:
        try:
            try:
                if not os.path.exists(LOG_FILE_PATH):
                    print ("Exiting.. because log file path does not exist")
                    sys.exit(0)

                logging.basicConfig(level=logging.DEBUG,
                                    format=log_format,
                                    datefmt=log_date_format,
                                    filename=LOG_FILE_PATH+"migrate-vsel.log")

                logging.debug("Checking if temporary backup directory exists...")

                backup_dir = sys.argv[1]
                if not os.path.exists(backup_dir):
                    logging.error("Exiting the backup process because backup directory %s does not exist" %backup_dir)
                    sys.exit(0)

                logging.debug("backup dir is : %s"  %backup_dir)
                logging.info("VSEL is installed on this machine")

                #exit if machine is managed from ePO
                if isSystemManaged():
                    logging.info("This system is managed from ePO")
                    logging.info("Policies and task will not be migrated for a managed system")
                    logging.info("Exiting migration process")
                    sys.exit(0)

                logging.debug("This system is running in unmanaged mode")
                logging.debug("reading nails.options file")
                if not os.path.exists(NAILS_OPTIONS_FILE):
                    logging.error("Exiting the backup process because nails.options not found")
                    sys.exit(0)

                logging.debug("Getting VSEL Install and Runtime Dir Path")
                lsh_dirs = getVSELDirs(NAILS_OPTIONS_FILE)

                if not lsh_dirs:
                    logging.error("Unable to get details of VSEL Install and Runtime Dir")
                    sys.exit(0)

                run_time_dir = lsh_dirs[1]
                install_dir = lsh_dirs[0]

                vsel_version = getVSELVersion(install_dir)
                logging.info("Currently installed version of VSEL is : %d" %vsel_version)

                if vsel_version < 190:
                    logging.error("Can not proceed the migration because VSEL version is less than 1.9.0")
                    print("Not migrating settings because installed version of VSEL is older than 1.9.0")
                    sys.exit(0)

                oas_config_file = run_time_dir + "/etc/nailsd.cfg"
                ods_config_file = run_time_dir + "/etc/ods.cfg"
                nails_db_path = run_time_dir + "/etc/nailsd.db"
                sqlite_path = install_dir + "/libexec/sqlite"
                ods_db_file = backup_dir + "/ods_db.txt"
                update_db_file = backup_dir + "/update_db.txt"

                logging.info("copying nailsd.cfg to backup dir")
                shutil.copy(oas_config_file, backup_dir)

                logging.info("copying ods.cfg to backup dir")
                shutil.copy(ods_config_file, backup_dir)

                logging.info("copying crontab entry")
                shutil.copy("/etc/crontab", backup_dir)

                logging.info("dumping the nailsd.db data to text files in backup dir")

                _retval = dumpUpdateTaskFromDB(sqlite_path, nails_db_path, update_db_file)
                if _retval == 1:
                    logging.error("Unable to take update task data from nailsd.db")
                    sys.exit(0)
                elif _retval == 2:
                    logging.debug("There is no Update task present in the system")
                    logging.info("Update Task migration will be skipped")
                else:
                    logging.info("Successfully copied Update Task details from nails.db")

                _retval = dumpODSTaskFromDB(sqlite_path, nails_db_path, ods_db_file)
                if _retval == 1:
                    logging.error("Unable to take ODS task data from nailsd.db")
                    sys.exit(0)
                elif _retval == 2:
                    logging.debug("There is no ODS task present in the system")
                    logging.info("ODS Task migration will be skipped")
                else:
                    logging.info("Successfully copied ODS task data from nailsd.db")

                logging.info("Taken backup Successfully ")

                # Migrate OAS Settings
                oas_config_file = backup_dir +"/nailsd.cfg"
                ods_config_file = backup_dir +"/ods.cfg"

                if not os.path.exists(oas_config_file) or not os.path.exists(ods_config_file):
                    logging.error("Exiting the migration because backup files do not exist")
                    sys.exit(0)

                parseOasFile(oas_config_file)
                # Migrate ODS setting
                if os.path.exists(ods_db_file):
                    if not migrateODS(ods_config_file, ods_db_file, clist):
                        logging.error("Migration of ODS task failed")
                    else:
                        logging.info("ODS task migration completed Successfully")
                else:
                    logging.info("Skipped ODS migration because There is no ODS task in DB")

                #migrate Update task settings
                if os.path.exists(update_db_file):
                    if not migrateUpdateTask(update_db_file):
                        logging.error("Migration of Update task failed")
                    else:
                        logging.info("Update Task migration completed Successfully")
                else:
                    logging.info("Skipped Update Task migration because There is no Update task in DB")

                #Lets wait for all mfetp processes to come up
                time.sleep(5)

                for _cmd in clist:
                # Do not enable or disable OAS till all config is done
                    if "--setoasglobalconfig --oas" in _cmd:
                        oasonoff = _cmd
                        continue
                    logging.debug("Executing command : %s " %_cmd)
                    _rescode, _res, n_err = executeCmd(_cmd, shellFlag=True)
                    logging.debug("Response code from CLI : %s" %_rescode)
                    logging.debug("Response from CLI : %s " %_res)
                #schedule the available tasks
                logging.info("Scheduling the Update and ODS tasks")

                for task in schedule_info:
                    _cmd = CLI_BIN + " --listtask"
                    _ret, _list_task_output, _err = executeCmd(_cmd, shellFlag=True)
                    regex = '(\d+)\s+'+task['name']+'\s+'  # find index of task

                    if _list_task_output:
                        index = re.findall(regex, _list_task_output)
                        if index:
                            task_index = index[0]
                            _cmd = CLI_BIN + " --scheduletask --index "+task_index+" "+task['schedule']
                            logging.debug("Executing command : %s " %_cmd)
                            executeCmd(_cmd, shellFlag=True)

                logging.info("Successfully updated the respective schedules of ODS and Update Tasks")
                #oas-on off
                executeCmd(oasonoff, shellFlag=True)
                logging.info("Migration from VSEL to ENSL completed Successfully")
                print ("Migration from VSEL to ENSL completed Successfully")

            # Exception due to sys.exit() does not need to log the traceback
            except SystemExit:
                pass
            except:
                _e = sys.exc_info()[1]
                logging.exception(" %s" %str(_e))
                logging.error("Exception occurred while migration")
                sys.exit(0)

        finally:
            lsh_dir = getVSELDirs(NAILS_OPTIONS_FILE)
            if not lsh_dir:
                logging.error("Unable to get VSEL runtime directory path")
            else:
                oas_file = lsh_dir[1] + "/etc/nailsd.cfg"
                ods_file = lsh_dir[1] + "/etc/ods.cfg"
                db_file = lsh_dir[1] + "/etc/nailsd.db"
                shutil.copy(oas_file, NAILS_BACKUP_DIR)
                shutil.copy(ods_file, NAILS_BACKUP_DIR)
                shutil.copy(db_file, NAILS_BACKUP_DIR)

# Start Here
if __name__ == "__main__":
    main()
