#!/bin/bash
# Copyright (C) 2020 McAfee, LLC. All Rights Reserved.
# Script to set the SELinux context for ENSL TP Quarantine files
# This script always exits with a return code of 0

# Unset all the language locale and set LANG to C to disable localization
unset LC_ALL
unset LANG
export LANG=C
export LC_ALL=C

# Change it to yes to see output
verboseMode="no"

# Set the environment variable where commands will be commonly present
PATH=${PATH}:/sbin:/usr/sbin:/bin:/usr/bin

#Check if Run By root or not
if [ $EUID -ne 0 ]
then
    echo "This script can be run by root or a user with sudo privileges"
    exit 0
fi

usage()
{
    echo "Usage: $0 [quarantineDir] [force]"
    echo "Run this script to set the SELinux context for ENSL TP Quarantine files"
    echo "Provide the path of the directory to set, or it will be read from the configuration"
    echo "Applicable only for RHEL 7 and above when McAfeeENS-selinux is installed and SELinux is in targeted mode"
    exit 0
}

showMessage()
{
    content=$1
    if [ ${verboseMode} = "yes" ]
    then
        echo ${content}
    fi
}

# Set the SELinux context for TP to specific SELinux contexts
# To reset the context, $1 is set to default
# - Quarantine directory and all its contents
updateSELinuxContext()
{
    if [ -n "${1}" ]
    then
        reset=1
    else
        reset=0
    fi
    chconBinPath="/usr/bin/chcon"
    restoreBinPath="/usr/sbin/restorecon"
    if [ -f ${chconBinPath} ]
    then
        # Read from prefs.xml if not externally provided
        if [ -z "${quarantineDir}" ]
        then
            if [ -f /var/McAfee/ens/tp/prefs.xml ]
            then
                quarantineDir=$(sed -n -e "s/&apos;/\'/g" -e 's:.*<QuarantineDirectory>\(.*\)</QuarantineDirectory>.*:\1:p' /var/McAfee/ens/tp/prefs.xml)
            fi
        fi
        if [ "${reset}" -eq 0 ]
        then
            [ -n "${quarantineDir}" ] && [ -d "${quarantineDir}" ] && ${chconBinPath} -t mfe_ens_tpd_quarantine_t -R "${quarantineDir}" > /dev/null 2>&1 || :
        else
            [ -n "${quarantineDir}" ] && [ -d "${quarantineDir}" ] && ${restoreBinPath} -R "${quarantineDir}" > /dev/null 2>&1 || :
        fi
    fi
}

if [ $# -gt 2 ]
then
    usage
fi

# Flag used to track if restorecon should be run on supported systems even if ENS policy RPM is not installed
# 1 to run restore forcefully
forceRestore=0

numargs=$#
for ((i=1 ; i <= numargs ; i++))
do
    case $1 in
        "force")
            forceRestore="1"
            ;;
        *)
            quarantineDir=$1
            # Ignore setting SELinux context on path which is not a directory
            if [ ! -d "${quarantineDir}" ]
            then
                exit 0
            fi
            ;;
    esac
    shift
done

# Fetch current distribution details
distribRelString="Unknown"
if [ -f "/etc/redhat-release" ]
then
    distribRel=$(cat /etc/redhat-release)
elif [ -f "/etc/system-release" ]
then
    distribRel=$(cat /etc/system-release)
elif [ -f "/etc/SuSE-release" ]
then
    # Need only the first line
    distribRel=$(head -n 1 /etc/SuSE-release)
elif [ -f "/etc/os-release" ]
then
    # /etc/os-release has all the information
    distribRelString=$(awk -F '=' '/^ID=/ {print $2}' /etc/os-release)
    distribRelMajNum=$(awk -F '=' '/^VERSION_ID=/ {print $2}' /etc/os-release | sed -e 's/\.//g' -e 's/"//g')
    distribRel=$(awk -F '=' '/^PRETTY_NAME=/ {print $2}' /etc/os-release)
fi
# Parse and format to identify OS if not already set
if [ -n "$distribRel" -a "$distribRelString" = "Unknown" ]
then
    # Replace any space or () with _
    # Converts SUSE Linux Enterprise Server 12 (x86_64) to SUSE_Linux_Enterprise_Server_12__x86_64_
    distribRelString=$(echo ${distribRel//[ ()]/_})
    # Delete any spaces a-z A-Z which will leave Distribution release number and ()
    # Converts "SUSE Linux Enterprise Server 12 (x86_64)" to "12(86_64)"
    distribRelMajNumTemp=$(echo ${distribRel//[a-zA-Z ]/})
    # Replace any () with . and then cut it on "." to get the major release number
    # Converts "12(86_64)" to "12.86_64" and then to "12"
    distribRelMajNum=$(echo ${distribRelMajNumTemp//[()]/.} | cut -d '.' -f1)
    # Replace any () with . and then cut it on "." to get the major release number
    # Converts "12(86_64)" to "12.86_64" and then to "86_64"
    distribRelMinNum=$(echo ${distribRelMajNumTemp//[()]/.} | cut -d '.' -f2)
fi
# Flag to track distribution support
isSupported="no"
# Enable case insensitive match
shopt -s nocasematch
# Check if distribution release string starts with Red for RedHat
redHatSearchPattern="^Red"
if [[ $distribRelString =~ $redHatSearchPattern ]]
then
    # RHEL 7 and above is supported
    if [ ${distribRelMajNum} -ge 7 ]
    then
        isSupported="yes"
    fi
fi
# For unsupported distribution, do not spam the logs that SELinux is unsupported
if [ "${isSupported}" = "no" ]
then
    showMessage "Not setting SELinux context as distribution ${distribRel} is unsupported"
else
    if [ "${forceRestore}" -eq 1 ]
    then
        updateSELinuxContext "resetToDefault"
    else
        # Ensure selinux policiy is loaded
        isSELinuxPolicyLoadedg=$(semodule -l | grep -q mfe_ens_tp)
        if [ $? -eq 0 ]
        then
            updateSELinuxContext
        else
            showMessage "Not setting SELinux context as mfe_ens_tp for ${distribRel} is not loaded"
        fi
    fi
fi

exit 0
